// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <limits.h>

#include "lasc.h"

// LASCColorMap

LASCColorMap::LASCColorMap()
{
  size = 0;
  colors = NULL;
  itt = NULL;
}

LASCColorMap::~LASCColorMap()
{
  if (colors)
    delete [] colors;
}

int LASCColorMap::load()
{
  // Determine if we can read the file

  ifstream f(fileName);
  if (!f)
    return 0;
  
  // Figure out the number of colors

  float junk;
  while (1) {
    for (int i=0; i<3; i++)
      f >> junk;
    if (f.eof())
      break;
    size++;
  }
  f.close();

  // allocate space for our colors

  colors = new RGBColor[size];

  // and read the colors

  f.open(fileName);
  for (int i=0; i<size; i++)
    f >> colors[i];
  f.close();

  return 1;
}

int LASCColorMap::save(const char* fn)
{
  ofstream f(fn);
  if (!f)
    return 0;

  f << *this;

  return 1;
}

void LASCColorMap::setITT(ITT* t)
{
  if (t) {
    if (t->getSize() == size)
      itt = t;
    else {
      itt = NULL;
      cerr << " colormap " << name << " and ittmap " 
	   << t->getName() << " are not the same size." << endl;
    }
  }
  else
    itt = NULL;
}

unsigned char LASCColorMap::getColorChar(int i, int count, float (RGBColor::*getColor)())
{
  int index = (int)((i * size / count) + .5);
  if (index >=0 && index < size) {
    if (itt) {
      index = (int)(((*itt)[index] * size) + .5) - 1;
      if (index >=0 && index < size)
	return (unsigned char)((colors[index].*getColor)() * UCHAR_MAX);
      else
	return 0;
    }
    else
      return (unsigned char)((colors[index].*getColor)() * UCHAR_MAX);
  }
  else
    return 0;
}

unsigned short LASCColorMap::getColorShrt(int i, int count, float (RGBColor::*getColor)())
{
  int index = (int)((i * size / count) + .5);
  if (index >=0 && index < size) {
    if (itt) {
      index = (int)(((*itt)[index] * size) + .5) - 1;
      if (index >=0 && index < size)
	return (unsigned short)((colors[index].*getColor)() * USHRT_MAX);
      else
	return 0;
    }
    else
      return (unsigned short)((colors[index].*getColor)() * USHRT_MAX);
  }
  else
    return 0;
}

ostream& operator<<(ostream& s, LASCColorMap& c)
{
  if (c.colors)
    for (int i=0; i<c.size; i++)
      s << c.colors[i];

  return s;
}
