//time.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2012
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

#ifdef ROAR_HAVE_H_SYS_TIME
#include <sys/time.h>
#endif
#ifdef ROAR_HAVE_H_TIME
#include <time.h>
#endif

int roar_time_to_msg(struct roar_message * mes,  struct roar_time    * time) {
 uint64_t * data;

 if ( mes == NULL || time == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 memset(mes, 0, sizeof(struct roar_message));

 data = (uint64_t*)mes->data;

 data[0] = ROAR_HOST2NET64(0);
 data[1] = ROAR_HOST2NET64(time->t_sec);
 data[2] = ROAR_HOST2NET64(time->t_ssec);
 data[3] = ROAR_HOST2NET64(time->c_freq);
 data[4] = ROAR_HOST2NET64(time->c_drift);

 mes->datalen = 5*8;

 if ( time->c_drift == 0 ) {
  mes->datalen -= 8;
  if ( time->c_freq == 0 ) {
   mes->datalen -= 8;
   if ( time->t_ssec == 0 ) {
    mes->datalen -= 8;
   }
  }
 }

 return 0;
}

int roar_time_from_msg(struct roar_time  * time, struct roar_message * mes) {
 uint64_t * data;

 if ( mes == NULL || time == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 data = (uint64_t*)mes->data;

 if ( mes->datalen < 8 ) {
  roar_err_set(ROAR_ERROR_BADMSG);
  return -1;
 }

 if ( mes->datalen % 8 ) {
  roar_err_set(ROAR_ERROR_BADMSG);
  return -1;
 }

 if ( ROAR_NET2HOST64(data[0]) != 0 ) {
  roar_err_set(ROAR_ERROR_NSVERSION);
  return -1;
 }

 memset(time, 0, sizeof(struct roar_time));

 if ( mes->datalen >= 16 ) {
  time->t_sec = ROAR_NET2HOST64(data[1]);
  if ( mes->datalen >= 24 ) {
   time->t_ssec = ROAR_NET2HOST64(data[2]);
   if ( mes->datalen >= 32 ) {
    time->c_freq = ROAR_NET2HOST64(data[3]);
    if ( mes->datalen >= 40 ) {
     time->c_drift = ROAR_NET2HOST64(data[4]);
    }
   }
  }
 }

 return 0;
}

int roar_get_time  (struct roar_connection * con, struct roar_time * time) {
 struct roar_message mes;

 if ( con == NULL || time == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 memset(&mes, 0, sizeof(mes));

 mes.cmd     = ROAR_CMD_GETTIMEOFDAY;
 mes.stream  = -1;
 mes.datalen = 8; /* 64 bit */
 // mes.data is already cleared.

 if ( roar_req(con, &mes, NULL) == -1 )
  return -1;

 if ( mes.cmd != ROAR_CMD_OK ) {
  return -1;
 }

 if ( roar_time_from_msg(time, &mes) == -1 )
  return -1;

 return 0;
}

int roar_clock_gettime(struct roar_time * rt, int clock) {
#ifdef ROAR_HAVE_GETTIMEOFDAY
 struct timeval tv;
#elif defined(ROAR_HAVE_TIME)
 time_t now;
#endif

 ROAR_DBG("roar_clock_gettime(rt=%p, clock=%i) = ?", rt, clock);

 if ( rt == NULL ) {
  roar_err_set(ROAR_ERROR_FAULT);
  return -1;
 }

 memset(rt, 0, sizeof(struct roar_time));

 if ( clock == ROAR_CLOCK_DEFAULT )
  clock = ROAR_CLOCK_REALTIME;

 switch (clock) {
  case ROAR_CLOCK_REALTIME:
    ROAR_DBG("roar_clock_gettime(rt=%p, clock=(%i)ROAR_CLOCK_REALTIME) = ?", rt, clock);
#ifdef ROAR_HAVE_GETTIMEOFDAY
    if ( gettimeofday(&tv, NULL) == -1 ) {
     roar_err_from_errno();
     ROAR_DBG("roar_clock_gettime(rt=%p, clock=(%i)ROAR_CLOCK_REALTIME) = -1 //error=%i", rt, clock, roar_error);
     return -1;
    }
    rt->t_sec  = tv.tv_sec;
    rt->t_ssec = (uint64_t)tv.tv_usec * (uint64_t)18446744073709ULL;
    rt->c_freq = 1000000LLU*1000000000LLU;
#elif defined(ROAR_HAVE_TIME)
    now = time(NULL);
    rt->t_sec  = now;
    rt->c_freq = 1000000000LLU;
#else
    ROAR_DBG("roar_clock_gettime(rt=%p, clock=(%i)ROAR_CLOCK_REALTIME) = -1 // error=NOTSUP", rt, clock);
    roar_err_set(ROAR_ERROR_NOTSUP);
    return -1;
#endif
    ROAR_DBG("roar_clock_gettime(rt=%p, clock=(%i)ROAR_CLOCK_REALTIME) = 0", rt, clock);
    return 0;
   break;
 }

 ROAR_DBG("roar_clock_gettime(rt=%p, clock=%i) = -1 // error=NOTSUP", rt, clock);
 roar_err_set(ROAR_ERROR_NOTSUP);
 return -1;
}

//ll
