#ifndef _RHEOLEF_GEOELEMENT_H
#define _RHEOLEF_GEOELEMENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// prototype for the geo_element hierarchy of classes
// and used for non-regression tests of the polymorphic_array class
//
// author: Pierre.Saramito@imag.fr
//
// date: 19 december 2010
//
#include "rheolef/distributed.h"
#include "rheolef/polymorphic_traits.h"

#include <boost/array.hpp>
#include <boost/mpl/vector.hpp>
#include <boost/mpl/size.hpp>

namespace rheolef {

class geo_element /* : boost::noncopyable */ {
public:
    typedef std::vector<int>::size_type size_type;
    typedef enum {
	p = 0, 
	e = 1, 
	t = 2, 
	q = 3, 
	T = 4, 
	P = 5, 
	H = 6, 
	enum_type_max = 7 
    } enum_type; 

    static char tab_name [enum_type_max];

    virtual ~geo_element () {}
    virtual enum_type type() const = 0;
    virtual char      name() const = 0;
    virtual size_type size() const = 0;
    virtual size_type dimension() const = 0;
    virtual size_type  operator[] (size_type i) const = 0;
    virtual size_type& operator[] (size_type i)       = 0;
    template <class Pool> struct get;
};
inline
std::ostream&
operator<< (std::ostream& os, const geo_element& K) {
	os << K.name() << "\t";
        for (geo_element::size_type iloc = 0; iloc < K.size(); iloc++) {
	  os << K[iloc];
	  if (iloc < K.size() - 1) os << " ";
	}
	return os;
}
template <class Array>
void
hard_copy (const geo_element& K1, Array& K2) {
  	check_macro (K1.size() == K2.size(), "invalid element sizes: K1.size="<<K1.size()<<" and K2.size="<<K2.size());
        for (geo_element::size_type iloc = 0; iloc < K1.size(); iloc++) {
       	    K2[iloc] = K1[iloc];
	}
} 
class geo_element_t : public geo_element {
    static const enum_type _type =  t;
    static const char      _name = 't';
    static const size_type _size =  3;
    static const size_type _dim  =  2;
public:
    typedef boost::array<size_type,_size> array_type;

    geo_element_t () : geo_element() {
	std::fill (_vertex.begin(), _vertex.end(), std::numeric_limits<size_type>::max());
    }
    geo_element_t (size_type a, size_type b, size_type c) : geo_element() {
	_vertex[0] = a;
	_vertex[1] = b;
	_vertex[2] = c;
    } 
    geo_element_t (const array_type& K) {  _vertex = K; }
    geo_element_t& operator= (const geo_element& K) {
	check_macro (K.type() == type(), "incompatible assignment from `"<<K.name()<<"' to `"<<name()<<"' geo_element");	
	hard_copy (K, _vertex);
        return *this;
    }
    ~geo_element_t () {}
    enum_type type() const      { return _type; };
    char      name() const      { return _name; };
    size_type size() const      { return _size; }
    size_type dimension() const { return _dim; };
    size_type  operator[] (size_type i) const { return _vertex[i]; }
    size_type& operator[] (size_type i)       { return _vertex[i]; }
    template<class Archive>
    void serialize (Archive& ar, const unsigned int version) { ar & _vertex; }
// data:
protected:
    array_type _vertex;
};
// =====================================================================
class geo_element_q : public geo_element {
    static const enum_type _type =  q;
    static const char      _name = 'q';
    static const size_type _size =  4;
    static const size_type _dim  =  2;
public:
    typedef boost::array<size_type,_size> array_type;
    geo_element_q () : geo_element() { 
	std::fill (_vertex.begin(), _vertex.end(), std::numeric_limits<size_type>::max());
    }
    geo_element_q (size_type a, size_type b, size_type c, size_type d) : geo_element() {
	_vertex[0] = a;
	_vertex[1] = b;
	_vertex[2] = c;
	_vertex[3] = d;
    } 
    geo_element_q (const array_type& K) {  _vertex = K; }
    geo_element_q& operator= (const geo_element& K) {
	check_macro (K.type() == type(), "incompatible assignment from `"<<K.name()<<"' to `"<<name()<<"' geo_element");	
	hard_copy (K, _vertex);
        return *this;
    }
    ~geo_element_q () {}
    enum_type type() const      { return _type; };
    char      name() const      { return _name; };
    size_type size() const      { return _size; }
    size_type dimension() const { return _dim; };
    size_type  operator[] (size_type i) const { return _vertex[i]; }
    size_type& operator[] (size_type i)       { return _vertex[i]; }
    template<class Archive> 
    void serialize (Archive& ar, const unsigned int version) { ar & _vertex; }
// data:
protected:
    array_type _vertex;
};
class geo_element_T : public geo_element {
    static const enum_type _type =  T;
    static const char      _name = 'T';
    static const size_type _size =  4;
    static const size_type _dim  =  3;
public:
    typedef boost::array<size_type,_size> array_type;
    geo_element_T () : geo_element() { 
	std::fill (_vertex.begin(), _vertex.end(), std::numeric_limits<size_type>::max());
    }
    geo_element_T (size_type a, size_type b, size_type c, size_type d) : geo_element() {
	_vertex[0] = a;
	_vertex[1] = b;
	_vertex[2] = c;
	_vertex[3] = d;
    } 
    geo_element_T (const array_type& K) {  _vertex = K; }
    geo_element_T& operator= (const geo_element& K) {
	check_macro (K.type() == type(), "incompatible assignment from `"<<K.name()<<"' to `"<<name()<<"' geo_element");	
	hard_copy (K, _vertex);
        return *this;
    }
    ~geo_element_T () {}
    enum_type type() const      { return _type; };
    char      name() const      { return _name; };
    size_type size() const      { return _size; }
    size_type dimension() const { return _dim; };
    size_type  operator[] (size_type i) const { return _vertex[i]; }
    size_type& operator[] (size_type i)       { return _vertex[i]; }
    template<class Archive> 
    void serialize (Archive& ar, const unsigned int version) { ar & _vertex; }
// data:
protected:
    array_type _vertex;
};
class geo_element_P : public geo_element {
    static const enum_type _type =  P;
    static const char      _name = 'P';
    static const size_type _size =  6;
    static const size_type _dim  =  3;
public:
    typedef boost::array<size_type,_size> array_type;
    geo_element_P () : geo_element() { 
	std::fill (_vertex.begin(), _vertex.end(), std::numeric_limits<size_type>::max());
    }
    geo_element_P (size_type a, size_type b, size_type c, size_type d,
                   size_type e, size_type f) : geo_element() {
	_vertex[0] = a;
	_vertex[1] = b;
	_vertex[2] = c;
	_vertex[3] = d;
	_vertex[4] = e;
	_vertex[5] = f;
    } 
    geo_element_P (const array_type& K) {  _vertex = K; }
    geo_element_P& operator= (const geo_element& K) {
	check_macro (K.type() == type(), "incompatible assignment from `"<<K.name()<<"' to `"<<name()<<"' geo_element");	
	hard_copy (K, _vertex);
        return *this;
    }
    ~geo_element_P () {}
    enum_type type() const      { return _type; };
    char      name() const      { return _name; };
    size_type size() const      { return _size; }
    size_type dimension() const { return _dim; };
    size_type  operator[] (size_type i) const { return _vertex[i]; }
    size_type& operator[] (size_type i)       { return _vertex[i]; }
    template<class Archive> 
    void serialize (Archive& ar, const unsigned int version) { ar & _vertex; }
// data:
protected:
    array_type _vertex;
};
class geo_element_H : public geo_element {
    static const enum_type _type =  H;
    static const char      _name = 'H';
    static const size_type _size =  8;
    static const size_type _dim  =  3;
public:
    typedef boost::array<size_type,_size> array_type;
    geo_element_H () : geo_element() { 
	std::fill (_vertex.begin(), _vertex.end(), std::numeric_limits<size_type>::max());
    }
    geo_element_H (size_type a, size_type b, size_type c, size_type d,
                   size_type e, size_type f, size_type g, size_type h) : geo_element() {
	_vertex[0] = a;
	_vertex[1] = b;
	_vertex[2] = c;
	_vertex[3] = d;
	_vertex[4] = e;
	_vertex[5] = f;
	_vertex[6] = g;
	_vertex[7] = h;
    } 
    geo_element_H (const array_type& K) {  _vertex = K; }
    geo_element_H& operator= (const geo_element& K) {
	check_macro (K.type() == type(), "incompatible assignment from `"<<K.name()<<"' to `"<<name()<<"' geo_element");	
	hard_copy (K, _vertex);
        return *this;
    }
    ~geo_element_H () {}
    enum_type type() const      { return _type; };
    char      name() const      { return _name; };
    size_type size() const      { return _size; }
    size_type dimension() const { return _dim; };
    size_type  operator[] (size_type i) const { return _vertex[i]; }
    size_type& operator[] (size_type i)       { return _vertex[i]; }
    template<class Archive> 
    void serialize (Archive& ar, const unsigned int version) { ar & _vertex; }
// data:
protected:
    array_type _vertex;
};
//
// input associated to pool allocation: class function
//
template <class Pool>
struct geo_element::get {
  geo_element* operator() (std::istream& is, Pool& pool) {
    char c;
    is >> std::ws >> c;
    geo_element* ptr = 0;
    switch (c) {
      case 't' : {
		ptr = pool.template new_entry<geo_element_t>();
		break;
		}   
      case 'q' : {
		ptr = pool.template new_entry<geo_element_q>();
		break;
      		}
      case 'T' : {
		ptr = pool.template new_entry<geo_element_T>();
		break;
      		}
      case 'P' : {
		ptr = pool.template new_entry<geo_element_P>();
		break;
      		}
      case 'H' : {
		ptr = pool.template new_entry<geo_element_H>();
		break;
      		}
      default: error_macro ("unexpected element variant `"<<c<<"'");
    }
    geo_element& K = *ptr;
    for (size_t i = 0, n = K.size(); i < n; i++) {
	is >> K[i];
    }
    return ptr;
  }
};
// -------------------------------------------------------------------
// helper for long typedef list:
// -------------------------------------------------------------------

namespace mpl = boost::mpl;

template <>
struct polymorphic_traits<geo_element> {
    typedef mpl::vector5<
	geo_element_t,
	geo_element_q,
	geo_element_T,
	geo_element_P,
	geo_element_H>
    derived_type;
    static const size_t n_derived = mpl::size<derived_type>::value;
};

} // namespace rheolef

#ifdef _RHEOLEF_HAVE_MPI
// =====================================================================
// Some serializable types, like geo_element, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous 
// copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  template <> struct is_mpi_datatype<rheolef::geo_element_t> : mpl::true_ { };
  template <> struct is_mpi_datatype<rheolef::geo_element_q> : mpl::true_ { };
  template <> struct is_mpi_datatype<rheolef::geo_element_T> : mpl::true_ { };
  template <> struct is_mpi_datatype<rheolef::geo_element_P> : mpl::true_ { };
  template <> struct is_mpi_datatype<rheolef::geo_element_H> : mpl::true_ { };
 } // namespace mpi
} // namespace boost
#endif // _RHEOLEF_HAVE_MPI

#endif // _RHEOLEF_GEOELEMENT_H
