#ifndef _RHEOLEF_FORM_ELEMENT_H
#define _RHEOLEF_FORM_ELEMENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/scalar_traits.h"
#include "rheolef/element.h"
#include "rheolef/geo.h"
#include "rheolef/quadrature.h"
#include "rheolef/basis_on_lattice.h"
#include "rheolef/tensor.h"

#include <boost/numeric/ublas/matrix.hpp>

namespace rheolef {
namespace ublas = boost::numeric::ublas;

/*Class:form_element
NAME: @code{form_element} - bilinear form on a single element
@cindex  bilinear form
@cindex  geometrical element
@clindex basis
@clindex numbering
@clindex geo_element
SYNOPSYS:
  @noindent
  The @code{form_element} class defines functions that compute
  a bilinear form defined between two polynomial basis on a single
  geometrical element. This bilinear form is represented
  by a matrix.

  The bilinear form is designated by a string, e.g. "mass", "grad_grad", ...
  indicating the form. The form depends also of the geometrical element:
  triangle, square, tetrahedron (@pxref{geo_element internal}).

IMPLEMENTATION NOTE:
  @noindent
  The @code{form_element} class 
  is managed by (@pxref{smart_pointer internal}).
  This class uses a pointer on a pure virtual class @code{form_element_rep}
  while the effective code refers to the specific concrete derived classes:
  mass, grad_grad, etc.

AUTHORS: Pierre.Saramito@imag.fr
DATE:   7 january 2004
End:
*/

template<class T, class M>
class form_element_rep {
public:
// typedefs:
    typedef geo_element::size_type          size_type;
    typedef typename scalar_traits<T>::type float_type;
    typedef basic_point<float_type>         vertex_type;
    typedef element<float_type>             element_type;
    typedef geo_basic<float_type,M>         geo_type;

// allocators:

    form_element_rep();

    form_element_rep(
	const element_type&     X,
	const element_type&     Y,
	const geo_type&         omega);

    virtual ~form_element_rep();
    void     initialize_all();

// the main functionality, virtual, defined in concrete derived classes:

    virtual void operator() (const geo_element& K, ublas::matrix<T>& m) const = 0;
    virtual void initialize () = 0;
    virtual bool is_symmetric () const = 0;

// accessors:

    const element_type& get_first_element() const;
    const element_type& get_second_element() const;

    const basis&  get_first_basis() const;
    const basis&  get_second_basis() const;
    const basis&  get_p1_transformation () const;
    size_type     coordinate_dimension() const;
    size_type     n_derivative() const;
    void          set_n_derivative (size_type n);

    const vertex_type& dis_vertex(size_type dis_iv) const;

// predefined forms:

    void build_scalar_mass      (const geo_element& K, ublas::matrix<T>& m) const;
    void build_scalar_grad_grad (const geo_element& K, ublas::matrix<T>& a) const;

// piola related:

    void  jacobian_piola_transformation (const geo_element& K, size_type q, tensor& DF) const;
    T det_jacobian_piola_transformation (const geo_element& K, size_type q) const;
    T det_jacobian_piola_transformation (const tensor& DF, size_type d) const;
    tensor pseudo_inverse_jacobian_piola_transformation (const tensor& DF, size_type map_d) const;

// utilities:

    static void cumul_otimes (
    	ublas::matrix<T>& m, 
    	const T& w,
    	basis_on_quadrature::const_iterator phi,
    	basis_on_quadrature::const_iterator last_phi,
    	basis_on_quadrature::const_iterator first_psi,
    	basis_on_quadrature::const_iterator last_psi);

    static void cumul_otimes (
        ublas::matrix<T>& m,
	const tensor& Dw,
	basis_on_quadrature::const_iterator_grad grad_phi,
	basis_on_quadrature::const_iterator_grad last_grad_phi,
	basis_on_quadrature::const_iterator_grad first_grad_psi,
	basis_on_quadrature::const_iterator_grad last_grad_psi,
	size_type map_d);

#ifdef TODO

    // for scalar-weighted forms:
    void set_weight (const field& wh) const;

    const numbering& get_first_numbering() const;
    const numbering& get_second_numbering() const;

    const geo&       get_global_geo() const;
    fem_helper::coordinate_type coordinate_system_type() const;

    bool is_weighted() const;
    const field& get_weight () const;
    const basis& get_weight_basis () const;
    T weight (const geo_element& K, size_type q, size_t i_comp = 0) const;
    void  weight (const geo_element& K, size_type q, tensor& W) const;


// toolbox for building elementary forms:

    point piola_transformation (const geo_element& K, size_type q) const;

    T weight_coordinate_system (const geo_element& K, size_type q) const;
    static void cumul_otimes (
	ublas::matrix<T>& m, 
	const T& w,
	basis_on_quadrature::const_iterator phi,
	basis_on_quadrature::const_iterator last_phi,
	basis_on_quadrature::const_iterator_grad first_grad_psi,
	basis_on_quadrature::const_iterator_grad last_grad_psi);
    static void cumul_otimes (
	ublas::matrix<T>& m, 
	const T& w,
	basis_on_quadrature::const_iterator phi,
	basis_on_quadrature::const_iterator last_phi,
	basis_on_quadrature::const_iterator_hessian first_hessian_psi,
	basis_on_quadrature::const_iterator_hessian last_hessian_psi);
    static void cumul_otimes (
	ublas::matrix<T>& m, 
	const T& w,
	basis_on_quadrature::const_iterator_grad grad_phi,
	basis_on_quadrature::const_iterator_grad last_grad_phi,
	basis_on_quadrature::const_iterator_grad first_grad_psi,
	basis_on_quadrature::const_iterator_grad last_grad_psi);

    void build_general_mass(const geo_element& K, ublas::matrix<T>& m) const;
    void build_d_ds        (const geo_element& K, ublas::matrix<T>& m) const;
    void build_d2_ds2 	   (const geo_element& K, ublas::matrix<T>& m) const;
    void build_d_ds_d_ds   (const geo_element& K, ublas::matrix<T>& m) const;
    void build_grad_grad   (const geo_element& K, ublas::matrix<T>& m) const;
    void build_gradt_grad  (const geo_element& K, ublas::matrix<T>& m) const;
    void build_div_div     (const geo_element& K, ublas::matrix<T>& m) const;
    void d_dx              (const geo_element& K, ublas::matrix<T>& m, size_type idx) const;

    void set_use_coordinate_system_weight (bool use) const;
    bool use_coordinate_system_weight () const;
    void set_use_coordinate_system_dual_weight (bool use) const;
    bool use_coordinate_system_dual_weight () const;
#endif // TODO

// data:
protected:
    element_type           _X;	// also origin      dof numbering
    element_type           _Y;  // also destination dof numbering
    geo_type               _omega; // for the access to distributed vertices

    bool                   _initialized;
    size_type              _n_derivative; // e.g. =2 for grad_grad and =0 for mass
    quadrature             _quad;
    basis_on_quadrature	   _bx_table;
    basis_on_quadrature	   _by_table;
    basis_on_quadrature    _tr_p1_table;

#ifdef TODO
    field                  _wh;
    bool                   _is_weighted;
    basis_on_quadrature    _bw_table;
    bool                   _use_coordinate_system_weight;
    bool                   _use_coordinate_system_dual_weight;
#endif // TODO

// internal:
public:
    // automatically generated implementation in form_element_make.cc :
    static form_element_rep<T,M>* make_ptr (
	std::string           name,
	const element_type&   X,
	const element_type&   Y,
	const geo_type&       omega);
};
// -----------------------------------------------------------------------
// inlined
// -----------------------------------------------------------------------

template<class T, class M>
inline
form_element_rep<T,M>::form_element_rep()
 : _X(),
   _Y(),
   _omega(),
   _initialized(false),
   _n_derivative(std::numeric_limits<size_type>::max()),
   _quad(),
   _bx_table(),
   _by_table(),
   _tr_p1_table()
{
}
template<class T, class M>
inline
form_element_rep<T,M>::form_element_rep (
    const element_type&     X,
    const element_type&     Y,
    const geo_type&         omega)
 : _X (X),
   _Y (Y),
   _omega (omega),
   _initialized(false),
   _n_derivative(std::numeric_limits<size_type>::max()),
   _quad(),
   _bx_table(),
   _by_table(),
   _tr_p1_table()
{
}
template<class T, class M>
inline
form_element_rep<T,M>::~form_element_rep() 
{
}
template<class T, class M>
inline
const typename form_element_rep<T,M>::vertex_type&
form_element_rep<T,M>::dis_vertex(size_type dis_iv) const
{
  return _omega.dis_vertex (dis_iv);
}
template<class T, class M>
inline
const typename form_element_rep<T,M>::element_type&
form_element_rep<T,M>::get_first_element() const
{
  return _X;
}
template<class T, class M>
inline
const typename form_element_rep<T,M>::element_type&
form_element_rep<T,M>::get_second_element() const
{
  return _Y;
}
template<class T, class M>
inline
const basis& 
form_element_rep<T,M>::get_first_basis() const
{
  return get_first_element().get_basis();
}
template<class T, class M>
inline
const basis& 
form_element_rep<T,M>::get_second_basis() const
{
  return get_first_element().get_basis();
}
template<class T, class M>
inline
const basis&
form_element_rep<T,M>::get_p1_transformation () const
{
  return get_first_element().get_p1_transformation();
}
template<class T, class M>
inline
typename form_element_rep<T,M>::size_type
form_element_rep<T,M>::coordinate_dimension() const
{
  return _omega.dimension();
}
template<class T, class M>
inline
typename form_element_rep<T,M>::size_type
form_element_rep<T,M>::n_derivative() const
{
  return _n_derivative;
}
template<class T, class M>
inline
void
form_element_rep<T,M>::set_n_derivative (size_type n)
{
  _n_derivative = n;
}
#ifdef TODO
inline
const numbering& 
form_element_rep::get_first_numbering() const
{
  return get_first_space().get_numbering();
}
inline
const numbering& 
form_element_rep::get_second_numbering() const
{
  return get_second_space().get_numbering();
}
inline
const field&
form_element_rep::get_weight () const
{
  return _wh;
}
inline
bool
form_element_rep::is_weighted() const
{
  return _is_weighted;
}
inline
const basis&
form_element_rep::get_weight_basis () const
{
  return get_weight().get_space().get_basis();
}
inline
void
form_element_rep::set_use_coordinate_system_weight (bool use) const
{
    _use_coordinate_system_weight = use;
}
inline
bool
form_element_rep::use_coordinate_system_weight () const
{
    return _use_coordinate_system_weight;
}
inline
void
form_element_rep::set_use_coordinate_system_dual_weight (bool use) const
{
    _use_coordinate_system_dual_weight = use;
}
inline
bool
form_element_rep::use_coordinate_system_dual_weight () const
{
    return _use_coordinate_system_dual_weight;
}
#endif // TODO

// ----------------------------------------------------------
// wrapper class
// ----------------------------------------------------------

//<form_element:
template <class T, class M>
class form_element : public smart_pointer<form_element_rep<T,M> > {
public:

// typedefs:

    typedef form_element_rep<T,M>      rep;
    typedef smart_pointer<rep>         base;
    typedef typename rep::size_type    size_type;
    typedef typename rep::vertex_type  vertex_type;
    typedef typename rep::element_type element_type;
    typedef typename rep::geo_type     geo_type;

// constructors:

    form_element ();
    form_element (
	std::string	      name,
	const element_type&   X,
	const element_type&   Y,
	const geo_type&       omega);

// accessors & modifier:

    void operator() (const geo_element& K, ublas::matrix<T>& m) const;
    virtual bool is_symmetric () const;

#ifdef TODO
    void set_weight (const field& wh) const;
    const field& get_weight() const;
    void set_use_coordinate_system_weight(bool use) const;
#endif // TODO
};
//>form_element:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
template <class T, class M>
inline
form_element<T,M>::form_element ()
 : base (rep::make_ptr())
{
}
template <class T, class M>
inline
form_element<T,M>::form_element (
	std::string	      name,
	const element_type&   X,
	const element_type&   Y,
	const geo_type&       omega)
 : base (rep::make_ptr(name,X,Y,omega))
{
}
template <class T, class M>
inline
void
form_element<T,M>::operator() (const geo_element& K, ublas::matrix<T>& m) const
{
    base::data().operator() (K, m);
}
template <class T, class M>
inline
bool
form_element<T,M>::is_symmetric () const
{
    return base::data().is_symmetric();
}

}// namespace rheolef
#endif // _RHEOLEF_FORM_ELEMENT_H
