#ifndef _RHEO_D_DX0_H
#define _RHEO_D_DX0_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:d_dx0
NAME: @code{d_dx}@var{i} -- derivatives
@bfindex d_dx0
@bfindex d_dx1
@bfindex d_dx2
@cindex  derivative
@clindex form
@clindex space
@apindex P0
@apindex P1
@apindex P2
@apindex P1d
SYNOPSIS:
  @example
    	form (const space V, const space& M, "d_dx0");
    	form (const space V, const space& M, "d_dx1");
    	form (const space V, const space& M, "d_dx2");
  @end example
DESCRIPTION:       
  @noindent
  Assembly the form associated to a
  derivative operator from the @code{V} finite element space
  to the @code{M} one:
@iftex
  $$
      b_i(u,q) = \int_\Omega
         { {\partial u} \over {\partial x_i}} \, q \, dx,
	\ \ i\in \{ 0,1,2 \}
  $$
@end iftex
@ifnottex
  @example
                 /
                 |  d u
      b_i(u,q) = |  ----  q  dx,  i = 0,1,2
                 |  d xi
                 / Omega
  @end example
@end ifnottex
  In the axisymetric @code{rz} case, the form is defined by
@iftex
  $$
      b_0(u,q) = \int_\Omega 
         { {\partial u} \over {\partial r}} \, q \, r dr dz,
  $$
@end iftex
@ifnottex
  @example
                 /
                 |  d u
      b_0(u,q) = |  ---  q  r dr dz
                 |  d r
                 / Omega
  @end example
@end ifnottex
  @noindent
  If the V space is a @code{P1} (resp. @code{P2})
  finite element space, the M space may be
  a @code{P0} (resp. @code{P1d}) one.

EXAMPLE:
  @noindent
  The following piece of code build the Laplacian form
  associated to the P1 approximation:
  @example
   	geo omega ("square");
   	space V (omega, "P1");
   	space M (omega, "P0");
   	form  b (V, M, "d_dx0");
  @end example
LIMITATIONS:
  @noindent
  Only edge, triangular and tetrahedal finite element meshes are yet supported.
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
SEE ALSO:
    "form"(3)
DATE:
    28 november 1997
End:
*/

#include "rheolef/form_element_rep.h"
namespace rheolef { 

class d_dx0: public form_element_rep {

public:
// allocator:

    d_dx0() : form_element_rep() {}

// virtual accessor:

    void operator() (const geo_element& K, ublas::matrix<Float>& m) const;
    size_type n_derivative() const;
};
}// namespace rheolef
#endif // _RHEO_D_DX0_H
