///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// convect a.grad matrix for scalar, vectorial and tensorial valued space
//
// for symmetric tensorial valued space, the matrix convect is block-diagonal
// with a 2 factor for (i,j) blocks, 0 <= i,j < 3 when i != j
//
#include "convect.h"
#include "rheolef/ublas_matrix_range.h"
using namespace std;
using namespace ublas;
namespace rheolef { 

static
double 
integrate1(double a0,double a1,double a2,double c, double d) 
{
  return a0/3*(d*d*d-c*c*c) + a1/4*(d*d*d*d-c*c*c*c) + \
    a2/5*(d*d*d*d*d-c*c*c*c*c);
}
static
double
integrate2(double a0,double a1,double a2,double c, double d) 
{
  return a0/2*(d*d-c*c) + (a1-a0)/3*(d*d*d-c*c*c) + \
    (a2-a1)/4*(d*d*d*d-c*c*c*c) - a2/5*(d*d*d*d*d-c*c*c*c*c);
}
static
double
integrate3(double a0,double a1,double a2,double c, double d) 
{
  return a0*(d-c) + (a1-2*a0)/2*(d*d-c*c) + \
    (a2-2*a1+a0)/3*(d*d*d-c*c*c) + (-2*a2+a1)/4*(d*d*d*d-c*c*c*c) + \
    a2/5*(d*d*d*d*d-c*c*c*c*c);
}
static
void
compute_int(double a0,double a1,double a2,double s,
		 double& int1,double& int2,double& int3) 
{
  double delta = a1*a1-4*a0*a2;
  if ((a2==0)&&(a1==0)) {
    if (a0>=0) {
      int1 = 0 ; int2 = 0; int3 = 0;
    }
    else {
      int1 = integrate1(a0,a1,a2,0,1); 
      int2 = integrate2(a0,a1,a2,0,1); 
      int3 = integrate3(a0,a1,a2,0,1); 
    }
  }
  else if ( (a2==0) && (a1!=0) ) {
    if ( (-a0/a1<=0) || (-a0/a1>=1) ) {
      if ( (a0>0) || ((a0==0)&&(a1>0)) ) {
	int1 = 0 ; int2 = 0; int3 = 0;
      }
      else {
	int1 = integrate1(a0,a1,a2,0,1); 
	int2 = integrate2(a0,a1,a2,0,1); 
	int3 = integrate3(a0,a1,a2,0,1); 
      }
    }
    else {
      if (a0>0) {
	int1 = integrate1(a0,a1,a2,-a0/a1,1); 
	int2 = integrate2(a0,a1,a2,-a0/a1,1); 
	int3 = integrate3(a0,a1,a2,-a0/a1,1); 
      }
      else {
	int1 = integrate1(a0,a1,a2,0,-a0/a1); 
	int2 = integrate2(a0,a1,a2,0,-a0/a1); 
	int3 = integrate3(a0,a1,a2,0,-a0/a1); 
      }
    }
  }
  else if ( (a2!=0) && (delta<=0) ) {
    if (a2>0) {
      int1 = 0 ; int2 = 0; int3 = 0;
    }
    else {
      int1 = integrate1(a0,a1,a2,0,1); 
      int2 = integrate2(a0,a1,a2,0,1); 
      int3 = integrate3(a0,a1,a2,0,1); 
    }
  }
  else if ( (a2!=0) && (delta>0) ) {
    double r1 = (-a1+::sqrt(delta))/(2*a2);
    double r2 = (-a1-::sqrt(delta))/(2*a2);
    double s1 = (r1>r2) ? r2 : r1;
    double s2 = (r1<r2) ? r2 : r1;
    if ( ((s1<=0)||(s1>=1)) && ((s2<=0)||(s2>=1)) ) {
      if ( (a0>0) || ((a0==0)&&(a1>0)) ) {
	int1 = 0 ; int2 = 0; int3 = 0;
      }
      else {
	int1 = integrate1(a0,a1,a2,0,1); 
	int2 = integrate2(a0,a1,a2,0,1); 
	int3 = integrate3(a0,a1,a2,0,1); 
      }
    }
    else if ( ((s1>0)&&(s1<1)) && ((s2<=0)||(s2>=1)) ) {
      if ( (a0>0) || ((a0==0)&&(a1>0)) ) {
	int1 = integrate1(a0,a1,a2,s1,1); 
	int2 = integrate2(a0,a1,a2,s1,1); 
	int3 = integrate3(a0,a1,a2,s1,1); 
      }
      else if ( (a0<0) || ((a0==0)&&(a1<0)) ) {
	int1 = integrate1(a0,a1,a2,0,s1); 
	int2 = integrate2(a0,a1,a2,0,s1); 
	int3 = integrate3(a0,a1,a2,0,s1); 
      }
    }
    else if ( ((s2>0)&&(s2<1)) && ((s1<=0)||(s1>=1)) ) {
      if ( (a0>0) || ((a0==0)&&(a1>0)) ) {
	int1 = integrate1(a0,a1,a2,s2,1); 
	int2 = integrate2(a0,a1,a2,s2,1); 
	int3 = integrate3(a0,a1,a2,s2,1); 
      }
      else if ( (a0<0) || ((a0==0)&&(a1<0)) ){
	int1 = integrate1(a0,a1,a2,0,s2); 
	int2 = integrate2(a0,a1,a2,0,s2); 
	int3 = integrate3(a0,a1,a2,0,s2); 
      }
    }
    else if ( ((s1>0)&&(s1<1)) && ((s2>0)&&(s2<1)) ) {
      if (a0>0) {
	int1 = integrate1(a0,a1,a2,s1,s2); 
	int2 = integrate2(a0,a1,a2,s1,s2); 
	int3 = integrate3(a0,a1,a2,s1,s2); 
      }
      else {
	int1 = integrate1(a0,a1,a2,0,s1) + integrate1(a0,a1,a2,s2,1); 
	int2 = integrate2(a0,a1,a2,0,s1) + integrate2(a0,a1,a2,s2,1);
	int3 = integrate3(a0,a1,a2,0,s1) + integrate3(a0,a1,a2,s2,1); 
      }
    }
  }
  int1 *= s;
  int2 *= s;
  int3 *= s;
}
void 
convect::operator() (const geo_element& K, ublas::matrix<Float>& m) const
{
  check_macro (K.size() == 3, "triangular element expected.");
  m.resize (3,9);
  m.clear();
  const field& uh = get_weight();
  geo::const_iterator_node p = begin_vertex();
  const point& p0 = p[K[0]];
  const point& p1 = p[K[1]];
  const point& p2 = p[K[2]];
  
  /////////////////// edge sigma_0 ////////////////
  double s = dist(p0,p1);
  
  //outward normal
  point n((p1[1]-p0[1])/s,(p0[0]-p1[0])/s);
  
  //v.n is evaluated in p0, (p0+p1)/2, p1
  double vn0 = dot(uh.vector_evaluate(p0),n);
  double vnm = dot(uh.vector_evaluate(0.5*(p0+p1)),n);
  double vn1 = dot(uh.vector_evaluate(p1),n);
  double a0 = vn0;
  double a1 = -3*vn0-vn1+4*vnm;
  double a2 = 2*(vn0+vn1-2*vnm);
  
  // int1 is the integral of alpha_K*s^2 over the edge
  // int2 is the integral of alpha_K*s(1-s) over the edge
  // int3 is the integral of alpha_K*(1-s)^2 over the edge
  double int1, int2, int3; 
  compute_int(a0,a1,a2,s,int1,int2,int3);
  m(0,0) += int3;
  m(1,0) += int2;
  m(0,1) += int2;
  m(1,1) += int1;
  m(0,3) -= int2;
  m(1,3) -= int1;
  m(0,4) -= int3; 
  m(1,4) -= int2;
  
  /////////////////// edge sigma_1 ////////////////
  s = dist(p1,p2);
  //outward normal
  n = point((p2[1]-p1[1])/s,(p1[0]-p2[0])/s);
  //v.n is evaluated in p1, (p1+p2)/2, p2
  vn0 = dot(uh.vector_evaluate(p1),n);
  vnm = dot(uh.vector_evaluate(0.5*(p1+p2)),n);
  vn1 = dot(uh.vector_evaluate(p2),n);
  a0 = vn0;
  a1 = -3*vn0-vn1+4*vnm;
  a2 = 2*(vn0+vn1-2*vnm);
  compute_int(a0,a1,a2,s,int1,int2,int3);
  m(2,2) += int1;
  m(1,2) += int2;
  m(2,1) += int2;
  m(1,1) += int3;
  m(2,5) -= int1;
  m(1,5) -= int2;
  m(2,6) -= int2;
  m(1,6) -= int3;
  
  /////////////////// edge sigma_2 ////////////////
  s = dist(p2,p0);
  //outward normal
  n = point((p0[1]-p2[1])/s,(p2[0]-p0[0])/s);
  //v.n is evaluated in p2, (p2+p0)/2, p0
  vn0 = dot(uh.vector_evaluate(p2),n);
  vnm = dot(uh.vector_evaluate(0.5*(p2+p0)),n);
  vn1 = dot(uh.vector_evaluate(p0),n);
  a0 = vn0;
  a1 = -3*vn0-vn1+4*vnm;
  a2 = 2*(vn0+vn1-2*vnm);
  compute_int(a0,a1,a2,s,int1,int2,int3);
  m(2,2) += int3;
  m(0,2) += int2;
  m(2,0) += int2;
  m(0,0) += int1;
  m(2,7) -= int2;
  m(0,7) -= int1;
  m(2,8) -= int3;
  m(0,8) -= int2;
}
void
convect::check_after_initialize () const
{
  check_macro (get_first_space() == get_second_space(),
    "incompatible spaces for the `convect' form.");
  check_macro (get_first_space().get_approx() == "P1d",
    "`convect' form: P1d approx required");
}
}// namespace rheolef
