#ifndef _RHEO_GEO_ELEMENT_V1_H
#define _RHEO_GEO_ELEMENT_V1_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:geo_element
NAME:  @code{geo_element} - element of a mesh
@cindex  geometrical element
@clindex reference element
@clindex geo_element
@clindex reference_element
@clindex geo
DESCRIPTION:
  Defines geometrical elements and sides
  as a set of vertice and edge indexes.
  This element is obtained after a Piola transformation
  from a reference element (@pxref{reference_element internal}).
  Indexes are related to arrays of edges and vertices.
  These arrays are included in the description of the mesh.
  Thus, this class is related of a given mesh instance
  (@pxref{geo class}).
EXAMPLE:
  This is the test of geo_element:
  @example
    geo_element K;
    K.set_name('t') ;
    cout << "n_vertices: " << K.size()      << endl
         << "n_edges   : " << K.n_edges()   << endl
         << "dimension : " << K.dimension() << endl << endl;
    for(geo_element::size_type i = 0; i < K.size(); i++) 
        K[i] = i*10 ;
    for(geo_element::size_type i = 0; i < K.n_edges(); i++)
        K.set_edge(i, i*10+5) ;
    cout << "vertices: local -> global" << endl;
    for (geo_element::size_type vloc = 0; vloc < K.size(); vloc++)
        cout << vloc << "-> " << K[vloc] << endl;
    cout << endl 
         << "edges: local -> global" << endl;
    for (geo_element::size_type eloc = 0; eloc < K.n_edges(); eloc++) @{
        geo_element::size_type vloc1 = subgeo_local_vertex(1, eloc, 0);
        geo_element::size_type vloc2 = subgeo_local_vertex(1, eloc, 1);
        cout << eloc << "-> " << K.edge(eloc) << endl
             << "local_vertex_from_edge(" << eloc 
             << ") -> (" << vloc1 << ", " << vloc2 << ")" << endl;
    @}
  @end example
SEE ALSO: "geo"(3)
AUTHOR: Pierre.Saramito@imag.fr
DATE:   18 jan 1998
METHODS: @geo_element
End:
*/

#include "rheolef/reference_element.h"

namespace rheolef { 
//<geo_element:
class geo_element : public reference_element {
public:

// allocators/deallocators:

    geo_element(enum_type t = max_size);
    geo_element(const geo_element&);
    explicit geo_element (const class tiny_element&);
    void copy (const geo_element&);
    void copy (const class tiny_element&);
    geo_element& operator = (const geo_element&);
    ~geo_element();

// accessors:

    size_type index() const;
    size_type operator [] (size_type i) const;
    size_type side(size_type i_side) const;
    void build_side(size_type i_side, geo_element& S) const;


    size_type edge (size_type i_edge) const;
    size_type face (size_type i_face) const;

    size_type subgeo(const geo_element& S) const;
    size_type subgeo (size_type subgeo_dim, size_type i_subgeo) const;
    size_type subgeo_vertex (size_type subgeo_dim, size_type i_subgeo,
			   size_type i_subgeo_vertex) const;
    void build_subgeo(size_type subgeo_dim, size_type i_subgeo, geo_element& S) const;
    size_type subgeo_local_index(const geo_element& S) const;

// modifiers:

    void set_type (enum_type t);
    void set_type (size_type sz, size_type dim);
    void set_name (char      name);

    void set_index(size_type idx);
    size_type& operator [] (size_type i);
    void set_side (size_type i_side, size_type idx);

    void set_edge (size_type i_edge, size_type idx);
    void set_face (size_type i_face, size_type idx);
    void set_subgeo (size_type subgeo_dim, size_type i_subgeo, 
			   size_type idx);
    void set_subgeo(const geo_element& S, size_type idx);

// inputs/outputs:

    friend std::istream& operator >> (std::istream&, geo_element&);
    friend std::ostream& operator << (std::ostream&, const geo_element&);
    std::ostream& dump(std::ostream& s = std::cerr) const;
    void check() const;

// data:
protected:
    size_type *_heap;

// memory management:
    void _heap_init();
    void _heap_close();
};
//>geo_element:

inline
geo_element::geo_element(enum_type t)
 : reference_element(t), _heap(0) 
{
    _heap_init();
}
inline
geo_element::~geo_element()
{
    _heap_close();
}
inline
void
geo_element::set_type (enum_type t)
{
    reference_element::set_type(t);
    _heap_init();
}
inline
void
geo_element::set_name (char name)
{
    reference_element::set_name(name);
    _heap_init();
}
inline
void 
geo_element::set_type (size_type n_vertex, size_type dim)
{
    reference_element::set_type(n_vertex,dim);
    _heap_init();
}
inline
geo_element::size_type
geo_element::index() const
{
    return *_heap;
}
inline
void
geo_element::set_index(size_type idx)
{
    *_heap = idx;
}
inline
geo_element::size_type
geo_element::operator [] (size_type i) const
{
    return *(_heap + heap_offset(0) + i);
}
inline
geo_element::size_type&
geo_element::operator [] (size_type i)
{
    return *(_heap + heap_offset(0) + i);
}
inline
geo_element::size_type
geo_element::subgeo (size_type subgeo_dim, size_type i_subgeo) const
{
    return *(_heap + heap_offset(subgeo_dim) + i_subgeo);
}
inline
void
geo_element::set_subgeo (size_type subgeo_dim, size_type i_subgeo,
	size_type idx)
{
    *(_heap + heap_offset(subgeo_dim) + i_subgeo) = idx;
}
inline
geo_element::size_type
geo_element::edge (size_type i_edge) const
{
    return subgeo (1, i_edge);
}
inline
geo_element::size_type
geo_element::face (size_type i_edge) const
{
    return subgeo (2, i_edge);
}
inline
geo_element::size_type
geo_element::side (size_type i_side) const
{
    return subgeo (dimension()-1, i_side);
}
inline
void
geo_element::set_edge (size_type i_edge, size_type idx)
{
    set_subgeo(1,i_edge,idx);
}
inline
void
geo_element::set_face (size_type i_face, size_type idx)
{
    set_subgeo(1,i_face,idx);
}
inline
void
geo_element::set_side (size_type i_side, size_type idx)
{
    set_subgeo(dimension()-1,i_side,idx);
}
inline
geo_element::size_type
geo_element::subgeo_vertex (size_type subgeo_dim, size_type i_subgeo,
			   size_type i_subgeo_vertex) const
{
    size_type local_vertex = subgeo_local_vertex(subgeo_dim, i_subgeo, i_subgeo_vertex);
    return operator[] (local_vertex);
}
inline
void
geo_element::build_side(size_type i_side, geo_element& S) const
{
    build_subgeo(dimension()-1, i_side, S);
}
}// namespace rheolef
#endif // _RHEO_GEO_ELEMENT_V1_H
