/*
 * tests.c - common code for the tests
 *
 * refdbg - GObject refcount debugger
 * Copyright (C) 2010 Stefan Kost <ensonic@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib-object.h>

#include "refdbg.h"

/* setup refdbg and finalize the test run */

void
test_init (const gchar *arg0)
{
  gchar *options = g_strdup_printf ("logname='%s.log'",arg0);

  refdbg_cmd (options);
  g_free (options);
}

static
canon_log_line (gchar *l)
{
  gchar *s;

  /* nuke hex addresses */
  s=l;
  while (s=strstr(s,"0x")) {
     s+=2;
     while (*s && isxdigit(*s)) {
       *s++='0';
     }
  }
  /* nuke timestamps addresses */
  s=l;
  while (s=strstr(s," 00:")) {
    s+=4;
    strncpy(s,"00.000000",9);
    s+=9;
  }
}

gint
test_fini (const gchar *arg0)
{
  const gchar *source_dir = g_getenv ("TESTS_SOURCE_DIR");
  const gchar *build_dir = g_getenv ("TESTS_BUILD_DIR");
  gchar *s;
  gchar *testlogname,*reflogname;
  FILE *tl,*rl;
  gchar ts[251], rs[251], *tsp, *rsp;
  gboolean fail = FALSE;
  guint l = 0;

  refdbg_save_log ();

  s = strrchr (arg0, G_DIR_SEPARATOR);
  testlogname = g_strdup_printf ("%s.log",arg0);
  reflogname = g_strdup_printf ("%s%s.ref",source_dir, s);

  /* compare the test log line by line to the reference log */
  /* TODO: ignore hex addresses and timestamps */
  tl = fopen (testlogname, "r");
  rl = fopen (reflogname, "r");
  if (tl && rl) {
    while (!feof (tl) && !feof (rl)) {
      tsp = fgets (ts, 250, tl);
      rsp = fgets (rs, 250, rl);
      if (tsp && rsp) {
        canon_log_line(g_strchomp(ts));
        canon_log_line(g_strchomp(rs));
        if (strcmp (ts, rs)) {
          fprintf(stderr, "logs differ at line %u\n'%s'\n'%s'\n", l, ts, rs);
          fail = TRUE;
          break;
        }
        l++;
      }
    }
    if(!fail && (!feof (tl) || !feof (rl))) {
      fprintf(stderr, "logs differ in length at line %u\n", l);
      fail = TRUE;
    }
  }
  if (tl) {
    fclose (tl);
  } else {
    fprintf(stderr, "test log missing: %s\n", testlogname);
    fail = TRUE;
  }
  if (rl) {
    fclose (rl);
  } else {
    fprintf(stderr, "reference log missing: %s\n", reflogname);
    fail = TRUE;
  }

  g_free (testlogname);
  g_free (reflogname);

  return fail;
}

/* reusable test types */

GType
test1_get_type (void)
{
  static GType type = 0;
  if (type == 0) {
    static GTypeInfo info = {
      sizeof (GObjectClass),
      NULL,                     /* base_init */
      NULL,                     /* base_finalize */
      NULL,                     /* class_init */
      NULL,                     /* class_finalize */
      NULL,                     /* class_data */
      sizeof (GObject),
      0,
      NULL                      /* instance_init */
    };
    type = g_type_register_static (G_TYPE_OBJECT, "refdbg-test1", &info, 0);
  }
  return type;
}


static void
test2_get_property (GObject *object,guint property_id,GValue *value,GParamSpec *pspec) {
  switch (property_id) {
    case 1:
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object,property_id,pspec);
      break;
  }
}

static void 
test2_set_property (GObject *object,guint property_id,const GValue *value,GParamSpec *pspec) {
  switch (property_id) {
    case 1:
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object,property_id,pspec);
      break;
  }
}

static void 
test2_class_init (GObjectClass *klass) {
  klass->set_property = test2_set_property;
  klass->get_property = test2_get_property;

  g_object_class_install_property (klass, 1,
    g_param_spec_boolean ("switch", "switch prop", "switch parameter",
       FALSE, G_PARAM_READWRITE));
}

GType
test2_get_type (void)
{
  static GType type = 0;
  if (type == 0) {
    static GTypeInfo info = {
      sizeof (GObjectClass),
      NULL,                     /* base_init */
      NULL,                     /* base_finalize */
      (GClassInitFunc)test2_class_init,                     /* class_init */
      NULL,                     /* class_finalize */
      NULL,                     /* class_data */
      sizeof (GObject),
      0,
      NULL                      /* instance_init */
    };
    type = g_type_register_static (G_TYPE_OBJECT, "refdbg-test2", &info, 0);
  }
  return type;
}

