/*
 * user.c - User functions for interacting with refdbg
 *
 * refdbg - GObject refcount debugger
 * Copyright (C) 2004-2005 Josh Green <jgreen@users.sourceforge.net>
 *               2010 Stefan Kost <ensonic@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 */
#include <stdio.h>
#include <stdlib.h>             /* qsort */
#include <string.h>
#include <glib-object.h>
#include <errno.h>

#include "refdbg.h"
#include "refdbg_priv.h"
#include "config.h"

/* default maximum results to display (so user isn't overwhelmed) */
guint refdbg_dispmax = REFDBG_DEFAULT_DISPMAX;

/* file name of the log file for refdbg_save_log() */
gchar *refdbg_logname;

static void objcount_GHFunc (gpointer key, gpointer value, gpointer user_data);
static int objbag_type_name_compar (const void *a, const void *b);
static void refdbg_display_objects_GHFunc (gpointer object, gpointer value,
    gpointer user_data);

typedef struct
{
  guint dead_count;
  guint total_count;
  guint refcount_sum;
  guint max_refcount;
} StatBag;

/**
 * refdbg_stats:
 *
 * Print some statistics on refdbg to stderr.
 */
void
refdbg_stats (void)
{
  refdbg_real_stats (FALSE);
}

/* the real stat function, so refdbg_fini can tell it not to display if
 * there was no activity. */
void
refdbg_real_stats (gboolean noshow_empty)
{
  guint bytes;
  StatBag bag = { 0 };
  guint alive_count;

  g_return_if_fail (refdbg_active);

  REFDBG_LOCK;

  bytes = event_array->len      /* size of event array in bytes */
      * (sizeof (RefEvent) + backtrace_count * sizeof (gpointer));

  g_hash_table_foreach (obj_hash, objcount_GHFunc, &bag);
  alive_count = bag.total_count - bag.dead_count;

  if (noshow_empty && bag.total_count == 0 && event_array->len == 0) {
    REFDBG_UNLOCK;
    return;
  }

  fprintf (stderr, "RefDbg Statistics:\n");
  fprintf (stderr, "Event Log Size    : %u (%u bytes)\n",
      event_array->len, bytes);
  fprintf (stderr, "Total Objects     : %u\n", bag.total_count);
  fprintf (stderr,
      "Alive Objects     : %u (%u max refcount, %0.1f avg refcount)\n",
      alive_count, bag.max_refcount,
      alive_count > 0 ? bag.refcount_sum / (float) alive_count : 0.0);
  fprintf (stderr, "Destroyed Objects : %u\n", bag.dead_count);

  /* NOTE: Cache is filled on demand, so if no demand, then it will be empty */
  fprintf (stderr, "Backtrace Cache   : %u addresses\n",
      refdbg_sym_hash_size ());

  REFDBG_UNLOCK;
}

/* count total seen objects as well as destroyed */
static void
objcount_GHFunc (gpointer key, gpointer value, gpointer user_data)
{
  StatBag *bag = (StatBag *) user_data;
  guint refcount = ((ObjInfo *) value)->refcount;

  if (refcount > 0) {           /* object still alive? */
    bag->refcount_sum += refcount;
    if (refcount > bag->max_refcount)
      bag->max_refcount = refcount;
  } else
    bag->dead_count++;          /* object was finalized? */

  bag->total_count++;
}

typedef struct
{
  GObject *object;
  ObjInfo *info;
} ObjBag;

typedef struct
{
  DetailRule *rule;
  FILE *file;
  int count;
  ObjBag *objs;                 /* array of all matching objects for sorting */
} DispObjBag;

/**
 * refdbg_display_objects:
 * @rule: Criteria for displaying object info or NULL for all active objects
 * @file: File handle to write output to
 *
 * Print reference counts and GType for objects by criteria.
 *
 * Returns: Count of matching objects displayed.
 */
int
refdbg_display_objects (DetailRule * rule, FILE * file)
{
  DispObjBag bag;
  GObject *obj;
  ObjInfo *info;
  const char *typename;
  int i;

  if (!file)
    file = stderr;

  bag.rule = rule;
  bag.file = file;
  bag.count = 0;

  /* we keep things locked, in case object info is cleared, maybe too paranoid? */
  REFDBG_LOCK;

  /* allocate for maximum possible objects */
  bag.objs = g_malloc (g_hash_table_size (obj_hash) * sizeof (ObjBag));

  /* iterate over the hash stuffing only those which follow the rule */
  g_hash_table_foreach (obj_hash, refdbg_display_objects_GHFunc, &bag);

  /* sort the array by type name (as suggested by ensonic) */
  if (bag.count > 0)
    qsort (bag.objs, bag.count, sizeof (ObjBag), objbag_type_name_compar);

  for (i = 0; i < bag.count; i++) {     /* loop over matching objects */
    obj = bag.objs[i].object;
    info = bag.objs[i].info;

    typename = g_type_name (info->type);

    if (info->refcount != 0)
      fprintf (file, "<%s> [%p] |%d|\n", typename, obj, info->refcount);
    else
      fprintf (file, "<%s> [%p] {Destroyed}\n", typename, obj);
  }

  REFDBG_UNLOCK;

  g_free (bag.objs);

  return (bag.count);
}

static int
objbag_type_name_compar (const void *a, const void *b)
{
  ObjInfo *ainfo = ((ObjBag *) a)->info, *binfo = ((ObjBag *) b)->info;
  int cmp;

  cmp = strcmp (g_type_name (ainfo->type), g_type_name (binfo->type));

  if (cmp == 0) {
    if (ainfo->refcount > binfo->refcount)
      return (-1);
    else if (ainfo->refcount < binfo->refcount)
      return (1);
    else
      return (0);
  } else
    return (cmp);
}

static void
refdbg_display_objects_GHFunc (gpointer object, gpointer value,
    gpointer user_data)
{
  DispObjBag *bag = (DispObjBag *) user_data;
  DetailRule *rule = bag->rule;
  guint refcount = ((ObjInfo *) value)->refcount;
  GType obj_type = ((ObjInfo *) value)->type;
  ObjBag *objbag;
  int obj_flags;
  int i;

  obj_flags = refcount != 0 ? OBJECT_RULE_FLAG_ACTIVE
      : OBJECT_RULE_FLAG_DESTROYED;

  if (rule) {
    if (!(obj_flags & rule->display_mask))
      return;

    if (rule->inc_objects) {    /* check include objects */
      for (i = 0; rule->inc_objects[i]; i++)
        if (object == rule->inc_objects[i])
          break;

      if (!rule->inc_objects[i])
        return;                 /* not found? - skip */
    }

    if (rule->exc_objects) {    /* check exclude objects */
      for (i = 0; rule->exc_objects[i]; i++)
        if (object == rule->exc_objects[i])
          break;

      if (rule->exc_objects[i])
        return;                 /* found? - skip */
    }

    if (rule->inc_types) {      /* check include types */
      for (i = 0; rule->inc_types[i]; i++)
        if (g_type_is_a (obj_type, rule->inc_types[i]))
          break;

      if (!rule->inc_types[i])
        return;                 /* not found? - skip */
    }

    if (rule->exc_types) {      /* check exclude types */
      for (i = 0; rule->exc_types[i]; i++)
        if (g_type_is_a (obj_type, rule->exc_types[i]))
          break;

      if (rule->exc_types[i])
        return;                 /* found? - skip */
    }
  } else /* no rule, only show active objects */ if (!(obj_flags &
        OBJECT_RULE_FLAG_ACTIVE))
    return;

  /* store object info for later sorting */
  objbag = &bag->objs[bag->count];
  objbag->object = object;
  objbag->info = (ObjInfo *) value;

  bag->count++;                 /* inc count of matching objects */
}

/**
 * refdbg_save_log:
 * 
 * Save object reference events to a file. The default file name is'refdbg.log'
 * in the current directory. It can be overridden using the 'logname' command.
 */
void
refdbg_save_log (void)
{
  refdbg_real_save_log (TRUE);
}

/* the real save log routine which allows log to not be saved if empty
   (prevents bogus start/init sequences from overwriting the good log file) */
void
refdbg_real_save_log (gboolean ifempty)
{
  FILE *file;
  int count;

  REFDBG_LOCK;

  if (!ifempty && event_array->len == 0 && g_hash_table_size (obj_hash) == 0) {
    REFDBG_UNLOCK;
    return;
  }

  REFDBG_UNLOCK;

  file = fopen (refdbg_logname, "w");
  if (!file) {
    g_critical ("Failed to open output log '%s': %s",
        refdbg_logname, g_strerror (errno));
    return;
  }

  refdbg_display_events (NULL, file);

  if (log_objects) {            /* log list of active objects? */
    fprintf (file, "[Active Objects]\n");
    count = refdbg_display_objects (NULL, file);
    fprintf (file, "[Total active objects: %d]\n", count);
  }

  fprintf (stderr, "Saved %ld bytes to event log '%s'\n", ftell (file), refdbg_logname);

  fclose (file);
}

const gchar *
refdbg_get_event_type_name (RefEvent * event)
{
  const gchar *typestr = NULL;

  if (event->type == EVENT_TYPE_PRE_NEW)
    typestr = "PRE_NEW";
  else if (event->type == EVENT_TYPE_NEW)
    typestr = "NEW";
  else if (event->type == EVENT_TYPE_REF)
    typestr = "REF";
  else if (event->type == EVENT_TYPE_UNREF)
    typestr = "UNREF";
  else if (event->type == EVENT_TYPE_PRE_FINALIZE)
    typestr = "PRE_FINALIZE";
  else if (event->type == EVENT_TYPE_FINALIZE)
    typestr = "FINALIZE";

  return typestr;
}

const gchar *
refdbg_format_time_stamp (guint32 timestamp)
{
  static gchar ts_str[2 + 1 + 2 + 1 + 6 + 1];
  guint minutes, secs, usecs;

  minutes = timestamp / 60000000;
  secs = (timestamp / 1000000) % 60;
  usecs = timestamp % 1000000;

  sprintf (ts_str, "%02d:%02d.%06d", minutes, secs, usecs);
  return ts_str;
}

void
refdbg_display_backtrace (void **trace, int bt_num, FILE * file)
{
  int i;
  AddrInfo *addrinfo;
  const char *s;
  const int pw = 2*sizeof(gpointer);

  /* back trace, loop over addresses (NULL terminated) */
  for (i = 0; i < bt_num && trace[i]; i++) {
    addrinfo = refdbg_sym_snarf (trace[i]);

    /* we use deprecated g_basename on purpose, since we don't want to alloc */
    s = addrinfo->obj_fname;
    if (!refdbg_enable_bt_paths && s)
      s = g_basename (s);

    /* index, address */
    fprintf (file, " #%-2d %0*p", i, pw, trace[i]);

    /* object file name and address offset in object file */
#if 0
    if (s)
      fprintf (file, " %s+%0*p", s, pw, (gpointer)(trace[i] - addrinfo->obj_addr));
    else
      fprintf (file, " <UnkownObj>+%0*p", pw, (gpointer)0);
#endif

    /* function name and address offset in function */
    /* TODO: would be nice to know the prototype and print the parameters */
    if (addrinfo->func)
      fprintf (file, " in %s ()+%0*p", addrinfo->func, pw,
          (gpointer)(addrinfo->func_addr ? trace[i] - addrinfo->func_addr : 0));
    else
      fprintf (file, " in ???");

    /* we use deprecated g_basename on purpose, since we don't want to alloc */
    s = addrinfo->src_fname;
    if (!refdbg_enable_bt_paths && s)
      s = g_basename (s);

    /* Source file and line number */
    if (s)
      fprintf (file, " at %s:%d\n", s, addrinfo->src_line);
    else
      fputc ('\n', file);
  }

}

/**
 * refdbg_display_events:
 * @rule: Rule criteria to match events to or %NULL to match all
 * @file: File to write events to or %NULL for stderr
 *
 * Display event log entries matching @rule criteria.
 */
void
refdbg_display_events (DetailRule * rule, FILE * file)
{
  RefEvent *event;
  guint event_flags;
  void **trace;
  gpointer data;
  const char *typestr, *timestamp;
  guint result_ofs = 0, result_count = 0, limit_ofs;
  int i, j, btnum;

  g_return_if_fail (refdbg_active);

  if (!file)
    file = stderr;

  REFDBG_LOCK;

  if (rule) {                   /* negative offset is offset from end */
    if (rule->limit_ofs < 0)
      limit_ofs = event_array->len + rule->limit_ofs;
    else
      limit_ofs = rule->limit_ofs;
  } else
    limit_ofs = 0;

  if (rule && rule->btnum != -1)
    btnum = rule->btnum;
  else
    btnum = backtrace_count;

  data = event_array->data;
  for (i = 0; i < event_array->len; i++) {
    event = (RefEvent *) data;
    trace = (void **) REF_EVENT_BACKTRACE (event);
    data += sizeof (RefEvent) + backtrace_count * sizeof (gpointer);

    if (rule) {
      event_flags = 0;
      event_flags |= (1 << event->type);
      if (event->error)
        event_flags |= (1 << (event->error + EVENT_RULE_FLAG_ERROR_SHIFT));

      /* skip if type/error not in display mask */
      if ((event_flags & rule->display_mask) == 0)
        continue;

      if (rule->inc_types) {    /* check include types */
        for (j = 0; rule->inc_types[j]; j++)
          if (g_type_is_a (event->obj_type, rule->inc_types[j]))
            break;

        if (!rule->inc_types[j])
          continue;             /* not found? - skip */
      }

      if (rule->exc_types) {    /* check exclude types */
        for (j = 0; rule->exc_types[j]; j++)
          if (g_type_is_a (event->obj_type, rule->exc_types[j]))
            break;

        if (rule->exc_types[j])
          continue;             /* found? - skip */
      }

      if (rule->inc_objects) {  /* check include objects */
        for (j = 0; rule->inc_objects[j]; j++)
          if (event->object == rule->inc_objects[j])
            break;

        if (!rule->inc_objects[j])
          continue;             /* not found? - skip */
      }

      if (rule->exc_objects) {  /* check exclude objects */
        for (j = 0; rule->exc_objects[j]; j++)
          if (event->object == rule->exc_objects[j])
            break;

        if (rule->exc_objects[j])
          continue;             /* found? - skip */
      }

      /* check requested start and end time */
      if (event->timestamp < rule->start_time)
        continue;
      if (rule->end_time > 0 && event->timestamp > rule->end_time)
        break;

      /* check if we are above result offset */
      if (result_ofs++ < limit_ofs)
        continue;
    }

    typestr = refdbg_get_event_type_name (event);
    timestamp = refdbg_format_time_stamp (event->timestamp);

    /* if error or PRE_NEW event and object wasn't fixed up (also error).. */
    if (event->error || (event->type == EVENT_TYPE_PRE_NEW && !event->object)) {
      if (event->error == EVENT_ERR_UNKNOWN_OBJECT)
        fprintf (file, "!%-12s {UNKNOWN} [%p] %s\n", typestr,
            event->object, timestamp);
      else if (event->error == EVENT_ERR_DESTROYED_OBJECT)
        fprintf (file, "!%-12s {DESTROYED} [%p] %s\n", typestr,
            event->object, timestamp);
      else if (event->error == EVENT_ERR_NOT_OBJECT)
        fprintf (file, "!%-12s {INVALID} [%p] %s\n", typestr,
            event->object, timestamp);
      else if (event->error == EVENT_ERR_INIT_REFCOUNT)
        fprintf (file, "!%-12s {INITREF} <%s> [%p] |%d != 1| %s\n", typestr,
            g_type_name (event->obj_type), event->object,
            event->refcount, timestamp);
      else if (event->error == EVENT_ERR_BAD_REFCOUNT)
        fprintf (file, "!%-12s {BADREF} <%s> [%p] |!%d| %s\n",
            typestr, g_type_name (event->obj_type), event->object,
            event->refcount, timestamp);
      else if (event->type == EVENT_TYPE_PRE_NEW)
        fprintf (file, "!%-12s {FAILED} [NULL] %s\n", typestr, timestamp);
    } else                      /* not error */
      fprintf (file, "%-12s <%s> [%p] |%d| %s\n", typestr,
          g_type_name (event->obj_type), event->object,
          event->refcount, timestamp);

    /* back trace */
    refdbg_display_backtrace (trace, btnum, file);

    if (rule && ++result_count >= rule->limit_max) {
      fprintf (file, "** Max result count of %d reached (set dispmax or"
          " use limit argument)\n", rule->limit_max);
      break;
    }
  }

  REFDBG_UNLOCK;
}
