/*
 * refdbg.c - The guts of the GObject reference debugger
 *
 * refdbg - GObject refcount debugger
 * Copyright (C) 2004-2005 Josh Green <jgreen@users.sourceforge.net>
 *               2010 Stefan Kost <ensonic@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib-object.h>

#include <dlfcn.h>
#include <execinfo.h>
#include <sys/time.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include "refdbg.h"
#include "refdbg_priv.h"
#include "config.h"


/**************************************************************************
 * !! NOTICE !! The following variables and functions are private and
 * should not be modified by the user directly. Use the functions
 * defined in refdbg.h to control refdbg (from GDB).
 **************************************************************************/

static void obj_hash_val_destroy (gpointer data);
static gboolean removeall_GHRFunc (gpointer key, gpointer value,
    gpointer user_data);
static gboolean timer_last_events_GHRFunc (gpointer key, gpointer value,
    gpointer user_data);
static gboolean refdbg_timer_callback (gpointer data);

gboolean refdbg_active = FALSE; /* set to TRUE if active */

/* number of backtrace stack frames to store for each ref/unref event,
   can be set at init time only */
int backtrace_count = REFDBG_DEFAULT_BACKTRACE_COUNT;

/* set to TRUE to enable paths in back traces (brief by default) */
gboolean refdbg_enable_bt_paths = FALSE;

/* set to FALSE by 'notimer' command to disable glib timer callback */
gboolean refdbg_enable_timer = TRUE;

/* lock for multi-thread sensitive data (all the following variables) */
GStaticRecMutex refdbg_mutex = G_STATIC_REC_MUTEX_INIT;


/* array of event rules (first rule is default rule) */
EventRule event_rules[REFDBG_MAX_EVENT_RULES];
int event_rule_count = 0;       /* # of active event rules (not including default) */

/* array of RefEvent structures + backtrace
   NOTE: Cannot be indexed directly using RefEvent, use REF_ARRAY_INDEX! */
GArray *event_array = NULL;

/* object hash (object => ObjInfo) */
GHashTable *obj_hash = NULL;

/* hash of event_array indexes (object => index + 1) of last event for
   each object matching timer rule */
static GHashTable *timer_last_events = NULL;

/* oldest index in event_array that matches timer criteria (used as an
   optimization to know when timer_last_events should be processed) set to
   -1 when there are no entries in timer_last_events */
static int timer_oldest_index = -1;

/* hash of event_array indexes (object => index + 1) of "stable"
   object refcounts (the last time an object went for timer_expire
   without changing) */
static GHashTable *timer_stable_refcounts = NULL;

/* number of objects currently in construction (to suppress ref/unref of
   unknown object during instance init routines) */
guint construct_count = 0;

/* lowest index in event_array that needs fixup or ref/unref unknown object
   checking (only valid if construct_count > 0) */
guint construct_index;

struct timeval start_time;      /* start time (when refdbg_init called) */

/* refcount timer expire time in microseconds (0 = disable) */
guint32 timer_expire = 0;

/* set to TRUE to save event log when RefDbg finishes */
gboolean save_event_log = TRUE;

/* set to TRUE to save list of active objects to end of log file on exit */
gboolean log_objects = TRUE;

/* set to TRUE to print object stats when RefDbg finishes */
gboolean display_object_stats = TRUE;

/* ID of timer glib source */
static guint refdbg_timer_id = 0;

/* function override test variable */
static gboolean override_test = FALSE;

/* GObject derived dummy type so we can test g_object_new override */
static GType override_test_type = 0;


/* original GObject function pointers */
static gpointer (*orig_object_newv) (GType object_type, guint n_parameters,
    GParameter * parameters);
static gpointer (*orig_object_ref) (gpointer);
static void (*orig_object_unref) (gpointer);
static void (*orig_type_free_instance) (GTypeInstance * instance);


/* init routine */
void __attribute__ ((constructor))
    refdbg_init (void)
{
  void *gobject_handle;
  const char *lname = "libgobject-2.0.so";
  const char *options;
  char *errstr;

  if (refdbg_active)
    return;

  g_type_init ();

  options = g_getenv ("REFDBG_OPTIONS");

  fprintf (stderr, "RefDbg " VERSION " init\n");
  if (options)
    fprintf (stderr, "REFDBG_OPTIONS: %s\n", options);

  REFDBG_LOCK;

  /* set default event rule (display all errors, no breaks, log all) */
  event_rules[0].inc_type = 0;
  event_rules[0].exc_type = 0;
  event_rules[0].inc_type_name = NULL;
  event_rules[0].exc_type_name = NULL;
  event_rules[0].inc_object = NULL;
  event_rules[0].exc_object = NULL;
  event_rules[0].display_mask = EVENT_RULE_MASK_ERROR;
  event_rules[0].break_mask = EVENT_RULE_MASK_NONE;
  event_rules[0].log_mask = EVENT_RULE_MASK_ALL;

  refdbg_logname = g_strdup (REFDBG_DEFAULT_LOGNAME);

  /* parse options and execute REFDBG_OPTIONS (if any) */
  if (options)
    refdbg_cmd (options);

  /* create ref/unref event array */
  event_array = g_array_new (FALSE, FALSE, sizeof (RefEvent)
      + backtrace_count * sizeof (gpointer));

  /* create object hash */
  obj_hash = g_hash_table_new_full (NULL, NULL, NULL, obj_hash_val_destroy);

  /* create timer last event hash */
  timer_last_events = g_hash_table_new (NULL, NULL);

  /* create timer stable refcount hash */
  timer_stable_refcounts = g_hash_table_new (NULL, NULL);

  /* initialize symbol snarfing system (FIXME - Can it fail, what then?) */
  refdbg_sym_snarf_init ();

  gettimeofday (&start_time, NULL);     /* get start time */

  refdbg_active = TRUE;

  REFDBG_UNLOCK;

  /* load gobject library (abort if not found) */
  gobject_handle = dlopen (lname, RTLD_LAZY);
  if (!gobject_handle) {
    errstr = dlerror ();
    g_critical ("Failed to load GObject library '%s': %s, aborting..", lname,
        errstr ? errstr : "<No error details>");
    exit (1);
  }

  /* get pointer to original g_object_newv (abort if not found) */
  orig_object_newv = dlsym (gobject_handle, "g_object_newv");
  if (!orig_object_newv) {
    errstr = dlerror ();
    g_critical ("Failed to find g_object_newv() in GObject library '%s': %s,"
        " aborting..", lname, errstr ? errstr : "<Symbol is NULL>");
    exit (1);
  }

  /* get pointer to original g_object_ref (abort if not found) */
  orig_object_ref = dlsym (gobject_handle, "g_object_ref");
  if (!orig_object_ref) {
    errstr = dlerror ();
    g_critical ("Failed to find g_object_ref() in GObject library '%s': %s,"
        " aborting..", lname, errstr ? errstr : "<Symbol is NULL>");
    exit (1);
  }

  /* get pointer to original g_object_unref (abort if not found) */
  orig_object_unref = dlsym (gobject_handle, "g_object_unref");
  if (!orig_object_unref) {
    errstr = dlerror ();
    g_critical ("Failed to find g_object_unref() in GObject library '%s': %s,"
        " aborting..", lname, errstr ? errstr : "<Symbol is NULL>");
    exit (1);
  }

  /* get pointer to original g_type_free_instance (abort if not found) */
  orig_type_free_instance = dlsym (gobject_handle, "g_type_free_instance");
  if (!orig_type_free_instance) {
    errstr = dlerror ();
    g_critical ("Failed to find g_type_free_instance() in GObject library"
        " '%s': %s, aborting..", lname, errstr ? errstr : "<Symbol is NULL>");
    exit (1);
  }

  if (!override_test_type) {
    static GTypeInfo override_test_type_info = {
      sizeof (GObjectClass),
      NULL,                     /* base_init */
      NULL,                     /* base_finalize */
      NULL,                     /* class_init */
      NULL,                     /* class_finalize */
      NULL,                     /* class_data */
      sizeof (GObject),
      0,
      NULL                      /* instance_init */
    };

    override_test_type
        = g_type_register_static (G_TYPE_OBJECT, "refdbg-override-test",
        &override_test_type_info, 0);
    g_assert (override_test_type);
  }

  /* Run test to see if LD_PRELOAD has correctly overridden g_object_new.. */
  override_test = TRUE;

  /* NOTE: Object is not created if g_object_newv is successfully intercepted. */
  g_object_newv (override_test_type, 0, NULL);

  if (override_test) {
    g_critical ("LD_PRELOAD function override not working. Need to build"
        " glib with --disable-visibility? (See README), aborting..");
    exit (1);
  }

  /* Install timer into main loop */
  if (refdbg_enable_timer)
    refdbg_timer_id = g_timeout_add (REFDBG_TIMER_INTERVAL,
        refdbg_timer_callback, NULL);
}

/* finish routine */
void __attribute__ ((destructor))
    refdbg_fini (void)
{
  int i;
  EventRule *rule;

  fprintf (stderr, "RefDbg shutdown\n");

  if (!refdbg_active)
    return;

  /* show stats after log save so addr cache gets correct count (on demand)
   * Don't save log if empty, to prevent bogus start/init sequences from
   * overwriting the good schtuff. */
  if (save_event_log)
    refdbg_real_save_log (FALSE);
  if (display_object_stats)
    refdbg_real_stats (TRUE);

  if (event_array->len) {
    /* loop over event rules (there is always a default rule, index 0) */
    for (i = event_rule_count; i >= 0; i--) {
      rule = &event_rules[i];

      if (rule->inc_type_name) {  /* unresolved include type? */
        g_warning ("GType '%s' has never been resolved", rule->inc_type_name);
        g_free (rule->inc_type_name);
        rule->inc_type_name = NULL;
      }

      if (rule->exc_type_name) {  /* unresolved exclude type? */
        g_warning ("GType '%s' has never been resolved", rule->exc_type_name);
        g_free (rule->exc_type_name);
        rule->exc_type_name = NULL;
      }
    }
  }

  REFDBG_LOCK;

  refdbg_active = FALSE;

  g_free (refdbg_logname);
  refdbg_logname = NULL;

  g_array_free (event_array, TRUE);
  g_hash_table_destroy (obj_hash);
  g_hash_table_destroy (timer_last_events);
  g_hash_table_destroy (timer_stable_refcounts);

  event_array = NULL;
  obj_hash = NULL;
  timer_last_events = NULL;
  timer_stable_refcounts = NULL;

  REFDBG_UNLOCK;
}

/* destroy function for obj_hash data */
static void
obj_hash_val_destroy (gpointer data)
{
  g_slice_free (ObjInfo, data);
}


/**
 * refdbg_clear:
 *
 * Clear event log and refcount timer data.
 */
void
refdbg_clear (void)
{
  REFDBG_LOCK;

  /* make sure no objects are in construction (new array index passed
     outside of lock) */
  if (construct_count > 0) {
    g_critical ("Cannot clear event log, GObjects in construction,"
        " try again later");
    REFDBG_UNLOCK;
    return;
  }

  /* empty the event log */
  g_array_set_size (event_array, 0);

  /* reset the timer information (references event log) */
  g_hash_table_foreach_remove (timer_last_events, removeall_GHRFunc, NULL);
  g_hash_table_foreach_remove (timer_stable_refcounts, removeall_GHRFunc, NULL);
  timer_oldest_index = -1;

  REFDBG_UNLOCK;
}

/* function to remove all entries in a hash */
static gboolean
removeall_GHRFunc (gpointer key, gpointer value, gpointer user_data)
{
  return (TRUE);
}

/* get elapsed time */
guint32
refdbg_get_timestamp (void)
{
  struct timeval curtime;
  guint32 timestamp;

  gettimeofday (&curtime, NULL);

  /* calculate time stamp offset in microseconds */
  timestamp = (curtime.tv_sec - start_time.tv_sec) * 1000000;
  if (curtime.tv_usec > start_time.tv_usec)
    timestamp += curtime.tv_usec - start_time.tv_usec;
  else
    timestamp -= start_time.tv_usec - curtime.tv_usec;  
  
  return (timestamp);
}

/* return flags from refdbg_event_dispatch */
#define DISPATCH_RET_BREAK  (1 << 0)
#define DISPATCH_RET_LOG    (1 << 1)
#define DISPATCH_RET_TIMER  (1 << 2)

/* event dispatch function - determines what to do with an event based on
   current event rules */
static int
refdbg_event_dispatch (RefEvent * event, guint expected)
{                               /* expected: for ERR_BAD_REFCOUNT */
  void **trace = REF_EVENT_BACKTRACE (event);
  gboolean display = FALSE;
  const char *typestr, *timestamp;
  EventRule *rule;
  int event_flags = 0;
  int i, retval = 0;

  /* calculate mask flags for this event */
  event_flags |= (1 << event->type);

  if (event->error)             /* EVENT_ERR_NONE isn't in rule masks */
    event_flags |= (1 << (event->error + EVENT_RULE_FLAG_ERROR_SHIFT));

  /* loop over event rules (there is always a default rule, index 0) */
  for (i = event_rule_count; i >= 0; i--) {
    rule = &event_rules[i];

    if (rule->inc_type_name) {  /* unresolved include type? */
      rule->inc_type = g_type_from_name (rule->inc_type_name);
      if (rule->inc_type) {     /* if resolved, free type name string */
        g_message ("GType '%s' has been resolved", rule->inc_type_name);
        g_free (rule->inc_type_name);
        rule->inc_type_name = NULL;
      }
    }

    if (rule->exc_type_name) {  /* unresolved exclude type? */
      rule->exc_type = g_type_from_name (rule->exc_type_name);
      if (rule->exc_type) {     /* if resolved, free type name string */
        g_message ("GType '%s' has been resolved", rule->exc_type_name);
        g_free (rule->exc_type_name);
        rule->exc_type_name = NULL;
      }
    }

    /* rule matches include/exclude type and include/exclude
       object criteria? (Don't match unresolved GTypes) */
    if ((!rule->inc_type || g_type_is_a (event->obj_type, rule->inc_type))
        && (!rule->exc_type || !g_type_is_a (event->obj_type, rule->exc_type))
        && (!rule->inc_object || rule->inc_object == event->object)
        && (!rule->exc_object || rule->exc_object != event->object)
        && !rule->inc_type_name && !rule->exc_type_name) {
      display = ((event_flags & rule->display_mask) != 0);
      if (event_flags & rule->break_mask)
        retval |= DISPATCH_RET_BREAK;
      if (event_flags & rule->log_mask)
        retval |= DISPATCH_RET_LOG;

      /* enable refcount timer for this event if TIMER flag is
         set, timer_expire is not 0, event type is NEW, REF or
         UNREF, no invalid object related errors are set and event
         will be logged */
      if ((rule->display_mask & EVENT_RULE_FLAG_TIMER) && timer_expire
          && (event_flags & (EVENT_RULE_FLAG_NEW | EVENT_RULE_FLAG_REF
                  | EVENT_RULE_FLAG_UNREF))
          && !(event_flags & (EVENT_RULE_FLAG_ERR_UNKNOWN_OBJECT
                  | EVENT_RULE_FLAG_ERR_DESTROYED_OBJECT
                  | EVENT_RULE_FLAG_ERR_NOT_OBJECT))
          && (retval & DISPATCH_RET_LOG))
        retval |= DISPATCH_RET_TIMER;

      /* if UNKNOWN/DESTROYED object error during object construction.. */
      if ((event->error == EVENT_ERR_UNKNOWN_OBJECT
              || event->error == EVENT_ERR_DESTROYED_OBJECT)
          && construct_count > 0) {     /* if PARANOID display flag is not set then don't display */
        if (!(rule->display_mask & EVENT_RULE_FLAG_PARANOID))
          display = FALSE;

        /* if PARANOID break flag is not set then don't break */
        if (!(rule->break_mask & EVENT_RULE_FLAG_PARANOID))
          retval &= ~DISPATCH_RET_BREAK;
      }

      break;
    }
  }

  if (display) {
    typestr = refdbg_get_event_type_name (event);
    timestamp = refdbg_format_time_stamp (event->timestamp);

    if (event->error) {
      if (event->error == EVENT_ERR_UNKNOWN_OBJECT)
        fprintf (stderr, "!%-12s {UNKNOWN} [%p] %s\n", typestr,
            event->object, timestamp);
      else if (event->error == EVENT_ERR_DESTROYED_OBJECT)
        fprintf (stderr, "!%-12s {DESTROYED} [%p] %s\n", typestr,
            event->object, timestamp);
      else if (event->error == EVENT_ERR_NOT_OBJECT)
        fprintf (stderr, "!%-12s {INVALID} [%p] %s\n", typestr,
            event->object, timestamp);
      else if (event->error == EVENT_ERR_INIT_REFCOUNT)
        fprintf (stderr, "!%-12s {INITREF} <%s> [%p] |%d != 1|"
            " %s\n", typestr,
            g_type_name (event->obj_type), event->object,
            event->refcount, timestamp);
      else if (event->error == EVENT_ERR_BAD_REFCOUNT)
        fprintf (stderr, "!%-12s {BADREF} <%s> [%p] |%d != %d| %s\n", typestr,
            g_type_name (event->obj_type), event->object,
            event->refcount, expected, timestamp);
    } else {                    /* not error */

      if (event->type != EVENT_TYPE_PRE_NEW)
        fprintf (stderr, "%-12s <%s> [%p] |%d| %s\n", typestr,
            g_type_name (event->obj_type), event->object,
            event->refcount, timestamp);
      else
        fprintf (stderr, "%-12s <%s> |%d| %s\n", typestr,
            g_type_name (event->obj_type), event->refcount, timestamp);
    }

    /* back trace */
    refdbg_display_backtrace (trace, backtrace_count, stderr);
  }

  /* returns flags for if event should be logged or a break point executed */
  return (retval);
}

/* function event helper, returns the same value as refdbg_event_dispatch,
 * event: Event to help out with
 * refcount_inc: Value to add to current refcount (0 = new or finalize event)
 * out_newindex: Only set on PRE_NEW event.  Index in event log of new event
 *   is stored for later fixup.
 */
static int
refdbg_helper (RefEvent * event, int refcount_inc, guint * out_newindex)
{                               /* use one buffer for event and trace so it doesn't have to be copied */
  guint32 timestamp = refdbg_get_timestamp ();
  ObjInfo *objinfo = NULL;
  int retval;

  REFDBG_LOCK;                  /* lock multi-thread sensitive data */

  /* calculate time stamp offset in microseconds */
  event->timestamp = timestamp;

  /* not new/finalize event and object is not NULL */
  if (refcount_inc != 0 && event->object) {     /* lookup object info in obj_hash */
    objinfo = g_hash_table_lookup (obj_hash, event->object);

    if (!objinfo)               /* unknown object pointer? */
      event->error = EVENT_ERR_UNKNOWN_OBJECT;
    else if (objinfo->refcount == 0)    /* zombie object back from the dead VIII */
      event->error = EVENT_ERR_DESTROYED_OBJECT;
    else if (G_IS_OBJECT (event->object)) {     /* valid obj? */
      /* get object type and refcount */
      event->obj_type = G_OBJECT_TYPE (event->object);
      event->refcount = ((GObject *) (event->object))->ref_count;

      /* last refcount set, not new object and refcount doesn't match? */
      if (event->refcount != objinfo->refcount)
        event->error = EVENT_ERR_BAD_REFCOUNT;

      /* not last reference and type unref? */
      if (event->refcount != 1 || event->type != EVENT_TYPE_UNREF) {    /* calculate new refcount */
        event->refcount += refcount_inc;
        objinfo->refcount = event->refcount;    /* assign new refcount */
      } /* don't calculate refcount if pre finalize event */
      else
        event->type = EVENT_TYPE_PRE_FINALIZE;
    } else
      event->error = EVENT_ERR_NOT_OBJECT;      /* bad object */
  } /* NULL object pointer and not PRE_NEW */
  else if (!event->object && event->type != EVENT_TYPE_PRE_NEW)
    event->error = EVENT_ERR_NOT_OBJECT;

  /* increment construct_count if PRE_NEW event
     (used in refdbg_event_dispatch) */
  if (event->type == EVENT_TYPE_PRE_NEW)
    construct_count++;

  /* dispatch the event as per current event rules */
  retval = refdbg_event_dispatch (event, objinfo ? objinfo->refcount : 0);

  /* append the event to the array (if log requested) */
  if (retval & DISPATCH_RET_LOG)
    g_array_append_val (event_array, *event);

  /* out_newindex is only set for PRE_NEW event (g_object_newv override) */
  if (out_newindex) {
    /* set construct_index if this is the only object in construction
       (first object in possibly multiple in construction) */
    if (construct_count == 1)
      construct_index = event_array->len - 1;

    /* return index to new event, or previous if not logged */
    *out_newindex = event_array->len - 1;
  }

  /* have any timer object refcounts possibly expired? */
  if (timer_oldest_index != -1
      && event->timestamp - REF_ARRAY_INDEX (timer_oldest_index)->timestamp
      > timer_expire) {
    /* clear oldest index - set in GHRFunc to the oldest remaining index */
    timer_oldest_index = -1;

    /* process timer object entries to see if any have expired */
    g_hash_table_foreach_remove (timer_last_events,
        timer_last_events_GHRFunc, GUINT_TO_POINTER (event->timestamp));
  }

  /* if new event then insert initial refcount and object type in hash */
  if (event->type == EVENT_TYPE_NEW) {
    objinfo = g_slice_new (ObjInfo);
    objinfo->type = event->obj_type;
    objinfo->refcount = event->refcount;
    g_hash_table_insert (obj_hash, event->object, objinfo);
  } else if (event->type == EVENT_TYPE_FINALIZE) {
    /* lookup object info in obj_hash and update to finalized value (0) */
    objinfo = g_hash_table_lookup (obj_hash, event->object);
    if (objinfo)
      objinfo->refcount = 0;
  }

  /* update timer last event hash if event matches timer criteria */
  if (retval & DISPATCH_RET_TIMER) {
    /* store index to new event (+1 since 0 is reserved) */
    g_hash_table_insert (timer_last_events, event->object,
        GUINT_TO_POINTER (event_array->len));

    /* set oldest timer event index if not set */
    if (timer_oldest_index == -1)
      timer_oldest_index = event_array->len - 1;
  }

  REFDBG_UNLOCK;

  return (retval);
}

#define REF_EVENT_GET_BACKTRACE(event)                                         \
do {                                                                           \
  if (backtrace_count > 0) {                                                   \
    /* get backtrace + 1 for us (we overwrite object later) */                 \
    int count = backtrace (&event->object, backtrace_count + 1);               \
                                                                               \
    /* NULL terminate the backtrace */                                         \
    if (count == 0)                                                            \
      (&event->object)[1] = NULL;                                              \
    else if (count <= backtrace_count)                                         \
      (&event->object)[count] = NULL;                                          \
  }                                                                            \
} while(0)


/* g_hash_table_foreach function to find expired timer objects */
static gboolean
timer_last_events_GHRFunc (gpointer key, gpointer value, gpointer user_data)
{
  guint index = GPOINTER_TO_UINT (value), index2;
  guint32 curtime = GPOINTER_TO_UINT (user_data);
  RefEvent *event, *stable;
  guint32 refdiff;
  int c;
  const gchar *timestamp;

  index--;                      /* stored as index + 1 */
  event = REF_ARRAY_INDEX (index);

  /* object refcount has expired? */
  if (curtime - event->timestamp > timer_expire) {      /* lookup last stable refcount (if any) */
    index2 = GPOINTER_TO_UINT (g_hash_table_lookup (timer_stable_refcounts,
            event->object));
    if (index2 != 0) {          /* previous stable refcount? */
      index2--;                 /* stored as index + 1 */
      stable = REF_ARRAY_INDEX (index2);

      /* previous stable refcount not equal to new stable refcount? */
      if (event->refcount != stable->refcount) {

        c = (event->refcount > stable->refcount) ? '+' : '-';

        refdiff = (event->refcount > stable->refcount)
            ? (event->refcount - stable->refcount)
            : (stable->refcount - event->refcount);

        timestamp = refdbg_format_time_stamp (event->timestamp);

        /* display TIMER event */
        fprintf (stderr, "TIMER <%s> [%p] |%d (%c%d)| %s\n",
            g_type_name (event->obj_type), event->object,
            event->refcount, c, refdiff, timestamp);
      }
    }

    /* store index to new stable refcount (+1 since 0 is reserved) */
    g_hash_table_insert (timer_stable_refcounts, event->object,
        GUINT_TO_POINTER (index + 1));

    return (TRUE);              /* return TRUE to delete the hash entry */
  }

  /* set timer_oldest_index if this is the oldest index so far */
  if (timer_oldest_index == -1 || index < timer_oldest_index)
    timer_oldest_index = index;

  return (FALSE);               /* don't delete the hash entry (hasn't expired yet) */
}

/* g_type_free_instance override function */
void
g_type_free_instance (GTypeInstance * instance)
{
  guint8 buf[REF_EVENT_MAX_SIZE];
  RefEvent *event = (RefEvent *) buf;
  gboolean known_object = FALSE;
  int retval = 0;

  REFDBG_LOCK;

  if (instance && g_hash_table_lookup (obj_hash, instance))
    known_object = TRUE;

  REFDBG_UNLOCK;

  if (known_object) {
    REF_EVENT_GET_BACKTRACE (event);

    event->type = EVENT_TYPE_FINALIZE;
    event->error = EVENT_ERR_NONE;
    event->object = instance;
    event->obj_type = G_OBJECT_TYPE ((GObject *) instance);
    event->refcount = 0;
  }

  orig_type_free_instance (instance);

  if (known_object)
    retval = refdbg_helper (event, 0, NULL);

  if (retval & DISPATCH_RET_BREAK)
    G_BREAKPOINT ();
}

/* g_object_newv override function */
gpointer
g_object_newv (GType object_type, guint n_parameters, GParameter * parameters)
{
  guint8 buf[REF_EVENT_MAX_SIZE];
  RefEvent *event = (RefEvent *) buf;
  RefEvent *newevent, *ci_event;
  gpointer object;
  guint newindex;
  int retval, count, i;
  guint refcount;

  if (override_test) {          /* doing function override test? */
    override_test = FALSE;
    return (NULL);
  }

  REF_EVENT_GET_BACKTRACE (event);

  event->type = EVENT_TYPE_PRE_NEW;     /* pre new event */
  event->error = EVENT_ERR_NONE;
  event->object = NULL;
  event->obj_type = object_type;
  event->refcount = 0;

  /* event structure gets re-used in below when "NEW" event is added */

  /* we add an event before creating the instance to catch ref/unref in
     construction properties, object is fixed up afterwards */
  retval = refdbg_helper (event, 0, &newindex);

  if (retval & DISPATCH_RET_BREAK)
    G_BREAKPOINT ();

  /* call original g_object_newv function */
  object = orig_object_newv (object_type, n_parameters, parameters);

  /* newindex is passed outside of lock, but is guaranteed to be valid as
     long as construct_count > 0 */

  REFDBG_LOCK;

  construct_count--;            /* object no longer in construction */

  /* fixup object for PRE_NEW event if it was logged */
  if (retval & DISPATCH_RET_LOG) {
    newevent = (RefEvent *) (event_array->data + newindex * REF_EVENT_SIZE);
    newevent->object = object;
    newevent->refcount = ((GObject *) object)->ref_count;
  }

  count = event_array->len;

  /* events added since oldest object in construction? */
  if (construct_index < count - 1) {
    ci_event = (RefEvent *) (event_array->data + (construct_index + 1)
        * REF_EVENT_SIZE);

    /* initial count to simulate ref/unref */
    refcount = 1;

    /* loop over events since first construction, fixing ref/unref on
       unknown/destroyed object entries that belong to us and printing
       an error otherwise (if this is the only object in construction).
       Breakpoints are not executed, since events have already occured. */
    for (i = construct_index + 1; i < count; i++) {
      if (ci_event->error == EVENT_ERR_UNKNOWN_OBJECT || ci_event->error == EVENT_ERR_DESTROYED_OBJECT) {       /* event involves this object and occurs after object created? */
        if (object && ci_event->object == object && i > newindex) {
          ci_event->obj_type = object_type;     /* set object type */

          if (refcount != 0) {  /* object not finalized? */
            if (ci_event->type == EVENT_TYPE_REF)
              refcount++;
            else if (ci_event->type == EVENT_TYPE_UNREF)
              refcount--;

            ci_event->refcount = refcount;
            ci_event->error = EVENT_ERR_NONE;   /* clear error */
          } else                /* object got finalized during construction! */
            refdbg_event_dispatch (ci_event, 0);
        } /* if no objs in construction or ref/unref before created */
        else if (construct_count == 0 || (object && ci_event->object == object))
          refdbg_event_dispatch (ci_event, 0);
      }

      /* advance to next event */
      ci_event = (RefEvent *) (((guint8 *) ci_event) + REF_EVENT_SIZE);
    }
  }

  REFDBG_UNLOCK;


  /* add "NEW" event (re-using event structure used by PRE_NEW event) */
  if (object) {
    ObjInfo *objinfo = g_hash_table_lookup (obj_hash, object);

    /* FIXME: for singletons _new() behaves like a _ref() */

    event->type = EVENT_TYPE_NEW;
    event->object = object;
    event->obj_type = object_type;
    event->refcount = ((GObject *) object)->ref_count;

    /* FIXME: it is possible to pass a container as a construct property and 
     * add the element to the container, which could result in a ref-count>1
     *
     * why do we have ObjInfo for some objects already? it is created in
     * refdbg_helper() for EVENT_TYPE_NEW
     */
    if (event->refcount != 1) {
      /*
      fprintf (stderr, "<%s> unexpected refcount > 1, real-refct = %d, tracked-refct = %d / %d\n",
          g_type_name (object_type), event->refcount, 
          (objinfo ? objinfo->refcount : 0), (newevent ? newevent->refcount : 0));
      */
      if (!objinfo)
        event->error = EVENT_ERR_INIT_REFCOUNT;
    } else
      event->error = EVENT_ERR_NONE;

    /* process the event */
    retval = refdbg_helper (event, 0, NULL);
  }

  if (retval & DISPATCH_RET_BREAK)
    G_BREAKPOINT ();

  return (object);
}

/* g_object_ref override function */
gpointer
g_object_ref (gpointer object)
{
  guint8 buf[REF_EVENT_MAX_SIZE];
  RefEvent *event = (RefEvent *) buf;
  int retval;

  REF_EVENT_GET_BACKTRACE (event);

  event->type = EVENT_TYPE_REF;
  event->error = EVENT_ERR_NONE;
  event->object = object;
  event->obj_type = 0;
  event->refcount = 0;

  retval = refdbg_helper (event, 1, NULL);

  if (retval & DISPATCH_RET_BREAK)
    G_BREAKPOINT ();

  return (orig_object_ref (object));
}

/* g_object_unref override function */
void
g_object_unref (gpointer object)
{
  guint8 buf[REF_EVENT_MAX_SIZE];
  RefEvent *event = (RefEvent *) buf;
  int retval;

  REF_EVENT_GET_BACKTRACE (event);

  event->type = EVENT_TYPE_UNREF;       /* finalize detected in refdbg_helper */
  event->error = EVENT_ERR_NONE;
  event->object = object;
  event->obj_type = 0;
  event->refcount = 0;

  retval = refdbg_helper (event, -1, NULL);

  if (retval & DISPATCH_RET_BREAK)
    G_BREAKPOINT ();

  orig_object_unref (object);
}


/* timeout callback for refcount timer object expire processing, also done
   in refdbg_helper, this is just a backup for when no refcount activity
   occurs for a while */
static gboolean
refdbg_timer_callback (gpointer data)
{
  guint32 timestamp;

  /* We try to lock and make sure that we are the only thread */
  if (REFDBG_TRYLOCK) {
    if (refdbg_mutex.depth > 1) {
      REFDBG_UNLOCK;
      return (TRUE);
    }

    timestamp = refdbg_get_timestamp ();

    /* any entries possibly expired? */
    if (timer_oldest_index != -1
        && timestamp - REF_ARRAY_INDEX (timer_oldest_index)->timestamp
        > timer_expire) {
      /* clear oldest index - set in GHRFunc to oldest remaining index */
      timer_oldest_index = -1;

      /* process timer object entries to see if any have expired */
      g_hash_table_foreach_remove (timer_last_events,
          timer_last_events_GHRFunc, GUINT_TO_POINTER (timestamp));
    }

    REFDBG_UNLOCK;
  }

  return (TRUE);
}
