/*
 * cmdparse.c - Command parser
 *
 * refdbg - GObject refcount debugger
 * Copyright (C) 2004-2005 Josh Green <jgreen@users.sourceforge.net>
 *               2010 Stefan Kost <ensonic@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 */
#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <sys/time.h>

#include "refdbg.h"
#include "refdbg_priv.h"

/* defines group flags and their or/and masks */
typedef struct
{
  char *name;                   /* flag name */
  guint32 or_mask;              /* or mask or NO_OR_MASK */
  guint32 and_mask;             /* and mask or NO_AND_MASK */
} GroupFlagMasks;

#define NO_OR_MASK 0
#define NO_AND_MASK 0xFFFFFFFF


static gboolean parse_rule (GScanner * scanner, EventRule * rule,
    gboolean execute);
static gboolean parse_display_rule (GScanner * scanner, DetailRule * rule);
static gboolean parse_object_rule (GScanner * scanner, DetailRule * rule);
static char *parse_gtype (GScanner * scanner);
static gboolean parse_flags (GScanner * scanner, guint32 flag_mask,
    char **flag_names, GroupFlagMasks * group_masks, guint32 * out_flags);
static gboolean parse_time_interval (GScanner * scanner, guint32 * start_time,
    guint32 * end_time);
static void display_rule (const EventRule * rule);
static void display_rule_mask (int mask);


static const GScannerConfig scanner_config = {
  " \t\n\r",                    /* cset_skip_characters */
  G_CSET_a_2_z "_" G_CSET_A_2_Z,        /* cset_identifier_first */
  G_CSET_a_2_z "_-0123456789" G_CSET_A_2_Z,     /* cset_identifier_nth */
  "#\n",                        /* cpair_comment_single */
  FALSE,                        /* case_sensitive */
  TRUE,                         /* skip_comment_multi */
  FALSE,                        /* skip_comment_single */
  FALSE,                        /* scan_comment_multi */
  TRUE,                         /* scan_identifier */
  TRUE,                         /* scan_identifier_1char */
  FALSE,                        /* scan_identifier_NULL */
  TRUE,                         /* scan_symbols */
  TRUE,                         /* scan_binary */
  TRUE,                         /* scan_octal */
  TRUE,                         /* scan_float */
  TRUE,                         /* scan_hex */
  FALSE,                        /* scan_hex_dollar */
  TRUE,                         /* scan_string_sq */
  TRUE,                         /* scan_string_dq */
  TRUE,                         /* numbers_2_int */
  FALSE,                        /* int_2_float */
  FALSE,                        /* identifier_2_string */
  TRUE,                         /* char_2_token */
  TRUE,                         /* symbol_2_token */
  FALSE,                        /* scope_0_fallback */
};

/* flag names for event rules */
static char *rule_flag_names[] = {
  "PreNew",
  "New",
  "Ref",
  "Unref",
  "PreFinalize",
  "Finalize",

  "EUnknownObj",
  "EDestroyedObj",
  "ENotObj",
  "EInitCount",
  "EBadCount",

  "Paranoid",
  "Timer",
  NULL
};

/* group flag masks for event rules */
static GroupFlagMasks rule_group_masks[] = {
  {"All", EVENT_RULE_MASK_ALL, NO_AND_MASK},
  {"None", NO_OR_MASK, EVENT_RULE_MASK_NONE},
  {"Event", EVENT_RULE_MASK_EVENT, NO_AND_MASK},
  {"Error", EVENT_RULE_MASK_ERROR, NO_AND_MASK},
  {NULL, 0, 0}
};

/* 'object' command flag names */
static char *object_rule_flag_names[] = {
  "Active",
  "Destroyed",
  NULL
};

/* 'object' command flag group masks */
static GroupFlagMasks object_rule_group_masks[] = {
  {"All", OBJECT_RULE_MASK_ALL, NO_AND_MASK},
  {NULL, 0, 0}
};

/**
 * refdbg_parse_commands:
 * @cmds: Commands to execute (separated by semi-colons ;)
 * @execute: Set to %TRUE to execute commands, %FALSE to parse only
 *
 * Execute refdbg commands.
 *
 * Returns: %TRUE on success, %FALSE otherwise (in which case err may be set)
 */
gboolean
refdbg_cmd (const char *cmds)
{                               /* just parse commands the first time to make sure all are OK */
  if (!refdbg_exec (cmds, FALSE))
    return (FALSE);
  return (refdbg_exec (cmds, TRUE));    /* now execute commands */
}

/* internal function to parse commands and execute if @execute is TRUE */
gboolean
refdbg_exec (const char *cmds, gboolean execute)
{
  GScanner *scanner;
  EventRule *rule, temprule;
  DetailRule detail_rule;
  char *var, *topic;
  guint token;
  int rulenum;
  int i;

  if (!cmds || strlen (cmds) == 0)
    return (TRUE);

  scanner = g_scanner_new ((GScannerConfig *) & scanner_config);
  g_scanner_input_text (scanner, cmds, strlen (cmds));
  scanner->input_name = "refdbg";

  while (g_scanner_peek_next_token (scanner) != G_TOKEN_EOF) {
    token = g_scanner_get_next_token (scanner); /* get token */

    if (token == ';')
      continue;                 /* command separator? */

    if (token != G_TOKEN_IDENTIFIER) {
      g_scanner_unexp_token (scanner, G_TOKEN_IDENTIFIER, NULL, NULL,
          NULL, NULL, TRUE);
      return (FALSE);
    }

    var = scanner->value.v_identifier;

    if (casecmp (var, "help") == 0) {   /* help command */
      topic = NULL;
      if (g_scanner_peek_next_token (scanner) == G_TOKEN_IDENTIFIER) {
        token = g_scanner_get_next_token (scanner);     /* read topic */
        topic = scanner->value.v_identifier;
      }

      if (execute)
        refdbg_help (topic);
    } else if (casecmp (var, "exit") == 0       /* exit/quit/q command */
        || casecmp (var, "quit") == 0 || casecmp (var, "q") == 0) {
      if (execute)
        refdbg_exit = TRUE;
    } else if (casecmp (var, "display") == 0) { /* display command */
      if (!parse_display_rule (scanner, &detail_rule))
        return (FALSE);
      if (execute)
        refdbg_display_events (&detail_rule, NULL);
      release_detail_rule (&detail_rule);
    } else if (casecmp (var, "clear") == 0) {   /* clear command */
      if (execute)
        refdbg_clear ();
    } else if (casecmp (var, "objects") == 0) {
      if (!parse_object_rule (scanner, &detail_rule))
        return (FALSE);
      if (execute)
        refdbg_display_objects (&detail_rule, NULL);
      release_detail_rule (&detail_rule);
    } else if (casecmp (var, "stats") == 0) {   /* stats command */
      if (execute)
        refdbg_stats ();
    } else if (casecmp (var, "btnum") == 0) {   /* backtrace max caller addrs *//* variable assignment? */
      if (g_scanner_peek_next_token (scanner) == '=') {
        g_scanner_get_next_token (scanner);     /* skip '=' */

        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT || (i = scanner->value.v_int) < 0) {
          g_scanner_error (scanner, "Unsigned integer value expected");
          return (FALSE);
        }

        if (refdbg_active) {
          g_scanner_error (scanner,
              "Assignment attempt to init-time only variable when refdbg already initialized");
          return (FALSE);
        }

        if (execute) {
          if (i > REFDBG_MAX_BACKTRACE_COUNT) {
            g_scanner_warn (scanner,
                "Value exceeds REFDBG_MAX_BACKTRACE_COUNT defined in refdbg.h, setting to max of %d",
                REFDBG_MAX_BACKTRACE_COUNT);
            i = REFDBG_MAX_BACKTRACE_COUNT;
          }

          backtrace_count = i;
        }
      } else if (execute)
        fprintf (stderr, "%d\n", backtrace_count);
    } else if (casecmp (var, "btpaths") == 0) { /* enable paths in backtraces? *//* variable assignment? */
      if (g_scanner_peek_next_token (scanner) == '=') {
        g_scanner_get_next_token (scanner);     /* skip '=' */

        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT) {
          g_scanner_error (scanner, "Unsigned integer value expected");
          return (FALSE);
        }

        i = scanner->value.v_int;
        if (execute)
          refdbg_enable_bt_paths = (i != 0);
      } else if (execute)
        fprintf (stderr, "%d\n", refdbg_enable_bt_paths);
    } else if (casecmp (var, "notimer") == 0) { /* disable glib timer callback */
      if (refdbg_active) {
        g_scanner_error (scanner,
            "The 'notimer' command is valid at init-time only, refdbg is already active");
        return (FALSE);
      }

      if (execute)
        refdbg_enable_timer = FALSE;
    } else if (casecmp (var, "dispmax") == 0) { /* max display event limit *//* variable assignment? */
      if (g_scanner_peek_next_token (scanner) == '=') {
        g_scanner_get_next_token (scanner);     /* skip '=' */

        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT) {
          g_scanner_error (scanner, "Unsigned integer value expected");
          return (FALSE);
        }

        if (scanner->value.v_int == 0) {
          g_scanner_error (scanner, "Expected positive dispmax value");
          return (FALSE);
        }

        if (execute)
          refdbg_dispmax = scanner->value.v_int;
      } else if (execute)
        fprintf (stderr, "%d\n", refdbg_dispmax);
    } else if (casecmp (var, "savelog") == 0) { /* save log now */
      if (g_scanner_peek_next_token (scanner) == '=') {
        g_scanner_error (scanner,
            "Command has no arguments (old savelog command?)");
        return (FALSE);
      }

      if (execute)
        refdbg_save_log ();
    } else if (casecmp (var, "logname") == 0) { /* set the logfile name */
      if (g_scanner_peek_next_token (scanner) == '=') {
        g_scanner_get_next_token (scanner);     /* skip '=' */

        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_STRING) {
          g_scanner_error (scanner, "String value expected");
          return (FALSE);
        }

        if (scanner->value.v_string == NULL || *scanner->value.v_string == '\0') {
          g_scanner_error (scanner, "Expected non empty logname value");
          return (FALSE);
        }

        if (execute) {
          g_free (refdbg_logname);
          refdbg_logname = g_strdup (scanner->value.v_string);
        }
      } else if (execute)
        fprintf (stderr, "%s\n", refdbg_logname);
    } else if (casecmp (var, "logobjs") == 0) { /* save active objects to log *//* variable assignment? */
      if (g_scanner_peek_next_token (scanner) == '=') {
        g_scanner_get_next_token (scanner);     /* skip '=' */

        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT) {
          g_scanner_error (scanner, "Unsigned integer value expected");
          return (FALSE);
        }

        i = scanner->value.v_int;
        if (execute)
          log_objects = (i != 0);
      } else if (execute)
        fprintf (stderr, "%d\n", log_objects);
    } else if (casecmp (var, "logexit") == 0) { /* save log on exit *//* variable assignment? */
      if (g_scanner_peek_next_token (scanner) == '=') {
        g_scanner_get_next_token (scanner);     /* skip '=' */

        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT) {
          g_scanner_error (scanner, "Unsigned integer value expected");
          return (FALSE);
        }

        i = scanner->value.v_int;
        if (execute)
          save_event_log = (i != 0);
      } else if (execute)
        fprintf (stderr, "%d\n", save_event_log);
    } else if (casecmp (var, "statsexit") == 0) { /* display stats on exit *//* variable assignment? */
      if (g_scanner_peek_next_token (scanner) == '=') {
        g_scanner_get_next_token (scanner);     /* skip '=' */

        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT) {
          g_scanner_error (scanner, "Unsigned integer value expected");
          return (FALSE);
        }

        i = scanner->value.v_int;
        if (execute)
          display_object_stats = (i != 0);
      } else if (execute)
        fprintf (stderr, "%d\n", display_object_stats);
    } else if (casecmp (var, "timer") == 0) {   /* timer interval (0 = disable) *//* variable assignment? */
      if (g_scanner_peek_next_token (scanner) == '=') {
        g_scanner_get_next_token (scanner);     /* skip '=' */

        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT) {
          g_scanner_error (scanner, "Unsigned integer value expected");
          return (FALSE);
        }

        i = scanner->value.v_int;
        if (i != 0 && i < 100) {
          g_scanner_error (scanner,
              "Timer interval (in milliseconds) too small");
          return (FALSE);
        }

        /* convert to microseconds */
        if (execute)
          timer_expire = i * 1000;
      } else if (execute)
        fprintf (stderr, "%d\n", timer_expire);
    } else if (casecmp (var, "time") == 0) {    /* get current time */
      if (execute) {
        guint32 timestamp = refdbg_get_timestamp ();

        fprintf (stderr, "%s\n", refdbg_format_time_stamp (timestamp));
      }
    } else if (casecmp (var, "addrule") == 0) { /* add rule *//* make sure we haven't reached max rules (count excludes r0) */
      if (event_rule_count >= REFDBG_MAX_EVENT_RULES - 1) {
        g_scanner_error (scanner, "Exceeded max event rule count of %d",
            REFDBG_MAX_EVENT_RULES);
        return (FALSE);
      }

      if (!parse_rule (scanner, &temprule, execute))
        return (FALSE);

      if (execute) {
        event_rule_count++;
        event_rules[event_rule_count] = temprule;
      }
    } else if (casecmp (var, "insrule") == 0) { /* insert rule *//* make sure we haven't reached max rules (count excludes r0) */
      if (event_rule_count >= REFDBG_MAX_EVENT_RULES - 1) {
        g_scanner_error (scanner, "Exceeded max event rule count of %d",
            REFDBG_MAX_EVENT_RULES);
        return (FALSE);
      }

      token = g_scanner_get_next_token (scanner);
      if (token != G_TOKEN_INT) {
        g_scanner_unexp_token (scanner, G_TOKEN_INT, NULL, NULL, NULL,
            "Rule index to insert at", TRUE);
        return (FALSE);
      }

      rulenum = scanner->value.v_int;   /* insert position */

      if (!parse_rule (scanner, &temprule, execute))    /* parse the rule */
        return (FALSE);

      if (execute) {            /* need to shuffle rules for insertion? */
        if (rulenum <= event_rule_count) {
          for (i = event_rule_count + 1; i > rulenum; i--)
            event_rules[i] = event_rules[i - 1];
        }

        event_rule_count++;
        event_rules[rulenum] = temprule;
      }
    } else if (casecmp (var, "delrule") == 0) { /* delete rule? */
      token = g_scanner_get_next_token (scanner);
      if (token != G_TOKEN_INT) {
        g_scanner_unexp_token (scanner, G_TOKEN_INT, NULL, NULL, NULL,
            "Rule index to delete", TRUE);
        return (FALSE);
      }

      rulenum = scanner->value.v_int;   /* rule to delete */

      if (rulenum > event_rule_count) {
        g_scanner_error (scanner, "No event rule with index %d", rulenum);
        return (FALSE);
      }

      if (event_rule_count == 0 && rulenum == 0) {
        g_scanner_error (scanner, "Cannot delete default rule");
        return (FALSE);
      }

      if (execute) {            /* free unresolved type names (if any) */
        g_free (event_rules[rulenum].inc_type_name);
        g_free (event_rules[rulenum].exc_type_name);

        /* need to shuffle rules? */
        if (rulenum < event_rule_count) {
          for (i = rulenum; i < event_rule_count; i++)
            event_rules[i] = event_rules[i + 1];
        }

        event_rule_count--;
      }
    } else if (casecmp (var, "rules") == 0) {   /* display rules */
      if (execute) {
        for (i = 0; i <= event_rule_count; i++) {
          rule = &event_rules[i];
          fprintf (stderr, "r%d: ", i);
          display_rule (rule);
        }
      }
    } else if (sscanf (var, "r%2u", &rulenum) == 1) {   /* a specific rule */
      if (rulenum > event_rule_count) {
        g_scanner_error (scanner, "No event rule with index %d", rulenum);
        return (FALSE);
      }

      token = g_scanner_peek_next_token (scanner);
      if (token == '=') {       /* assignment? */
        g_scanner_get_next_token (scanner);     /* skip '=' */
        if (!parse_rule (scanner, &temprule, execute))
          return (FALSE);

        if (execute) {          /* free unresolved type names (if any) */
          g_free (event_rules[rulenum].inc_type_name);
          g_free (event_rules[rulenum].exc_type_name);
          event_rules[rulenum] = temprule;
        }
      } else
        display_rule (&event_rules[rulenum]);
    } else {                    /* unexpected identifier */

      g_scanner_error (scanner, "Unknown variable or command '%s'",
          scanner->value.v_identifier);
      return (FALSE);
    }
  }

  return (TRUE);
}

/* parse a event rule string, @execute is mainly to supress warnings when
   only parsing and not executing */
static gboolean
parse_rule (GScanner * scanner, EventRule * rule, gboolean execute)
{
  char *typename, *s;
  guint token;
  guint flags;
  GType type;

  rule->inc_type = 0;
  rule->exc_type = 0;
  rule->inc_type_name = NULL;
  rule->exc_type_name = NULL;
  rule->inc_object = NULL;
  rule->exc_object = NULL;
  rule->display_mask = EVENT_RULE_MASK_ERROR;
  rule->break_mask = EVENT_RULE_MASK_NONE;
  rule->log_mask = EVENT_RULE_MASK_ALL;

  while (TRUE) {
    token = g_scanner_peek_next_token (scanner);
    if (token == G_TOKEN_EOF || token == ';')
      break;

    token = g_scanner_get_next_token (scanner);
    switch (token) {
      case '<':                /* start of include or exclude GType */
        if ((typename = parse_gtype (scanner))) {
          type = g_type_from_name (typename + (typename[0] == '!' ? 1 : 0));
          if (typename[0] != '!') {     /* regular type (no bang?) */
            if (rule->inc_type || rule->inc_type_name) {
              g_scanner_error (scanner, "Multiple include GTypes not allowed");
              goto err;
            }

            if (type)
              rule->inc_type = type;
            else
              rule->inc_type_name = g_strdup (typename);
          } else {              /* exclude type (we have bang!) */

            if (rule->exc_type || rule->exc_type_name) {
              g_scanner_error (scanner, "Multiple exclude GTypes not allowed");
              goto err;
            }

            if (type)
              rule->exc_type = type;
            else
              rule->exc_type_name = g_strdup (typename + 1);
          }

          g_free (typename);    /* free the typename */
        } else
          goto err;             /* parse_gtype failed */
        break;
      case '!':                /* start of exclude object address */
        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT) {
          g_scanner_unexp_token (scanner, G_TOKEN_INT, NULL, NULL, NULL,
              "GObject address to exclude", TRUE);
          goto err;
        }

        if (rule->exc_object) {
          g_scanner_error (scanner,
              "Multiple exclude object addresses not allowed");
          goto err;
        }

        rule->exc_object = GUINT_TO_POINTER (scanner->value.v_int);
        break;
      case G_TOKEN_INT:        /* include object address */
        if (rule->inc_object) {
          g_scanner_error (scanner,
              "Multiple include object addresses not allowed");
          goto err;
        }

        rule->inc_object = GUINT_TO_POINTER (scanner->value.v_int);
        break;
      case G_TOKEN_IDENTIFIER: /* D/B/L character for event masks */
        s = scanner->value.v_identifier;
        if (strlen (s) > 1) {
          g_scanner_error (scanner, "Invalid event rule format string");
          goto err;
        }

        switch (g_ascii_toupper (s[0])) {
          case 'D':
            if (!parse_flags (scanner, EVENT_RULE_MASK_ALL
                    | EVENT_RULE_FLAG_TIMER
                    | EVENT_RULE_FLAG_PARANOID, rule_flag_names,
                    rule_group_masks, &flags))
              goto err;
            rule->display_mask = flags;
            break;
          case 'B':
            if (!parse_flags (scanner, EVENT_RULE_MASK_ALL
                    | EVENT_RULE_FLAG_PARANOID, rule_flag_names,
                    rule_group_masks, &flags))
              goto err;
            rule->break_mask = flags;
            break;
          case 'L':
            if (!parse_flags (scanner, EVENT_RULE_MASK_ALL,
                    rule_flag_names, rule_group_masks, &flags))
              goto err;
            rule->log_mask = flags;
            break;
          default:
            g_scanner_error (scanner, "Invalid event rule format string");
            goto err;
        }
        break;
      default:
        g_scanner_error (scanner, "Invalid event rule format string");
        goto err;
    }
  }

  if (execute) {
    /* check if D:Timer set but L:New|Ref|Unref|Finalize is not */
    if (rule->display_mask & EVENT_RULE_FLAG_TIMER
        && (!(rule->log_mask & EVENT_RULE_FLAG_NEW)
            || !(rule->log_mask & EVENT_RULE_FLAG_REF)
            || !(rule->log_mask & EVENT_RULE_FLAG_UNREF)
            || !(rule->log_mask & EVENT_RULE_FLAG_FINALIZE))) {
      rule->log_mask |= EVENT_RULE_FLAG_NEW | EVENT_RULE_FLAG_REF
          | EVENT_RULE_FLAG_UNREF | EVENT_RULE_FLAG_FINALIZE;
      g_warning ("Event flag 'Timer' implies L:New|Ref|Unref|Finalize");
    }

    if (rule->inc_type_name)
      g_message ("GType '%s' has not yet been registered, will keep trying"
          " to resolve it", rule->inc_type_name);

    if (rule->exc_type_name)
      g_message ("GType '%s' has not yet been registered, will keep trying"
          " to resolve it", rule->exc_type_name);
  }

  return (TRUE);

err:

  g_free (rule->inc_type_name);
  g_free (rule->exc_type_name);

  return (FALSE);
}

/* parse a 'display' command event rule string */
static gboolean
parse_display_rule (GScanner * scanner, DetailRule * rule)
{
  GArray *inc_types, *exc_types, *inc_objects, *exc_objects;
  char *typename, *s;
  gpointer ptr;
  guint token;
  guint flags;
  GType type;
  int i;

  inc_types = g_array_new (TRUE, FALSE, sizeof (GType));
  exc_types = g_array_new (TRUE, FALSE, sizeof (GType));
  inc_objects = g_array_new (TRUE, FALSE, sizeof (gpointer));
  exc_objects = g_array_new (TRUE, FALSE, sizeof (gpointer));

  rule->inc_types = NULL;
  rule->exc_types = NULL;
  rule->inc_objects = NULL;
  rule->exc_objects = NULL;
  rule->inc_type_name = NULL;
  rule->exc_type_name = NULL;
  rule->display_mask = EVENT_RULE_MASK_ALL;
  rule->btnum = -1;
  rule->start_time = 0;
  rule->end_time = 0;
  rule->limit_max = refdbg_dispmax;
  rule->limit_ofs = 0;

  while (TRUE) {
    token = g_scanner_peek_next_token (scanner);
    if (token == G_TOKEN_EOF || token == ';')
      break;

    token = g_scanner_get_next_token (scanner);
    switch (token) {
      case '<':                /* start of include or exclude GType */
        if ((typename = parse_gtype (scanner))) {
          s = typename + (typename[0] == '!' ? 1 : 0);  /* ptr to name */
          type = g_type_from_name (s);  /* resolve the type name */
          if (!type) {
            g_scanner_error (scanner, "Could not resolve GType '%s'", s);
            goto err;
          }

          if (typename[0] != '!')       /* regular type (no bang?) */
            g_array_append_val (inc_types, type);
          else
            g_array_append_val (exc_types, type);       /* exclude type */

          g_free (typename);    /* free the typename */
        } else
          goto err;             /* parse_gtype failed */
        break;
      case '!':                /* start of exclude object address */
        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT) {
          g_scanner_unexp_token (scanner, G_TOKEN_INT, NULL, NULL, NULL,
              "GObject address to exclude", TRUE);
          goto err;
        }

        ptr = GUINT_TO_POINTER (scanner->value.v_int);
        g_array_append_val (exc_objects, ptr);
        break;
      case '[':                /* time interval? */
        if (!parse_time_interval (scanner, &rule->start_time, &rule->end_time))
          goto err;
        break;
      case G_TOKEN_INT:        /* include object address */
        ptr = GUINT_TO_POINTER (scanner->value.v_int);
        g_array_append_val (inc_objects, ptr);
        break;
      case G_TOKEN_IDENTIFIER: /* string identifier */
        s = scanner->value.v_identifier;

        if (casecmp (s, "D") == 0) {    /* D:<Flags> options? */
          if (!parse_flags (scanner, EVENT_RULE_MASK_ALL,
                  rule_flag_names, rule_group_masks, &flags))
            goto err;

          rule->display_mask = flags;
        } else if (casecmp (s, "btnum") == 0) { /* backtrace max callers */
          token = g_scanner_get_next_token (scanner);
          if (token != '=') {
            g_scanner_error (scanner, "Expected assignment '=' to btnum");
            goto err;
          }

          token = g_scanner_get_next_token (scanner);
          if (token != G_TOKEN_INT) {
            g_scanner_error (scanner, "Unsigned integer value expected");
            goto err;
          }

          rule->btnum = scanner->value.v_int;
        } else if (casecmp (s, "limit") == 0) { /* limit argument */
          gboolean neg_ofs = FALSE;

          token = g_scanner_get_next_token (scanner);
          if (token != '=') {
            g_scanner_error (scanner, "Expected assignment '=' to limit");
            goto err;
          }

          /* a negative number indicates offset from end */
          token = g_scanner_get_next_token (scanner);
          if (token == '-') {
            neg_ofs = TRUE;
            token = g_scanner_get_next_token (scanner);
          }

          if (token != G_TOKEN_INT) {
            g_scanner_error (scanner, "Integer value expected");
            goto err;
          }

          i = scanner->value.v_int;

          if (g_scanner_peek_next_token (scanner) == ',') {
            if (!neg_ofs)
              rule->limit_ofs = i;
            else
              rule->limit_ofs = -i;

            g_scanner_get_next_token (scanner); /* skip ',' */

            token = g_scanner_get_next_token (scanner);
            if (token != G_TOKEN_INT) {
              g_scanner_error (scanner, "Unsigned integer value expected");
              goto err;
            }

            rule->limit_max = scanner->value.v_int;
          } else {
            if (neg_ofs) {      /* limit max should not be negative */
              g_scanner_error (scanner, "Unsigned integer value expected");
              goto err;
            }

            rule->limit_max = i;
          }

          if (rule->limit_max == 0) {
            g_scanner_error (scanner, "Expected positive limit max");
            goto err;
          }
        } else {
          g_scanner_error (scanner, "Invalid display rule format string");
          goto err;
        }
        break;
      default:
        g_scanner_error (scanner, "Invalid display rule format string");
        goto err;
    }
  }

  if (inc_types->len > 0) {
    rule->inc_types = (GType *) (inc_types->data);
    g_array_free (inc_types, FALSE);
  } else
    g_array_free (inc_types, TRUE);

  if (exc_types->len > 0) {
    rule->exc_types = (GType *) (exc_types->data);
    g_array_free (exc_types, FALSE);
  } else
    g_array_free (exc_types, TRUE);

  if (inc_objects->len > 0) {
    rule->inc_objects = (gpointer *) (inc_objects->data);
    g_array_free (inc_objects, FALSE);
  } else
    g_array_free (inc_objects, TRUE);

  if (exc_objects->len > 0) {
    rule->exc_objects = (gpointer *) (exc_objects->data);
    g_array_free (exc_objects, FALSE);
  } else
    g_array_free (exc_objects, TRUE);

  return (TRUE);

err:

  g_array_free (inc_types, TRUE);
  g_array_free (exc_types, TRUE);
  g_array_free (inc_objects, TRUE);
  g_array_free (exc_objects, TRUE);

  return (FALSE);
}

/* parse an 'object' command event rule string */
static gboolean
parse_object_rule (GScanner * scanner, DetailRule * rule)
{
  GArray *inc_types, *exc_types, *inc_objects, *exc_objects;
  char *typename, *s;
  gpointer ptr;
  guint token;
  guint flags;
  GType type;

  inc_types = g_array_new (TRUE, FALSE, sizeof (GType));
  exc_types = g_array_new (TRUE, FALSE, sizeof (GType));
  inc_objects = g_array_new (TRUE, FALSE, sizeof (gpointer));
  exc_objects = g_array_new (TRUE, FALSE, sizeof (gpointer));

  rule->inc_types = NULL;
  rule->exc_types = NULL;
  rule->inc_objects = NULL;
  rule->exc_objects = NULL;
  rule->inc_type_name = NULL;
  rule->exc_type_name = NULL;
  rule->display_mask = OBJECT_RULE_FLAG_ACTIVE;
  rule->btnum = -1;
  rule->start_time = 0;
  rule->end_time = 0;
  rule->limit_max = refdbg_dispmax;
  rule->limit_ofs = 0;

  while (TRUE) {
    token = g_scanner_peek_next_token (scanner);
    if (token == G_TOKEN_EOF || token == ';')
      break;

    token = g_scanner_get_next_token (scanner);
    switch (token) {
      case '<':                /* start of include or exclude GType */
        if ((typename = parse_gtype (scanner))) {
          s = typename + (typename[0] == '!' ? 1 : 0);  /* ptr to name */
          type = g_type_from_name (s);  /* resolve the type name */
          if (!type) {
            g_scanner_error (scanner, "Could not resolve GType '%s'", s);
            goto err;
          }

          if (typename[0] != '!')       /* regular type (no bang?) */
            g_array_append_val (inc_types, type);
          else
            g_array_append_val (exc_types, type);       /* exclude type */

          g_free (typename);    /* free the typename */
        } else
          goto err;             /* parse_gtype failed */
        break;
      case '!':                /* start of exclude object address */
        token = g_scanner_get_next_token (scanner);
        if (token != G_TOKEN_INT) {
          g_scanner_unexp_token (scanner, G_TOKEN_INT, NULL, NULL, NULL,
              "GObject address to exclude", TRUE);
          goto err;
        }

        ptr = GUINT_TO_POINTER (scanner->value.v_int);
        g_array_append_val (exc_objects, ptr);
        break;
      case G_TOKEN_INT:        /* include object address */
        ptr = GUINT_TO_POINTER (scanner->value.v_int);
        g_array_append_val (inc_objects, ptr);

        rule->display_mask = OBJECT_RULE_FLAG_ACTIVE
            | OBJECT_RULE_FLAG_DESTROYED;
        break;
      case G_TOKEN_IDENTIFIER: /* string identifier */
        s = scanner->value.v_identifier;

        if (casecmp (s, "O") != 0) {    /* Not O:<Flags> options? */
          g_scanner_error (scanner, "Invalid display rule format string");
          goto err;
        }

        if (!parse_flags (scanner, OBJECT_RULE_MASK_ALL,
                object_rule_flag_names, object_rule_group_masks, &flags))
          goto err;

        rule->display_mask = flags;
        break;
      default:
        g_scanner_error (scanner, "Invalid display rule format string");
        goto err;
    }
  }

  if (inc_types->len > 0) {
    rule->inc_types = (GType *) (inc_types->data);
    g_array_free (inc_types, FALSE);
  } else
    g_array_free (inc_types, TRUE);

  if (exc_types->len > 0) {
    rule->exc_types = (GType *) (exc_types->data);
    g_array_free (exc_types, FALSE);
  } else
    g_array_free (exc_types, TRUE);

  if (inc_objects->len > 0) {
    rule->inc_objects = (gpointer *) (inc_objects->data);
    g_array_free (inc_objects, FALSE);
  } else
    g_array_free (inc_objects, TRUE);

  if (exc_objects->len > 0) {
    rule->exc_objects = (gpointer *) (exc_objects->data);
    g_array_free (exc_objects, FALSE);
  } else
    g_array_free (exc_objects, TRUE);

  return (TRUE);

err:

  g_array_free (inc_types, TRUE);
  g_array_free (exc_types, TRUE);
  g_array_free (inc_objects, TRUE);
  g_array_free (exc_objects, TRUE);

  return (FALSE);
}

/* parses a [!]GType> argument and returns [!]GType string */
static char *
parse_gtype (GScanner * scanner)
{
  char *typename;
  guint token;
  gboolean not;

  token = g_scanner_get_next_token (scanner);
  if (token == '!') {           /* check for exclude token '!' */
    not = TRUE;
    token = g_scanner_get_next_token (scanner);
  } else
    not = FALSE;

  if (token != G_TOKEN_IDENTIFIER) {    /* check for GType identifier */
    g_scanner_unexp_token (scanner, G_TOKEN_IDENTIFIER, NULL, NULL,
        NULL, "GType string", TRUE);
    return (NULL);
  }

  if (not)
    typename = g_strconcat ("!", scanner->value.v_identifier, NULL);
  else
    typename = g_strdup (scanner->value.v_identifier);

  token = g_scanner_get_next_token (scanner);
  if (token != '>') {           /* check for exclude token '!' */
    g_scanner_unexp_token (scanner, '>', NULL, NULL,
        NULL, "GType terminator '>'", TRUE);
    g_free (typename);
    return (NULL);
  }

  return (typename);
}

/* parses :<Flags> field where <Flags> is found in flag_names or group_masks
   and is set in flag_mask */
static gboolean
parse_flags (GScanner * scanner, guint32 flag_mask, char **flag_names,
    GroupFlagMasks * group_masks, guint32 * out_flags)
{
  guint token;
  guint32 flags;
  int i;

  token = g_scanner_get_next_token (scanner);
  if (token != ':') {           /* check for colon ':' */
    g_scanner_unexp_token (scanner, ':', NULL, NULL, NULL,
        "Colon separator ':'", TRUE);
    return (FALSE);
  }

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_IDENTIFIER) {    /* check for first flag name */
    g_scanner_unexp_token (scanner, G_TOKEN_IDENTIFIER, NULL, NULL,
        NULL, "Flag name", TRUE);
    return (FALSE);
  }

  flags = 0;
  while (TRUE) {                /* loop while we have flag names *//* search for flag name in group masks */
    for (i = 0; group_masks[i].name; i++) {
      if (casecmp (scanner->value.v_identifier, group_masks[i].name) == 0) {
        flags |= group_masks[i].or_mask;
        flags &= group_masks[i].and_mask;
        break;
      }
    }

    /* flag name not found in group_masks? */
    if (!group_masks[i].name) { /* search for flag name in file names */
      for (i = 0; flag_names[i]; i++) {
        if (casecmp (scanner->value.v_identifier, flag_names[i]) == 0
            && (flag_mask & (1 << i))) {
          flags |= 1 << i;
          break;
        }
      }

      if (!flag_names[i]) {     /* flag not found? */
        g_scanner_error (scanner, "Unknown/invalid flag name '%s'",
            scanner->value.v_identifier);
        return (FALSE);
      }
    }

    token = g_scanner_peek_next_token (scanner);
    if (token == '|') {         /* check for flag separator */
      g_scanner_get_next_token (scanner);       /* skip '|' token */

      token = g_scanner_get_next_token (scanner);
      if (token != G_TOKEN_IDENTIFIER) {        /* check for flag name */
        g_scanner_unexp_token (scanner, G_TOKEN_IDENTIFIER, NULL,
            NULL, NULL, "Flag name", TRUE);
        return (FALSE);
      }
    } else
      break;                    /* not flag separator, break out */
  }

  *out_flags = flags;

  return (TRUE);
}

static gboolean
parse_time_interval (GScanner * scanner, guint32 * start_time,
    guint32 * end_time)
{
  guint token;
  guint minutes, seconds, msecs;

  token = g_scanner_get_next_token (scanner);
  if (token != '-') {           /* check for '-', start time present? */
    minutes = seconds = msecs = 0;

    if (token == G_TOKEN_INT) { /* integer minutes field? */
      minutes = scanner->value.v_int;

      token = g_scanner_get_next_token (scanner);
      if (token == ':') {       /* we have seconds? */
        token = g_scanner_get_next_token (scanner);
        if (token == G_TOKEN_FLOAT) {   /* seconds.msecs? */
          seconds = scanner->value.v_float;
          msecs = (scanner->value.v_float - seconds) * 1000000.0;
          token = g_scanner_get_next_token (scanner);
        } else if (token == G_TOKEN_INT) {      /* seconds but not msecs? */
          seconds = scanner->value.v_int;
          token = g_scanner_get_next_token (scanner);
        }
      }
    } else if (token == G_TOKEN_FLOAT) {        /* seconds.msecs field? */
      seconds = scanner->value.v_float;
      msecs = (scanner->value.v_float - seconds) * 1000000.0;
      token = g_scanner_get_next_token (scanner);
    } else {
      g_scanner_error (scanner, "Invalid start time value");
      return (FALSE);
    }

    *start_time = minutes * 60000000 + seconds * 1000000 + msecs;
  }

  if (token == '-') {           /* end time present? */
    minutes = seconds = msecs = 0;

    token = g_scanner_get_next_token (scanner);
    if (token == G_TOKEN_INT) { /* integer minutes field? */
      minutes = scanner->value.v_int;

      token = g_scanner_get_next_token (scanner);
      if (token == ':') {       /* we have seconds? */
        token = g_scanner_get_next_token (scanner);
        if (token == G_TOKEN_FLOAT) {   /* seconds.msecs? */
          seconds = scanner->value.v_float;
          msecs = (scanner->value.v_float - seconds) * 1000000.0;
          token = g_scanner_get_next_token (scanner);
        } else if (token == G_TOKEN_INT) {      /* seconds but not msecs? */
          seconds = scanner->value.v_int;
          token = g_scanner_get_next_token (scanner);
        }
      }

      *end_time = minutes * 60000000 + seconds * 1000000 + msecs;
    } else if (token == G_TOKEN_FLOAT) {        /* seconds.msecs field? */
      seconds = scanner->value.v_float;
      msecs = (scanner->value.v_float - seconds) * 1000000.0;
      token = g_scanner_get_next_token (scanner);

      *end_time = minutes * 60000000 + seconds * 1000000 + msecs;
    }
    /* we allow a trailing '-' for omitted end time */
  }

  if (token != ']') {
    g_scanner_error (scanner, "Invalid time interval string");
    return (FALSE);
  }

  return (TRUE);
}

/* display an event rule string to stderr */
static void
display_rule (const EventRule * rule)
{
  if (rule->inc_type)
    fprintf (stderr, "<%s> ", g_type_name (rule->inc_type));
  else if (rule->inc_type_name)
    fprintf (stderr, "<\"%s\"> ", rule->inc_type_name);

  if (rule->exc_type)
    fprintf (stderr, "<!%s> ", g_type_name (rule->exc_type));
  else if (rule->exc_type_name)
    fprintf (stderr, "<!\"%s\"> ", rule->inc_type_name);

  if (rule->inc_object)
    fprintf (stderr, "%p ", rule->inc_object);
  if (rule->exc_object)
    fprintf (stderr, "!%p ", rule->exc_object);

  fputs ("D:", stderr);
  display_rule_mask (rule->display_mask);

  fputs (" B:", stderr);
  display_rule_mask (rule->break_mask);

  fputs (" L:", stderr);
  display_rule_mask (rule->log_mask);

  fputc ('\n', stderr);
}

/* display an event rule mask description string to stderr */
static void
display_rule_mask (int mask)
{
  gboolean first;
  int i;

  first = FALSE;
  if ((mask & EVENT_RULE_MASK_ALL) == EVENT_RULE_MASK_ALL) {
    fprintf (stderr, "All");
    mask &= ~EVENT_RULE_MASK_ALL;
  } else if ((mask & EVENT_RULE_MASK_EVENT) == EVENT_RULE_MASK_EVENT) {
    fprintf (stderr, "Event");
    mask &= ~EVENT_RULE_MASK_EVENT;
  } else if ((mask & EVENT_RULE_MASK_ERROR) == EVENT_RULE_MASK_ERROR) {
    fprintf (stderr, "Error");
    mask &= ~EVENT_RULE_MASK_ERROR;
  } else if (mask == 0)
    fprintf (stderr, "None");
  else
    first = TRUE;

  for (i = 0; i < EVENT_RULE_FLAG_COUNT; i++) {
    if (mask & (1 << i)) {
      fprintf (stderr, "%s%s", first ? "" : "|", rule_flag_names[i]);
      first = FALSE;
    }
  }
}

/* initialize a detail rule structure */
void
init_detail_rule (DetailRule * rule)
{
  memset (rule, 0, sizeof (DetailRule));
  rule->btnum = -1;
  rule->limit_max = refdbg_dispmax;
}

/* free the allocated members of a detail rule */
void
release_detail_rule (DetailRule * rule)
{
  g_free (rule->inc_types);
  g_free (rule->exc_types);
  g_free (rule->inc_type_name);
  g_free (rule->exc_type_name);
  g_free (rule->inc_objects);
  g_free (rule->exc_objects);

  rule->inc_types = NULL;
  rule->exc_types = NULL;
  rule->inc_type_name = NULL;
  rule->exc_type_name = NULL;
  rule->inc_objects = NULL;
  rule->exc_objects = NULL;
}
