\name{ioprofile}
\title{Produce Individual Ordinal Time Profiles for Plotting}
\usage{
plot(ioprofile(z,curve.type="mean"),nind=1,observed=T,main=NULL,
     xlab=NULL,ylab=NULL,xlim=NULL,ylim=NULL,lty=NULL,pch=NULL,
     add=F,axes=F,bty="n",at=NULL,touch=F,...)
}
\alias{ioprofile}
\alias{ioprofile.default}
\alias{ioprofile.kalordinal}
\alias{plot.ioprofile}
\description{
  \code{ioprofile} is used for plotting individual ordinal profiles over
  time for objects obtained from dynamic models. It produces output for
  plotting recursive undelying means, recursive highest probabilities,
  and recursive cumulative probabilities for individual ordinal time
  profiles from such models.

  See \code{\link{moprofile}} for plotting marginal ordinal profiles.
}
\arguments{
  \item{z}{An object of class \code{kalordinal}, from
    \code{\link{kalord}}.}
  \item{curve.type}{Specifies the type of curves to be plotted. Must
    either be "mean" for recursive underlying means, "probability" for
    recursive highest probabilities, "both" for recursive predictions
    (recursive underlying means and recursive highest probabilities), or
    "cumulative" for recursive cumulative probabilities.}
  \item{nind}{Observation number(s) of individual(s) to be plotted.}
  \item{observed}{If TRUE, adds the corresponding observations to the
    plot. If cumulative curves have been chosen, they are added as a
    subtitle.}
  \item{main}{A main title for the plot.}
  \item{xlab}{A label for the x-axis.}
  \item{ylab}{A label for the y-axis.}
  \item{xlim}{The x limits (min,max) of the plot.}
  \item{ylim}{The y limits (min,max) of the plot.}
  \item{lty}{A vector of integers or character strings specifying the
    line type to be used as the default in plotting lines. For further
    information, see \code{\link{par}}.}
  \item{pch}{A vector of integers or single characters specifying
    symbols to be used as the default in plotting points. For further
    information, see \code{\link{par}}.}
  \item{add}{If TRUE, the graph is added to an existing plot.}
  \item{axes}{If FALSE, axes are not drawn around the plot.}
  \item{bty}{A character string which determined the type of box which
    is drawn about plots. For further information, see
    \code{\link{par}}.}
  \item{at}{The points at which tick-marks are to be drawn. For further
    information, see \code{\link{axis}}.}
  \item{touch}{If TRUE, the x-axis and y-axis will touch each other.}
}
\value{
  \code{ioprofile} returns information ready for plotting by
  \code{plot.ioprofile}.
}
\author{P.J. Lindsey}
\seealso{
  \code{\link{kalord}}, \code{\link{moprofile}},
  \code{\link{plot.ordinal}}, \code{\link{poprofile}}.
}
\examples{
library(ordinal)

#
# Binary data
#
data(cardiac.indiv)

y <- restovec(cardiac.indiv[,1:4],type="ordinal")

cov <- tcctomat(as.matrix(cardiac.indiv[,5:10]))

w <- rmna(y,ccov=cov)

rm(cardiac.indiv,y,cov)

# Time-constant and time-varying covariate with a frailty dependence.
z <- kalord(w,distribution="binary",mu=~age+ren+cop+dia+sex+pmi+times,
            ptvc=c(4.43357,-0.03128,-0.62602,-0.37679,-0.32969,-0.17013,
                   -0.12209,-0.09095),pinit=0.1196,dep="frailty")

# Recursive mean profiles.
par(mfrow=c(2,2))
plot(ioprofile(z,"mean"),nind=1)
plot(ioprofile(z,"mean"),nind=5)
plot(ioprofile(z,"mean"),nind=c(1,5))
plot(ioprofile(z,"mean"),nind=12)
par(mfrow=c(1,1))

# Recursive highest probability profiles.
par(mfrow=c(2,2))
plot(ioprofile(z,"prob"),nind=1)
plot(ioprofile(z,"prob"),nind=5)
plot(ioprofile(z,"prob"),nind=c(1,5))
plot(ioprofile(z,"prob"),nind=12)
par(mfrow=c(1,1))

# Recursive predicted profiles.
par(mfrow=c(2,3))
plot(ioprofile(z,"both"),nind=1)
plot(ioprofile(z,"both"),nind=5)
plot(ioprofile(z,"both"),nind=12)
plot(ioprofile(z,"both"),nind=c(1,5,12),add=T)
par(mfrow=c(1,1))

# Recursive cumulative probability profiles.
par(mfrow=c(2,2))
plot(ioprofile(z,"cum"),nind=1)
plot(ioprofile(z,"cum"),nind=5)
plot(ioprofile(z,"cum"),nind=c(1,5),add=T)
par(mfrow=c(1,1))

rm(w,z)

#
# Ordinal data
#
data(tmi2)

y <- restovec(tmi2[,1:4],type="ordinal")

cov <- tcctomat(tmi2[,5],name="distance")

w <- rmna(y,ccov=cov)

rm(tmi2,y,cov)

# Proportional-odds model with time-constant covariate with a Markov dependence.
z <- kalord(w,distribution="proportional-odds",ccov=~distance,
            preg=c(-1.89,11.652,-0.199),pinit=3.111,pdep=0.217,dep="Markov")

# Recursive mean profiles.
par(mfrow=c(2,2))
plot(ioprofile(z,"mean"),nind=1)
plot(ioprofile(z,"mean"),nind=268)
plot(ioprofile(z,"mean"),nind=c(1,268))
plot(ioprofile(z,"mean"),nind=117)
par(mfrow=c(1,1))

# Recursive highest probability profiles.
par(mfrow=c(2,2))
plot(ioprofile(z,"prob"),nind=1)
plot(ioprofile(z,"prob"),nind=268)
plot(ioprofile(z,"prob"),nind=c(1,268))
plot(ioprofile(z,"prob"),nind=117)
par(mfrow=c(1,1))

# Recursive predicted profiles.
par(mfrow=c(2,2))
plot(ioprofile(z,"both"),nind=120)
plot(ioprofile(z,"both"),nind=268)
plot(ioprofile(z,"both"),nind=c(1,117),add=T)
par(mfrow=c(1,1))

# Recursive cumulative probability profiles.
par(mfrow=c(2,2))
plot(ioprofile(z,"cum"),nind=1)
plot(ioprofile(z,"cum"),nind=268)
plot(ioprofile(z,"cum"),nind=c(117,120),add=T)
par(mfrow=c(1,1))

rm(w,z)
}
\keyword{hplot}
