#include <stdlib.h>
#include "ilbc/iLBC_define.h"
#include "ilbc/iLBC_encode.h"
#include "ilbc/iLBC_decode.h"
#include "phcodec.h"

#include "phmedia.h"
/* */

#include <../src/eXosip2.h>

static int ph_ilbc_encode(void *ctx, const void *src, int srcsize, void *dst, int dstsize);
static int ph_ilbc_decode(void *ctx, const void *src, int srcsize, void *dst, int dstsize);
static void *ph_ilbc_dec_init(void *dummy);
static void *ph_ilbc_enc_init(void *dummy);
static void ph_ilbc_dec_cleanup(void *ctx);
static void ph_ilbc_enc_cleanup(void *ctx);

static int ph_ilbc_get_frame_size(void *dummy)
{
	int packetization = atoi(DEFAULT_PTIME_VALUE);
	char* payload_mode = NULL;
	struct ph_media_payload_s* p_media_payload = NULL;
	
	p_media_payload = (struct ph_media_payload_s*)dummy;
	
	if (p_media_payload != NULL)
	{
		if ( (p_media_payload->fmtp != NULL) && 
				(strstr(p_media_payload->fmtp, "111") == p_media_payload->fmtp) && 
				((payload_mode = strstr(p_media_payload->fmtp, " mode=")) != NULL) )
		{
			payload_mode += 6;
			packetization = atoi(payload_mode);
			
			// Update payload's "ptime" with the desired "mode" that have priority.
			p_media_payload->ptime = packetization;
			p_media_payload->mode = packetization;
		}
		else if (p_media_payload->ptime != 0)
		{
			packetization = p_media_payload->ptime;
		}
	}
	
	return packetization;
}

static void *ph_ilbc_dec_init(void *dummy)
{
    iLBC_Dec_Inst_t *ctx;
	int packetization;

    ctx = malloc(sizeof(*ctx));

	packetization = ph_ilbc_get_frame_size(dummy);
	initDecode(ctx, packetization, 1);

    return ctx;
}

static void *ph_ilbc_enc_init(void *dummy)
{
    iLBC_Enc_Inst_t *ctx;
	int packetization;

    ctx = malloc(sizeof(*ctx));

	packetization = ph_ilbc_get_frame_size(dummy);
	initEncode(ctx, packetization);

    return ctx;
}

static void ph_ilbc_dec_cleanup(void *ctx)
{
  free(ctx);
}

static void ph_ilbc_enc_cleanup(void *ctx)
{
  free(ctx);
}

static int ph_ilbc_decode(void *ctx, const void *src, int srcsize, void *dst, int dstsize)
{
  int k;
  float decflt[BLOCKL_MAX], tmp;
  short *decshrt = (short*) dst;
  iLBC_Dec_Inst_t *dec = (iLBC_Dec_Inst_t *) ctx;

  iLBC_decode(decflt, (unsigned char *)src, dec, 1); 

  for (k=0; k< dec->blockl; k++)
  {
      tmp=decflt[k];
      if (tmp<MIN_SAMPLE)
      {
        tmp=MIN_SAMPLE;
      }
      else if (tmp>MAX_SAMPLE)
      {
        tmp=MAX_SAMPLE;
      }
      decshrt[k] = (short) tmp;
  }
  return (dec->blockl*2); 
}

static int ph_ilbc_encode(void *ctx, const void *src, int srcsize, void *dst, int dstsize)
{
  float tmp[BLOCKL_MAX];
  short *indata = (short *) src;
  int k;
  iLBC_Enc_Inst_t *enc = (iLBC_Enc_Inst_t *) ctx;

  for (k=0; k<enc->blockl; k++)
  {
    tmp[k] = (float)indata[k];
  }

  /* do the actual encoding */
  iLBC_encode((unsigned char *)dst, tmp, enc);

  return (enc->no_of_bytes);
}

#define				MAX_ILBC_MODE_PARAM_VALUE			"30"

static char* 
ph_ilbc_negotiate_attrib(char *attr_name, char *local_attr_value, char *remote_attr_value)
{
  if (strcmp(attr_name, "fmtp") == 0)
  {
    int local_mode = atoi(MAX_ILBC_MODE_PARAM_VALUE);	// Max acceptable "mode" for iLBC.
    int remote_mode = 0;
    char* local_mode_str = NULL;
    char* remote_mode_str = NULL;
    char* final_mode_str = NULL;
    
    if (remote_attr_value == NULL)
    {
      return NULL;
    }
    else
	{
      remote_mode_str = strstr(remote_attr_value, " mode=");
    }
    
    if (local_attr_value != NULL)
    {
      local_mode_str = strstr(local_attr_value, " mode=");
    }
    
    if (local_mode_str != NULL)
    {
      sscanf(local_mode_str, " mode=%d", &local_mode);
    }
    
    if (remote_mode_str != NULL)
    {
      sscanf(remote_mode_str, " mode=%d", &remote_mode);
    }
    
    if (remote_mode >= local_mode)
    {
      return osip_strdup(remote_attr_value);
    }
    else if ( (local_attr_value == NULL) && (remote_mode_str != NULL) )
    {
      int mode_value_index = (remote_mode_str - remote_attr_value) + 6;
      
      final_mode_str = osip_strdup(remote_attr_value);
      final_mode_str[mode_value_index] = '\0';
      strncat(final_mode_str, MAX_ILBC_MODE_PARAM_VALUE, 2);
      
      return final_mode_str;
    }
  }
  
  return NULL;
}

int ph_ilbc_encoded_frame_size_get(int packetization)
{
	// iLBC supports only packetization of 20ms and 30ms.
	
	if (packetization == 20)
	{
		return NO_OF_BYTES_20MS;
	}
	else if (packetization == 30)
	{
		return NO_OF_BYTES_30MS;
	}
	
	// This should not happen.
	return 0;
}

static phcodec_t ilbc_codec =
{
  "ILBC", 8000, NO_OF_BYTES_20MS, BLOCKL_20MS*2, ph_ilbc_enc_init, ph_ilbc_dec_init, ph_ilbc_enc_cleanup, ph_ilbc_dec_cleanup, 
  ph_ilbc_encode, ph_ilbc_decode, NULL, ph_ilbc_negotiate_attrib, ph_ilbc_encoded_frame_size_get
};


#ifdef WIN32
#define DLLEXPORT __declspec(dllexport)
#else
#define DLLEXPORT
#endif


DLLEXPORT int ph_codec_plugin_init(void (*codec_register)(phcodec_t *))
{
  codec_register(&ilbc_codec);

  return 0;
}
