/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "CQuteCom.h"

#include <presentation/PFactory.h>
#include <presentation/PQuteCom.h>

#include <control/profile/CUserProfileHandler.h>
#include <control/dtmf/CDtmfThemeManager.h>

#include <model/commandserver/ContactInfo.h>
#include <model/commandserver/CommandServer.h>
#include <model/QuteCom.h>
#include <model/dtmf/DtmfThemeManager.h>

#include <thread/ThreadEvent.h>
#include <util/Logger.h>

CQuteCom::CQuteCom(QuteCom & qutecomPhone, bool runInBackground)
	: _qutecomPhone(qutecomPhone) {

	_pQuteCom = PFactory::getFactory().createPresentationQuteCom(*this, runInBackground);

	_cUserProfileHandler = new CUserProfileHandler(_qutecomPhone.getUserProfileHandler(), *this);

	_qutecomPhone.initFinishedEvent += boost::bind(&CQuteCom::initFinishedEventHandler, this, _1);
	_qutecomPhone.exitEvent += boost::bind(&CQuteCom::exitEventHandler, this);

	//DTMFThemeManager
	_qutecomPhone.dtmfThemeManagerCreatedEvent += boost::bind(&CQuteCom::dtmfThemeManagerCreatedEventHandler, this, _1, _2);

	CommandServer & commandServer = CommandServer::getInstance(_qutecomPhone);
	commandServer.showAddContactEvent += boost::bind(&CQuteCom::showAddContactEventHandler, this, _1);
	commandServer.bringMainWindowToFrontEvent += boost::bind(&CQuteCom::bringMainWindowToFrontEventHandler, this);

	// Start the model thread
	typedef ThreadEvent0<void ()> MyThreadEvent;
	MyThreadEvent * event = new MyThreadEvent(boost::bind(&CQuteCom::start, this));
	PFactory::postEvent(event);
}

CQuteCom::~CQuteCom() {
	/*if (_pQuteCom) {
		delete _pQuteCom;
	}*/
}

void CQuteCom::initPresentationThreadSafe() {
}

Presentation * CQuteCom::getPresentation() const {
	return _pQuteCom;
}

CQuteCom & CQuteCom::getCQuteCom() const {
	return (CQuteCom &) *this;
}

CUserProfileHandler & CQuteCom::getCUserProfileHandler() const {
	return *_cUserProfileHandler;
}

QuteCom & CQuteCom::getQuteCom() const {
	return _qutecomPhone;
}

void CQuteCom::start() {
	_qutecomPhone.start();
}

void CQuteCom::terminate() {
	_qutecomPhone.prepareToTerminate();
}

void CQuteCom::dtmfThemeManagerCreatedEventHandler(QuteCom & sender, DtmfThemeManager & dtmfThemeManager) {
	_cDtmfThemeManager = new CDtmfThemeManager(dtmfThemeManager, *this);
}

void CQuteCom::initFinishedEventHandler(QuteCom & sender) {
	LOG_DEBUG("QuteCom::init() finished");
}

void CQuteCom::exitEventHandler() {
	typedef ThreadEvent0<void ()> MyThreadEvent;
	MyThreadEvent * event = new MyThreadEvent(boost::bind(&CQuteCom::exitEventHandlerThreadSafe, this));
	PFactory::postEvent(event);
}

void CQuteCom::exitEventHandlerThreadSafe() {
	_qutecomPhone.terminate();
	if (_pQuteCom) {
		_pQuteCom->exitEvent();
	}
}

void CQuteCom::showAddContactEventHandler(ContactInfo contactInfo) {
	typedef ThreadEvent1<void (ContactInfo), ContactInfo> MyThreadEvent;
	MyThreadEvent * event = new MyThreadEvent(boost::bind(&CQuteCom::showAddContactEventHandlerThreadSafe, this, _1), contactInfo);
	PFactory::postEvent(event);
}

void CQuteCom::showAddContactEventHandlerThreadSafe(ContactInfo contactInfo) {
	if (_pQuteCom) {
		_pQuteCom->showAddContact(contactInfo);
	}
}

void CQuteCom::bringMainWindowToFrontEventHandler() {
	typedef ThreadEvent0<void ()> MyThreadEvent;
	MyThreadEvent * event = new MyThreadEvent(boost::bind(&CQuteCom::bringMainWindowToFrontEventHandlerThreadSafe, this));
	PFactory::postEvent(event);
}

void CQuteCom::bringMainWindowToFrontEventHandlerThreadSafe() {
	if (_pQuteCom) {
		_pQuteCom->bringMainWindowToFront();
	}
}

void CQuteCom::enableSerialization(bool enable) {
	_qutecomPhone.enableSerialization(enable);
}

CDtmfThemeManager & CQuteCom::getCDtmfThemeManager() const {
	return *_cDtmfThemeManager;
}
