/* Copyright (C) 2004 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* This file provides some helper-types needed by QTads.
 */

#ifndef QTADSTYPES_H
#define QTADSTYPES_H

#include "config.h"

#include <qstring.h>
#include <qevent.h>

/* Extends QKeyEvent to provide information about whether an event
 * timed out or not.  (This is used for timed input events, as provided
 * by the OS-layer function os_get_event().)  In addition, provide info
 * about whether the event is the result of an EOF situation (this
 * happens when the game is about to be terminated).
 */
class QTadsKeyEvent: public QKeyEvent {
  private:
	bool fTimedOut;
	bool fEOF;

  public:
	QTadsKeyEvent( QEvent::Type type, int key, int ascii, int state,
		       const QString& text = QString::null, bool autorep = false,
		       ushort count = 1, bool timedOut = false, bool isEOF = false )
	: QKeyEvent(type, key, ascii, state, text, autorep, count), fTimedOut(timedOut),
	  fEOF(isEOF)
	{}

	QTadsKeyEvent( bool timedOut, bool isEOF = false )
	: QKeyEvent(QEvent::KeyPress, 0, 0, 0), fTimedOut(timedOut), fEOF(isEOF)
	{}

	QTadsKeyEvent( const QKeyEvent& e )
	: QKeyEvent(e), fTimedOut(false), fEOF(false)
	{}

	bool
	hasTimedOut()
	{
		return this->fTimedOut;
	}

	bool
	isEOF()
	{
		return this->fEOF;
	}
};


/* Stores information about how characters appear on screen.
 */
struct QTadsTextFormat {
	QTadsTextFormat()
	: high(false), italics(false)
	{}

	bool
	operator ==( const QTadsTextFormat& fmt )
	{
		return (this->high == fmt.high and this->italics == fmt.italics
		        and this->color == fmt.color);
	}

	bool
	operator !=( const QTadsTextFormat& fmt )
	{
		return (not this->operator ==(fmt));
	}

	bool high;    // Are the characters highlighted?
	bool italics; // Are the characters italized?
	QColor color; // The text's color.
};


/* Holds a string together with information about its appearance on the
 * screen.
 */
struct QTadsFormattedString {
	QTadsFormattedString( const QString& qStr, const QTadsTextFormat& fmt )
	: s(qStr), f(fmt)
	{}

	QString s;
	QTadsTextFormat f;
};

#endif // QTADSTYPES_H
