//
// C++ Implementation:
//
// Description:
//
//
// Author: Sebastian Holtermann <sebholt@xwmw.org>, (C) 2011
//
// Copyright: See COPYING file that comes with this distribution
//
//

#include "switcher.hpp"

#include <config.hpp>
#include <mview/mv_base.hpp>
#include <mview/mv_mixer_simple.hpp>
#include <mview/mv_mixer_ctl.hpp>
#include <mview/mv_info.hpp>
#include <mview/switcher_setup.hpp>

#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QTabBar>
#include <QFile>
#include <QEvent>
#include <QMenu>
#include <QContextMenuEvent>

#include <iostream>


namespace MView
{


Switcher::Switcher (
	QWidget * parent_n ) :
QWidget ( parent_n ),
_mixer_setup ( 0 ),
_view ( 0 ),
_view_setup ( 0 )
{
	_vtype_names[0] = tr ( "&Simple mixer" );
	_vtype_names[1] = tr ( "&Element mixer" );
	_vtype_names[2] = tr ( "&Control info" );

	// Tab bar
	{
		_tab_bar = new QTabBar;
		_tab_bar->hide();
		for ( unsigned int ii=0; ii < 3; ++ii ) {
			_tab_bar->addTab ( _vtype_names[ii] );
		}
		_tab_bar->installEventFilter ( this );

		connect ( _tab_bar, SIGNAL ( currentChanged ( int ) ),
			this, SLOT ( select_view_type_int ( int ) ) );
	}

	// Root layout
	QVBoxLayout * lay_box ( new QVBoxLayout );
	lay_box->addWidget ( _tab_bar );
	setLayout ( lay_box );
}


Switcher::~Switcher ( )
{
}


void
Switcher::set_mixer_setup (
	::MView::Switcher_Setup * setup_n )
{
	if ( _mixer_setup != 0 ) {
		clear_view();
	}

	_mixer_setup = setup_n;

	if ( _mixer_setup != 0 ) {
		select_view_type ( _mixer_setup->view_type );
	}
}



void
Switcher::select_snd_ctl (
	const ::QSnd::Control_Address & ctl_n )
{
	if ( ( _mixer_setup == 0 ) || ( _view == 0 ) ) {
		return;
	}

	if ( ctl_n != _mixer_setup->mixer_dev.ctl_addr ) {
		_view->set_mdev_setup ( 0 );
		_mixer_setup->mixer_dev.ctl_addr = ctl_n;
		_view->set_mdev_setup ( &_mixer_setup->mixer_dev );
	}
}


void
Switcher::select_view_simple_mixer ( )
{
	select_view_type ( 0 );
}


void
Switcher::select_view_element_mixer ( )
{
	select_view_type ( 1 );
}


void
Switcher::select_view_control_info ( )
{
	select_view_type ( 2 );
}


void
Switcher::select_view_type_int (
	int type_n )
{
	select_view_type ( (unsigned int)type_n );
}


void
Switcher::select_view_type (
	unsigned int type_n )
{
	if ( _mixer_setup == 0 ) {
		return;
	}
	if ( type_n >= num_view_types() ) {
		return;
	}

	bool changed ( false );
	if ( _mixer_setup->view_type != type_n ) {
		changed = true;
		clear_view();
		_mixer_setup->view_type = type_n;
	}

	if ( _view == 0 ) {
		changed = true;
		create_view();
	}

	update_tab_bar_vis();

	if ( changed ) {
		emit sig_view_type_changed();
	}
}



void
Switcher::clear_view ( )
{
	if ( _view != 0 ) {
		delete _view;
		_view = 0;
		_view_setup = 0;
	}
}


void
Switcher::create_view ( )
{
	if ( _mixer_setup == 0 ) {
		return;
	}

	if ( _mixer_setup->view_type == 0 ) {

		_view =	new ::MView::MV_Mixer_Simple ( this );
		_view_setup = &_mixer_setup->mv_simple;

	} else if ( _mixer_setup->view_type == 1 ) {

		_view =	new ::MView::MV_Mixer_CTL ( this );
		_view_setup = &_mixer_setup->mv_ctl;

	} else if ( _mixer_setup->view_type == 2 ) {

		_view =	new ::MView::MV_Info ( this );
		_view_setup = &_mixer_setup->mv_info;

	}

	if ( _view != 0 ) {
		_view->set_mdev_setup ( &_mixer_setup->mixer_dev );
		_view->set_inputs_setup ( &_mixer_setup->inputs );
		_view->set_view_setup ( _view_setup );

		connect ( _view, SIGNAL ( sig_mdev_reload_request() ),
			this, SLOT ( reload_mdev_setup() ), Qt::QueuedConnection );

		layout()->addWidget ( _view );
		_view->setFocus();
	}
}


void
Switcher::show_vtype_select (
	bool flag_n )
{
	if ( _mixer_setup == 0 ) {
		return;
	}

	if ( _mixer_setup->show_vtype_select != flag_n ) {
		_mixer_setup->show_vtype_select = flag_n;
		if ( flag_n || ( _mixer_setup->view_type == 0 ) ) {
			update_tab_bar_vis();
		} else {
			select_view_type ( 0 );
		}

		emit sig_show_vtype_select ( flag_n );
	}

}


void
Switcher::toggle_vtype_select ( )
{
	if ( _mixer_setup == 0 ) {
		return;
	}

	show_vtype_select ( !_mixer_setup->show_vtype_select );
}


void
Switcher::update_tab_bar_vis ( )
{
	if ( _mixer_setup == 0 ) {
		return;
	}

	// Adjust tab bar index
	if ( (int)_mixer_setup->view_type != _tab_bar->currentIndex() ) {
		_tab_bar->setCurrentIndex ( _mixer_setup->view_type );
	}

	// Show tab bar on demand
	if ( ( _mixer_setup->view_type != 0 ) &&
		!_mixer_setup->show_vtype_select )
	{
		show_vtype_select ( true );
		return;
	}

	_tab_bar->setVisible ( _mixer_setup->show_vtype_select );
}


void
Switcher::reload_mdev_setup ( )
{
	if ( ( _view == 0 ) || ( _mixer_setup == 0 ) ) {
		return;
	}

	_view->set_mdev_setup ( 0 );
	_view->set_mdev_setup ( &_mixer_setup->mixer_dev );
}


void
Switcher::reload_inputs_setup ( )
{
	if ( ( _view == 0 ) || ( _mixer_setup == 0 ) ) {
		return;
	}

	_view->set_inputs_setup ( 0 );
	_view->set_inputs_setup ( &_mixer_setup->inputs );
}


void
Switcher::reload_view_setup ( )
{
	if ( ( _view == 0 ) || ( _view_setup == 0 ) ) {
		return;
	}

	_view->set_view_setup ( 0 );
	_view->set_view_setup ( _view_setup );
}


bool
Switcher::eventFilter (
	QObject * obj_n,
	QEvent * event_n )
{
	bool res ( false );
	if ( obj_n == _tab_bar ) {
		if ( event_n->type() == QEvent::ContextMenu ) {
			// View selection tab bar context menu
			QContextMenuEvent * ev_cm (
				static_cast < QContextMenuEvent * > ( event_n ) );

			QMenu menu;
			QAction * act_close = menu.addAction (
				tr ( "&Close view type selection" ) );
			connect ( act_close, SIGNAL ( triggered ( bool ) ),
				this, SLOT ( show_vtype_select ( bool ) ) );
			menu.exec ( ev_cm->globalPos() );

			res = true;
		}
	}
	return res;
}


} // End of namespace
