#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2011  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: widget-set.c
# Original copyright and license:
#/*
#    GNUbik -- A 3 dimensional magic cube game.
#    Copyright (C) 2003, 2004  John Darrington
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 3 of the License,  or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#*/


import os

import gobject
import gtk
from gtk import gdk

from .debug import *
from . import config
from . import dialogs
from . import colour_sel
from . import plugin_hooks
import state

from . import glarea
from .confstore import confstore


class Application (object):
    def __init__(self):
        # create cube state
        self.solved = False
        self.current_movement = state.CurrentMovement()
        
        #
        confstore.init_notify()
        self.plugin_helper = plugin_hooks.PluginHelper(self)
        
        # UI
        ui_file = os.path.join(config.UI_DIR, 'pybik.ui')
        self.ui_xml = gtk.Builder()
        self.ui_xml.props.translation_domain = config.PACKAGE
        self.ui_xml.add_from_file(ui_file)
        debug('Connect signals:', ui_file)
        missing = self.ui_xml.connect_signals(self)
        if missing:
            debug(' ', len(missing), 'missing handlers:')
            for m in missing:
                debug('    def', m+'(self, unused_widget, *unused_args):')
                debug('        pass')
        else:
            debug('  no missing handlers')
        
        self.window = self.ui_xml.get_object('window_main')
        self.statusbar = self.ui_xml.get_object('statusbar_main')
        self.entry_formula = self.ui_xml.get_object('entry_formula')
        
        self.window.set_icon_from_file(config.IMAGE_FILE)
        
        self.window.connect("delete-event", self.gtk_main_quit)
        
        self.create_menu()
        self.create_sidepane()
        
        self.cube_area = glarea.CubeArea()
        drawingarea_cube = self.ui_xml.get_object('drawingarea_cube')
        self.cube_area.init(drawingarea_cube, self)
        self.cube_area.connect('texture-changed', self.on_texture_changed)
        
    def on_texture_changed(self, unused_widget):
        if colour_sel.ColorSelectDialog.this_dlg:
            colour_sel.ColorSelectDialog.this_dlg.redraw_swatches()
            
    def create_menu(self):
        ui_def = ('''
            <ui>
                <menubar name="menubar1">
                    <menu action="menu_game">
                        <menuitem name="menu_new_random" action="action_new" />
                        <menuitem name="menu_new_solved" action="action_new_solved" />
                        <separator/>
                        <menuitem name="menu_quit" action="action_quit" />
                    </menu>
                    <menu action="menu_edit">
                        <menuitem name="menu_colors" action="action_colors" />
                        <menuitem name="menu_preferences" action="action_preferences" />
                    </menu>
                    <menu action="menu_view">
                        <menuitem name="menu_play_toolbar" action="toggleaction_play_toolbar" />
                        <menuitem name="menu_statusbar" action="toggleaction_statusbar" />
                    </menu>
                    <menu action="menu_help">
                        <menuitem name="menu_info" action="action_info" />
                    </menu>
                </menubar>
                <toolbar name="toolbar_play">
                    <toolitem name="toolbutton_rewind"  action="action_rewind" />
                    <toolitem name="toolbutton_previous" action="action_previous" />
                    <toolitem name="toolbutton_stop"    action="action_stop" />
                    <toolitem name="toolbutton_play"    action="action_play" />
                    <toolitem name="toolbutton_next"    action="action_next" />
                    <toolitem name="toolbutton_forward" action="action_forward" />
                    <separator />
                    <toolitem name="toolbutton_add_mark" action="action_add_mark" />
                    <toolitem name="toolbutton_remove_mark" action="action_remove_mark" />
                </toolbar>
                <accelerator action="action_rewind" />
                <accelerator action="action_forward" />
                <accelerator action="action_focus_formula" />
            </ui>
        ''')
        
        self.main_actions = gtk.ActionGroup('main_actions')
        self.play_actions = gtk.ActionGroup('play_actions')
        #name,           stock_id,  label,          accel,      tooltip,    callback
        self.main_actions.add_actions([
        ("menu_game",    None,      _('_Game')),
        ("menu_edit",    None,      _('_Edit')),
        ("menu_view",    None,      _('_View')),
        ("menu_help",    None,      _('_Help')),
        
        ("action_new",  'gtk-new',  _("_New random"),  None,
                                    _('New random cube'),   self.on_action_new_activate),
        ("action_new_solved",None,  _("New _solved"),  '<Ctrl><Shift>n',
                                    _('New solved cube'),   self.on_action_new_solved_activate),
        ("action_quit", "gtk-quit", None,           None,       None,       self.gtk_main_quit),
        
        ("action_preferences","gtk-preferences",None,None,      None,
                                                            self.on_action_preferences_activate),
        ("action_colors","gtk-select-color",None,   None,       None,   self.on_action_colors_activate),
        ("action_info", 'gtk-about',None,           None,       None,   self.on_action_info_activate),
        
        ('action_focus_formula', None, None,        '<Ctrl>l',  None,   self.on_action_focus_formula),
        ])
        
        #name,          stock_id,   label,          accelerator,tooltip,   callback,   isactive
        self.main_actions.add_toggle_actions([
        ("toggleaction_play_toolbar",None,_("_Play Toolbar"),None,
                                    _("Play Toolbar"), self.on_toggleaction_play_toolbar_toggled,   True),
        ("toggleaction_statusbar",None,_("_Status Bar"),None,
                                    _("Status Bar"),   self.on_toggleaction_statusbar_toggled,      True),
        ])
        
        self.play_actions.add_actions([
        #name,              stock_id,               label,      accel,      tooltip,    callback
        ("action_rewind",   "gtk-media-rewind",     None,       '<Ctrl>r',
                    _("Go to the previous mark (or the beginning) of the sequence of moves"),
                    self.on_action_rewind_activate),
        ("action_previous", "gtk-media-previous",   None,       None,
                    _("Make one step backwards"),              self.on_action_previous_activate),
        ("action_stop",     "gtk-media-stop",       None,       None,
                    _("Stop running the sequence of moves"),   self.on_action_stop_activate),
        ("action_play",     "gtk-media-play",       None,       None,
                    _("Run forward through the sequence of moves"),    self.on_action_play_activate),
        ("action_next",     "gtk-media-next",       None,       None,
                    _("Make one step forwards"),               self.on_action_next_activate),
        ("action_forward",  "gtk-media-forward",    None,       '<Ctrl><Shift>r',
                    _("Go to the next mark (or the end) of the sequence of moves"),
                    self.on_action_forward_activate),
        ("action_add_mark", "gtk-add",              _("Add mark"),     None,
                    _("Mark the current place in the sequence of moves"),
                    self.on_action_add_mark_activate),
        ("action_remove_mark",  "gtk-remove",       _('Remove mark'),   None,
                    _("Remove the mark at the current place in the sequence of moves"),
                    self.on_action_remove_mark_activate),
        ])
        
        self.uimanager = gtk.UIManager()
        self.uimanager.insert_action_group(self.main_actions)
        self.uimanager.insert_action_group(self.play_actions)
        self.uimanager.add_ui_from_string(ui_def)
        
        menubar = self.uimanager.get_widget('/menubar1')
        vbox = self.ui_xml.get_object('vbox1')
        vbox.pack_start(menubar, fill=False, expand=False)
        vbox.reorder_child(menubar, 0)
        
        self.play_toolbar = self.uimanager.get_widget('/toolbar_play')
        self.play_toolbar.get_accessible().set_name('ToolbarPlay')
        vbox = self.ui_xml.get_object('vbox2')
        vbox.pack_start(self.play_toolbar, fill=False, expand=False)
        vbox.reorder_child(self.play_toolbar, 0)
        
        accel = self.uimanager.get_accel_group()
        self.window.add_accel_group(accel)
    
        self.playbarstate = PlaybarState(self)
        self.playbarstate.empty()
        
    @debug_func
    def create_sidepane(self):
        # Get the menu structure asked for by the collection of plugins.
        list_ = self.plugin_helper.startup_plugins()
        
        treeview = self.ui_xml.get_object('treeview_scripts')
        treestore = gtk.TreeStore(gobject.TYPE_STRING, gobject.TYPE_PYOBJECT)
        treeview.set_model(treestore)
        
        column = gtk.TreeViewColumn(_('Name'), gtk.CellRendererText(), text=0)
        column.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
        column.set_fixed_width(200)
        treeview.append_column(column)
        
        # add the items
        tree_iters = {'':None}
        for func_name, func in list_:
            p = func_name.split('/', 4)
            for i in xrange(1, 5):
                p_i = p[:i]
                p_is = '/'.join(p_i)
                p_i_p = p[:i-1]
                p_is_p = '/'.join(p_i_p)
                name = p_i[-1]
                if p_is not in tree_iters:
                    #debug('|'.join((name, p_is, func_name)))
                    ti = treestore.append(tree_iters[p_is_p], [name, func if p_is==func_name else None])
                    tree_iters[p_is] = ti
        treeview.connect('row-activated', self.on_run_script)
        #treeview.expand_all()
        treeview.show_all()

    def is_animating(self):
        return self.cube_area.animation_in_progress
        
    def on_run_script(self, treeview, path, unused_view_column):
        if self.is_animating():
            return
        unused_name, func = treeview.get_model()[path]
        if self.plugin_helper.call(func):
            self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
            self.update_ui()
            move_data = self.current_movement.request_play()
            if move_data:
                self.cube_area.animate_rotation(self, move_data)
            else:
                self.cube_area.render()
                
    @debug_func
    def start_main_loop(self):
        self.window.show()
        gtk.quit_add(0, self.on_quit)
        gtk.main()
        
    def first_new_game(self, dimension, solved):
        if solved:
            self.solved = True
            
        if self.solved:
            if dimension is None:
                dimension = confstore.dimension
            self.current_movement.set_solved(dimension)
            self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        else:
            if dimension is None:
                self.current_movement.set_saved_game(
                                            confstore.dimension,
                                            confstore.saved_state,
                                            confstore.saved_moves,
                                            confstore.saved_pos)
                self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
            else:
                self.current_movement.set_random(dimension)
                self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
                
        self.cube_area.render()
        self.update_ui(first=True)
        
    @debug_func
    def new_game(self, new_size, solved):
        assert not self.is_animating()
        
        if solved is not None:
            self.solved = solved
        
        if new_size is not None:
            self.cube_area.re_initialize()
            if new_size <= 0:
                new_size = self.current_movement.current_cube_state.dimension
                
        if self.solved:
            self.current_movement.set_solved(new_size)
        else:
            self.current_movement.set_random(new_size)
        self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        self.cube_area.render()
        self.current_movement.all_moves.reset()
        self.update_ui()
        
        return False
    
    # Request that the game be restarted
    # If data is non zero,  then all it's data will be reallocated
    @debug_func
    def request_new_game(self, unused_widget=None, new_size=None, solved=None):
        self.current_movement.request_stop()
        if self.is_animating():
            self.current_movement.request_abort()
        gobject.idle_add(self.new_game, new_size, solved)
    
    
    # Action handlers
    @debug_func
    def on_quit(self):
        (unused_dimension, confstore.saved_state, confstore.saved_moves,
                    confstore.saved_pos) = self.current_movement.get_saved_game()
        return 0
    gtk_main_quit = gtk.main_quit
    def on_action_new_activate(self, *unused_args):
        self.request_new_game(solved=False)
    def on_action_new_solved_activate(self, *unused_args):
        self.request_new_game(solved=True)
    def on_action_colors_activate(self, unused_widget):
        colour_sel.open_ColorSelectDialog(self)
    def on_action_preferences_activate(self, unused_widget):
        dialogs.preferencesdialog(self)
    def on_toggleaction_play_toolbar_toggled(self, widget):
        self.play_toolbar.props.visible = widget.props.active
    def on_toggleaction_statusbar_toggled(self, widget):
        self.statusbar.props.visible = widget.props.active
    def on_action_info_activate(self, unused_widget):
        dialogs.show_about(self.window)
        
    def on_action_focus_formula(self, *unused_args):
        self.entry_formula.grab_focus()
        
    def on_action_rewind_activate(self, unused_widget):
        if not self.is_animating():
            needs_update = self.current_movement.do_rewind_to_mark()
            self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
            if needs_update:
                #TODO: render updates internal cube state in at idle time
                # maybe that is the reason for the following bug
                # internal state should applied before gobject.idle_add
                # this is also for do_fast_forward
                self.cube_area.render()
                #FIXME: sometimes 'rotated center' is shown by mistake instead of 'solved'
                gobject.idle_add(self.update_ui)
    def on_action_previous_activate(self, unused_widget):
        if not self.is_animating():
            move_data = self.current_movement.request_back()
            if move_data:
                self.cube_area.animate_rotation(self, move_data)
    def on_action_stop_activate(self, unused_widget):
        self.current_movement.request_stop()
    def on_action_play_activate(self, unused_widget):
        if not self.is_animating():
            move_data = self.current_movement.request_play()
            if move_data:
                self.cube_area.animate_rotation(self, move_data)
    def on_action_next_activate(self, unused_widget):
        if not self.is_animating():
            move_data = self.current_movement.request_next()
            if move_data:
                self.cube_area.animate_rotation(self, move_data)
    def on_action_forward_activate(self, unused_widget):
        self.current_movement.do_fast_forward()
        self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        self.cube_area.render()
        gobject.idle_add(self.update_ui)
    def on_action_add_mark_activate(self, unused_widget):
        self.current_movement.request_mark_move_queue(True)
        self.update_ui()
    def on_action_remove_mark_activate(self, unused_widget):
        self.current_movement.request_mark_move_queue(False)
        self.update_ui()
    
    #Callbacks for PreferencesDialog
    def on_button_size_reset_activate(self, unused_widget):
        del confstore.dimension
    def on_button_animspeed_reset_activate(self, unused_widget):
        del confstore.frameQty
    def on_button_lighting_reset_activate(self, unused_widget):
        del confstore.lighting
    
    @debug_func
    def on_colour_select(self, unused_widget):
        colour_sel.open_ColorSelectDialog(self)
    def on_button_color_reset_activate(self, unused_widget):
        i = colour_sel.ColorSelectDialog.this_dlg.active_face[0]
        del confstore.colors[i].color
    def on_button_pattern_reset_activate(self, unused_widget):
        i = colour_sel.ColorSelectDialog.this_dlg.active_face[0]
        del confstore.colors[i].pattern
    def on_button_image_reset_activate(self, unused_widget):
        i = colour_sel.ColorSelectDialog.this_dlg.active_face[0]
        del confstore.colors[i].facetype
        del confstore.colors[i].imagefile
        del confstore.colors[i].imagemode

    
    # Callbacks for editing formulas
    def on_entry_formula_activate(self, unused_widget=None):
        # while animating leads to a invalid cube
        if self.is_animating():
            #TODO: Better: stop animating and proceed
            return
        code = self.entry_formula.get_text()
        pos = self.entry_formula.get_position()
        self.current_movement.set_from_formula(code, pos)
        self.cube_area.apply_to_glmodel(self.current_movement.current_cube_state)
        self.cube_area.render()
        gobject.idle_add(self.update_ui)
        
    def on_entry_formula_icon_press(self, unused_widget, icon_info, event):
        # while animating leads to a invalid cube
        if self.is_animating():
            #TODO: Better: stop animating and proceed
            return
        if event.button==1 and event.type==gdk.BUTTON_PRESS:
            if icon_info.value_nick == 'primary':
                #debug('activate')
                return self.on_entry_formula_activate()
            if icon_info.value_nick == 'secondary':
                #debug('clear')
                self.entry_formula.set_text('')
                self.on_entry_formula_activate()
        
    def set_entry_formula(self, code, pos, unused_markup):
        #debug('markup',markup)
        self.entry_formula.set_text(code)
        self.entry_formula.set_position(pos)
        #TODO: seems, that gtk.Entry doesn't work with pango markup, use one-line gtk.TextView?
        
    def update_statusbar(self, first=False):
        '''This function must be called before any action that change the move queue length
        to reflect total count of moves and after each single move'''
        # update statusbar
        # in cube.py        NOT_SOLVED = 0,  SOLVED = 1,  HALF_SOLVED = 2
        #TODO: HALF_SOLVED is broken in some ways
        #       * see comment in on_action_rewind_activate
        #       * should only shown if asymetric pattern or image
        
        #FIXME: cube.cube_status_check() is broken, dont use it now
        #status = cube.cube_status_check()
        status = 0
        
        if first:
            plural = 0
        else:
            plural = self.current_movement.all_moves.queue_length
        status_text = [_('not solved'), _('solved'), _('solved')][status]
        #status_text = [_('not solved'), _('solved'), _('rotated center')][status]
        pass
        
        mesg = ngettext("{current} / {total} move",
                        "{current} / {total} moves",
                        plural).format(
                            current=self.current_movement.all_moves.current_place,
                            total=self.current_movement.all_moves.queue_length)
        context = self.statusbar.get_context_id("move-count")
        self.statusbar.pop(context)
        self.statusbar.push(context, mesg)
        
    def update_ui(self, first=False):
        # update toolbar
        if self.current_movement.all_moves.at_start():
            if self.current_movement.all_moves.at_end():
                self.playbarstate.empty()
            else:
                self.playbarstate.first()
        else:
            if self.current_movement.all_moves.at_end():
                self.playbarstate.last(self.current_movement.all_moves.tail.mark_after)
            else:
                self.playbarstate.mid(self.current_movement.all_moves.prev().mark_after)
                
        # update formula
        code, pos, markup = self.current_movement.all_moves.format(self.current_movement.current_cube_state.dimension)
        self.set_entry_formula(code, pos, markup)
        
        self.update_statusbar(first=first)
            
    def error_dialog(self, message):
        '''Popup an error dialog box'''
        dialog = gtk.MessageDialog(self.window,
                            gtk.DIALOG_MODAL, gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE,
                            message)
        
        dialog.set_transient_for(self.window)
        # Destroy the dialog when the user responds to it (e.g. clicks a button)
        #dialog.connect ("response", gtk.Widget.destroy)
        #dialog.show_all ()
        dialog.run()
        dialog.destroy()
        
        
class PlaybarState (object):
    def __init__(self, application):
        actions = [ 'action_rewind', 'action_previous',
                    'action_stop', 'action_play',
                    'action_next', 'action_forward',
                    'action_add_mark', 'action_remove_mark']
        self.play_button_list = [application.play_actions.get_action(a) for a in actions]
    #@debug_func
    def set_toolbar_state(self, sflags, vflags, mark):
        if mark:
            vflags ^= 0b11
        for a in reversed(self.play_button_list):
            a.set_sensitive(sflags & 1)
            a.set_visible(vflags & 1)
            sflags >>= 1
            vflags >>= 1
    def empty(self):          self.set_toolbar_state(0b00000000, 0b11011101, False)
    def first(self):          self.set_toolbar_state(0b00011100, 0b11011101, False)
    def mid(self, mark):      self.set_toolbar_state(0b11011111, 0b11011110, mark)
    def last(self, mark):     self.set_toolbar_state(0b11000011, 0b11011110, mark)
    def playing(self, mark):  self.set_toolbar_state(0b00100000, 0b11101110, mark)
    

