#!/usr/bin/python
# -*- coding: utf-8 -*-

#  Copyright © 2011  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

from pybiklib.debug import debug_func, debug

# A list with fbdurl strings to identify elements as the key, and
# face and index as the value.
# key: - symbolic face
#      - symbolic block
# value: face, index to cube.faces
symbolic_mapping = {
        ('u', 'flu'): (0, 0),  ('u', 'lu'):  (0, 1),  ('u', 'blu'): (0, 2),
        ('u', 'fu'):  (0, 3),  ('u', 'u'):   (0, 4),  ('u', 'bu'):  (0, 5),
        ('u', 'fru'): (0, 6),  ('u', 'ru'):  (0, 7),  ('u', 'bru'): (0, 8),
        
        ('d', 'dfl'): (1, 0),  ('d', 'dl'):  (1, 1),  ('d', 'bdl'): (1, 2),
        ('d', 'df'):  (1, 3),  ('d', 'd'):   (1, 4),  ('d', 'bd'):  (1, 5),
        ('d', 'dfr'): (1, 6),  ('d', 'dr'):  (1, 7),  ('d', 'bdr'): (1, 8),
        
        ('l', 'flu'): (2, 0),  ('l', 'lu'):  (2, 1),  ('l', 'blu'): (2, 2),
        ('l', 'fl'):  (2, 3),  ('l', 'l'):   (2, 4),  ('l', 'bl'):  (2, 5),
        ('l', 'dfl'): (2, 6),  ('l', 'dl'):  (2, 7),  ('l', 'bdl'): (2, 8),
        
        ('r', 'fru'): (3, 0),  ('r', 'ru'):  (3, 1),  ('r', 'bru'): (3, 2),
        ('r', 'fr'):  (3, 3),  ('r', 'r'):   (3, 4),  ('r', 'br'):  (3, 5),
        ('r', 'dfr'): (3, 6),  ('r', 'dr'):  (3, 7),  ('r', 'bdr'): (3, 8),
        
        ('f', 'flu'): (4, 0),  ('f', 'fu'):  (4, 1),  ('f', 'fru'): (4, 2),
        ('f', 'fl'):  (4, 3),  ('f', 'f'):   (4, 4),  ('f', 'fr'):  (4, 5),
        ('f', 'dfl'): (4, 6),  ('f', 'df'):  (4, 7),  ('f', 'dfr'): (4, 8),
        
        ('b', 'blu'): (5, 0),  ('b', 'bu'):  (5, 1),  ('b', 'bru'): (5, 2),
        ('b', 'bl'):  (5, 3),  ('b', 'b'):   (5, 4),  ('b', 'br'):  (5, 5),
        ('b', 'bdl'): (5, 6),  ('b', 'bd'):  (5, 7),  ('b', 'bdr'): (5, 8),
    }
    
# http://www.spiegel.de/spiegel/a-258266.html

# Solution are 3-tuples_
#  1. position on the cube, e.g. "rf" for the front-right edge or "ufr" for
#     the uppper-front-right corner. The order does not matter, "ufr" is equal to "fur".
#     Multiple blocks are separated by spaces.
#  2. The block at the position. The order matters and depends on the first field.
#     The number of blocks must be the same as in field 1.
#     "fr|fl": "fr" or "fl" is allowed at the position
#     "!fru": every block except "fru" is allowed at the position
#     "*fru": a block in any rotation state of "fru" is allowed at the position (fru, ruf, ufr)
#     "!*fru": every block except all "fru" rotations is allowed at the position
#     "f?u": "?" can be every face
#  3. The moves that should be aapplied to the cube.
top_edge = [
        ("rf", "uf", "f'"),
        ("df", "uf", "f'f'"),
        ("fr", "uf", "u2fu2'"),
        ("fu", "uf", "fu2fu2'"),
        ("fd", "uf", "dl2d'l2'"),
        
        ("lf", "uf", "f"),
        ("fl", "uf", "u2'f'u2"),
        
        ("dr", "*uf", "d'"),
        ("dl", "*uf", "d"),
        ("db", "*uf", "dd"),
        
        ("br", "*uf", "u2"),
        ("bl", "*uf", "u2'"),
        
        ("ur", "*uf", "r"),
        ("ub", "*uf", "b"),
        ("ul", "*uf", "l"),
        
        ("f", "!f", "u2"),
    ]
top_corner = [
        ("fld", "ruf", "r'dr"),
        ("fld", "ufr", "ddfd'f'"),
        #("fld", "fru", "f'dffd'f'"),
        ("fld", "fru", "dr'drfd'd'f'"),
        
        ("rfd", "*fru", "d'"),
        ("brd", "*fru", "dd"),
        ("lbd", "*fru", "d"),
        
        ("fru fru", "!fru *fru", "r'd'r"),
        ("rbu", "*fru", "b'd'd'b"),
        ("blu", "*fru", "bdb'"),
        ("lfu", "*fru", "f'df"),
    ]
middle_slice = [
        ("fd", "fr", "d'r'drdfd'f'"),
        ("fd", "fl", "dld'l'd'f'df"),
        
        ("rd", "fr|fl", "d'"),
        ("bd", "fr|fl", "dd"),
        ("ld", "fr|fl", "d"),
        
        ("fd rd bd ld fr", "*?d *?d *?d *?d !fr", "d'r'drdfd'f'"),
        ("fd rd bd ld fl", "*?d *?d *?d *?d !fl", "dld'l'd'f'df"),
    ]
bottom_edge_place = [
        ("df dl db dr", "!*df !*dl !*db !*dr", "d"),
        ("dr dl", "*df *dl", "dfldl'd'f'"),
        ("df db", "*dr *db", "dfldl'd'f'"),
        ("df db", "*db *df", "dfldl'd'f'"),
    ]
bottom_edge_orient = [
        ("dl", "?d", "lu2lu2lu2lu2"),
        
        ("db", "?d", "d"),
        ("df", "?d", "d'"),
        ("dr", "?d", "dd"),
        
        ("df", "dr", "d"),
        ("df", "dl", "d'"),
        ("df", "db", "dd"),
    ]
bottom_corner_place = [
        ("dfr dfl dbr dbl", "!*dfr !*dfl !*dbr !*dbl", "fuffuuffu'f'dfuffuuffu'f'd'"),
        ("dbr dfr", "*dbr !*dfr",                      "fuffuuffu'f'dfuffuuffu'f'd'"),
    ]
bottom_corner_orient = [
        ("dfl dfl dfl dfl", "!dfl !dlb !dbr !drf", "lf'l'flf'l'f"),
        
        ("dlb dlb dlb dlb", "!dfl !dlb !dbr !drf", "d"),
        ("drf drf drf drf", "!dfl !dlb !dbr !drf", "d'"),
        ("dbr dbr dbr dbr", "!dfl !dlb !dbr !drf", "dd"),
        
        ("df", "dr", "d"),
        ("df", "dl", "d'"),
        ("df", "db", "dd"),
    ]
    
solved_face_colors = {}

def rotate_symbol(symbol, turns):
    for j in xrange(turns):
        symbol = ({'l': 'f', 'f': 'r', 'r': 'b', 'b': 'l'}.get(c,c) for c in symbol)
    return ''.join(symbol)
    
def get_color(cube, face, block):
    face, index = symbolic_mapping[(face, ''.join(sorted(block)))]
    return cube.faces[face][index]
    
@debug_func
def test_face(cube, position, condition, face):
    color1 = get_color(cube, position[face], position)
    color2 = solved_face_colors[condition[face]]
    return color1 == color2
    
@debug_func
def test_basic_condition(cube, position, condition):
    if len(position) != len(condition):
        raise ScriptError((position, condition))
    for i in xrange(len(position)):
        if not test_face(cube, position, condition, i):
            return False
    return True
    
def opposite(face):
    return {  'f': 'b', 'b': 'f',
              'l': 'r', 'r': 'l',
              'u': 'd', 'd': 'u',
           }[face]
    
@debug_func
def test_pattern_condition(cube, position, condition):
    if '?' in condition:
        conditions = (condition.replace('?', face, 1)
                        for face in 'flubrd'
                            if face not in condition
                            if opposite(face) not in condition)
        return test_or_conditions(cube, position, conditions)
    else:
        return test_basic_condition(cube, position, condition)
        
def rotated_conditions(condition):
    for i in range(len(condition)):
        yield condition[i:] + condition[:i]
    
@debug_func
def test_prefix_condition(cube, position, condition):
    if condition.startswith('!*'):
        return not test_or_conditions(cube, position, rotated_conditions(condition[2:]))
    elif condition.startswith('*'):
        return test_or_conditions(cube, position, rotated_conditions(condition[1:]))
    elif condition.startswith('!'):
        return not test_pattern_condition(cube, position, condition[1:])
    else:
        return test_pattern_condition(cube, position, condition)
    
@debug_func
def test_or_conditions(cube, position, conditions):
    for prefix_cond in conditions:
        if test_prefix_condition(cube, position, prefix_cond):
            return True
    return False
    
@debug_func
def test_and_conditions(cube, positions, conditions):
    if len(positions) != len (conditions):
        raise ScriptError((positions, conditions))
    for position, or_cond in zip(positions, conditions):
        if not test_or_conditions(cube, position, or_cond.split('|')):
            return False
    return True
    
def execute(cube, stage, turns):
    stage = stage[:]
    #TODO: workaround, solutions should decide themselves, if a turn is needed
    for turn in xrange(turns):
        debug('turn: {}/{}'.format(turn+1, turns))
        count = 0
        i = 0
        while i < len(stage):
            positions, conditions, moves = stage[i]
            if test_and_conditions(cube, positions.split(' '), conditions.split(' ')):
                debug('positive: {}. "{}", "{}", "{}"'.format(i, positions, conditions, moves))
                if count > 4 * len(stage): # this value is just guessed
                    plugin_error_dialog(
                        'An infinite loop was detected. '
                        'This is probably an error in the solution.')
                    return
                count += 1
                
                moves_ = plugin_rotate_flubrd(moves, cube.dimension)
                for move in moves_:
                    cube._rotate_slice(*move)
                i = 0
                continue
            i += 1
        for i, s in enumerate(stage):
            positions, conditions, moves = stage[i]
            positions = rotate_symbol(positions, 1)
            conditions = rotate_symbol(conditions, 1)
            moves = rotate_symbol(moves, 1)
            stage[i] = positions, conditions, moves
        
def spiegel_solve(stage):
    cube = plugin_cube_state()
    if cube.dimension != 3:
        plugin_error_dialog(_('This script only works on 3x3x3 cubes.'))
        return
    for face in 'flubrd':
        solved_face_colors[face] = get_color(cube, face, face)
        
    stages = [  (top_edge, 4),
                (top_corner, 4),
                (middle_slice, 4*4),
                (bottom_edge_place, 4*2),
                (bottom_edge_orient, 1),
                (bottom_corner_place, 4),
                (bottom_corner_orient, 1),
            ]
                
    for i in xrange(stage):
        debug('execute {}/{}'.format(i+1, len(stages)))
        execute(cube, *stages[i])
        
scripts = [
        ('/'.join((_('Solvers'),
        # Translators: "Spiegel" is the name of a german magazine
                                _('Spiegel'))),                          lambda: spiegel_solve(7)),
        ('/'.join((_('Solvers'),_('Spiegel'),_('Top edges'))),           lambda: spiegel_solve(1)),
        ('/'.join((_('Solvers'),_('Spiegel'),_('Top corners'))),         lambda: spiegel_solve(2)),
        ('/'.join((_('Solvers'),_('Spiegel'),_('Middle slice'))),        lambda: spiegel_solve(3)),
        ('/'.join((_('Solvers'),_('Spiegel'),_('Bottom edge place'))),   lambda: spiegel_solve(4)),
        ('/'.join((_('Solvers'),_('Spiegel'),_('Bottom edge orient'))),  lambda: spiegel_solve(5)),
        ('/'.join((_('Solvers'),_('Spiegel'),_('Bottom corner place'))), lambda: spiegel_solve(6)),
        ('/'.join((_('Solvers'),_('Spiegel'),_('Bottom corner orient'))),lambda: spiegel_solve(7)),
    ]

