/**
 ** EEPROM
 ** Copyright 1999 Pete Zaitcev
 ** This code is licensed under GNU General Public License.
 **/
/* #include <asm/contregs.h> */
/* #include <asm/asi.h> */
#include <phys_jk.h>
#include <general.h>		/* __P() */
#include <romlib.h>		/* printk */
#include <system.h>		/* udelay(), idprom, init_eeprom proto. */

#define GPIO0_EECLK   0x01	/* Serial EEPROM */
#define GPIO0_EEDATA  0x02
#define GPIO0_SCCLK   0x10	/* Oops, SmartCard. Anyone got any ideas?? */
#define GPIO0_SCDATA  0x20

/*
 * One important note about delays:
 * Often an I/O takes longer than the delay, as can be seen on an oscylloscope.
 * So, we aways do I/O in a regular way so that if it takes long indeed,
 * our clock meander is not skewed.
 * We do not abolish software busy loops in case we hit a system with
 * relatively fast I/O.
 */
#define GPIO_EE_QARTERCYCLE()  udelay(3)

static int eeprom_send_byte(unsigned int ebase, unsigned char abyte);
static unsigned char eeprom_receive_byte(unsigned int ebase, int last);
static int eeprom_reset(unsigned int ebase);
static void eeprom_start_cond(unsigned int ebase);
static void eeprom_stop_cond(unsigned int ebase);

/*
 * Serial E2PROM replaces IDPROM in Espresso.
 *
 * init_eeprom fills idprom[] from EEPROM.
 */
void init_eeprom(void)
{
	unsigned int ebase;
	static unsigned char buf[16];
	int i;

	/*
	 * Typical base which PROM sets. Replace this with configuration
	 * space operations if a need ever arises.
	 */
	ebase = PHYS_JK_GPIO0;

	if (eeprom_reset(ebase) != 0) {
		printk("eeprom: pulled down on reset\n");
		for (;;) { }
	}

	eeprom_start_cond(ebase);
	if (eeprom_send_byte(ebase, 0xA0) != 0) {	/* Dummy write */
		printk("eeprom: w devsel failed.\n");
		for (;;) { }
	}
	if (eeprom_send_byte(ebase, 0x00) != 0) {	/* set page address */
		printk("eeprom: addr send failed.\n");
		for (;;) { }
	}
	eeprom_start_cond(ebase);			/* break write */
	if (eeprom_send_byte(ebase, 0xA1) != 0) {	/* select for read */
		printk("eeprom: r devsel failed.\n");
		for (;;) { }
	}
	for (i = 0; i < sizeof(buf)-1; i++) {
		buf[i] = eeprom_receive_byte(ebase, 0);
	}
	buf[i] = eeprom_receive_byte(ebase, 1);
	eeprom_stop_cond(ebase);

	if (buf[0] != 1) {
		printk("eeprom: invalid data (%x %x %x %x %x ...)\n",
		    buf[i+0], buf[i+1], buf[i+2], buf[i+3], buf[i+4]);
		for (;;) { }
	}

	bcopy(buf, idprom, 16);
}

/*
 * All I2C parts should go idle after a master driving 9 clock with no data.
 */
static int
eeprom_reset(unsigned int ebase)
{
	int i;
	unsigned char b;

	for (i = 0; i < 9; i++) {
		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);	/* this is for timing */
		stb_bypass(ebase, GPIO0_EEDATA);
		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);
		stb_bypass(ebase, GPIO0_EECLK|GPIO0_EEDATA);
		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);
		stb_bypass(ebase, GPIO0_EECLK|GPIO0_EEDATA);
		GPIO_EE_QARTERCYCLE();
		b = ldb_bypass(ebase);
		stb_bypass(ebase, GPIO0_EEDATA);
	}

	return (b & GPIO0_EEDATA) ? 0 : -1;
}

/*
 * Drive a start condition:
 * data flips high->low while clock is high.
 */
static void
eeprom_start_cond(unsigned int ebase)
{

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EEDATA);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EECLK|GPIO0_EEDATA);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EECLK);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, 0);
}

/*
 * Drive a stop condition:
 * data flips low->high while clock is high.
 */
static void
eeprom_stop_cond(unsigned int ebase)
{

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, 0);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EECLK);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EECLK|GPIO0_EEDATA);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EEDATA);
}

static int
eeprom_send_byte(unsigned int ebase, unsigned char abyte)
{
	int i;
	unsigned char bit;

	/*
	 * Shift the byte out
	 */
	for (i = 0; i < 8; i++) {
		bit = ((abyte >> (7-i)) & 1) << 1;	/* GPIO0_EEDATA */

		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);
		stb_bypass(ebase, bit);

		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);
		stb_bypass(ebase, GPIO0_EECLK|bit);

		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);
		stb_bypass(ebase, GPIO0_EECLK|bit);

		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);
		stb_bypass(ebase, bit);
	}

	/*
	 * Get the ack.
	 * We pull our data up, this allows to read what EEPROM replies.
	 */
	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EEDATA);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EECLK|GPIO0_EEDATA);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EECLK|GPIO0_EEDATA);

	GPIO_EE_QARTERCYCLE();
	bit = ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EEDATA);

	return ((bit & GPIO0_EEDATA) == 0) ? 0 : -1;
}

static unsigned char
eeprom_receive_byte(unsigned int ebase, int last)
{
	unsigned char acc;
	unsigned int b;
	int i;

	acc = 0;	/* not really needed, but */
	for (i = 0; i < 8; i++) {
		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);
		stb_bypass(ebase, GPIO0_EEDATA);

		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);
		stb_bypass(ebase, GPIO0_EECLK|GPIO0_EEDATA);

		GPIO_EE_QARTERCYCLE();
		ldb_bypass(ebase);
		stb_bypass(ebase, GPIO0_EECLK|GPIO0_EEDATA);

		GPIO_EE_QARTERCYCLE();
		b = ldb_bypass(ebase);
		stb_bypass(ebase, GPIO0_EEDATA);

		acc <<= 1;
		if (b & GPIO0_EEDATA) acc |= 1;
	}

	/*
	 * Send our ack
	 */
	b = (last) ? GPIO0_EEDATA : 0;

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, b);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EECLK|b);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, GPIO0_EECLK|b);

	GPIO_EE_QARTERCYCLE();
	ldb_bypass(ebase);
	stb_bypass(ebase, b);

	return acc;
}
