/****************************************************************************
** $Id: helpviewer.cpp,v 1.9 2006/02/01 07:47:52 mkalkbrenner Exp $
**
** Copyright (C) 1992-2002 Trolltech AS.  All rights reserved.
**
** This file is part of an example program for Qt.  This example
** program may be used, distributed and modified without limitation.
**
*****************************************************************************/

#include "helpviewer.h"
#include <qstatusbar.h>
#include <qpixmap.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qiconset.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qstylesheet.h>
#include <qmessagebox.h>
#include <qfiledialog.h>
#include <qapplication.h>
#include <qcombobox.h>
#include <qevent.h>
#include <qlineedit.h>
#include <qobjectlist.h>
#include <qfileinfo.h>
#include <qfile.h>
#include <qdatastream.h>
#include <qprinter.h>
#include <qsimplerichtext.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>

#include <ctype.h>
#include <stdlib.h>

/* XPM */
static char * findxpm[] = {
                              "22 22 6 1",
                              "     c None",
                              ".    c #0A0907",
                              "+    c #484638",
                              "@    c #7B725E",
                              "#    c #72C2E7",
                              "$    c #CCD1D2",
                              "                      ",
                              "                      ",
                              "      ++++++++.+      ",
                              "    ...@$$$$$$..+     ",
                              "   +@###+$$$$$.@..    ",
                              "  +###$$$@$$$$.$@.+   ",
                              " .@###$$$$@$$$.....   ",
                              " .####$$$$+$$$$$$#.   ",
                              " .####$$$$+$$$$$$$.   ",
                              " .####$$$$@$$$$$$$.   ",
                              "  +###$$$@@$$$$$$$.   ",
                              "   +####++@.$$$$$$.   ",
                              "    ...@@+@@.@$$$$.   ",
                              "      +$$.@@@.@$$#.   ",
                              "      +$$@.@@@.$$@.   ",
                              "      +$$$@.++.+$#.   ",
                              "      +$$$$@..+$@$.   ",
                              "      .$@@$@@@@@@@.   ",
                              "      .............   ",
                              "            .....     ",
                              "                      ",
                              "                      "};


/* XPM */
/* Drawn  by Mark Donohoe for the K Desktop Environment */
/* See http://www.kde.org */
static char* back[]={
                        "16 16 5 1",
                        "# c #000000",
                        "a c #ffffff",
                        "c c #808080",
                        "b c #c0c0c0",
                        ". c None",
                        "................",
                        ".......#........",
                        "......##........",
                        ".....#a#........",
                        "....#aa########.",
                        "...#aabaaaaaaa#.",
                        "..#aabbbbbbbbb#.",
                        "...#abbbbbbbbb#.",
                        "...c#ab########.",
                        "....c#a#ccccccc.",
                        ".....c##c.......",
                        "......c#c.......",
                        ".......cc.......",
                        "........c.......",
                        "................",
                        "......................"};


/* XPM */
/* Drawn  by Mark Donohoe for the K Desktop Environment */
/* See http://www.kde.org */
static char* forward[]={
                           "16 16 5 1",
                           "# c #000000",
                           "a c #ffffff",
                           "c c #808080",
                           "b c #c0c0c0",
                           ". c None",
                           "................",
                           "................",
                           ".........#......",
                           ".........##.....",
                           ".........#a#....",
                           "..########aa#...",
                           "..#aaaaaaabaa#..",
                           "..#bbbbbbbbbaa#.",
                           "..#bbbbbbbbba#..",
                           "..########ba#c..",
                           "..ccccccc#a#c...",
                           "........c##c....",
                           "........c#c.....",
                           "........cc......",
                           "........c.......",
                           "................",
                           "................"};


/* XPM */
/* Drawn  by Mark Donohoe for the K Desktop Environment */
/* See http://www.kde.org */
static char* home[]={
                        "16 16 4 1",
                        "# c #000000",
                        "a c #ffffff",
                        "b c #c0c0c0",
                        ". c None",
                        "........... ....",
                        "   ....##.......",
                        "..#...####......",
                        "..#..#aabb#.....",
                        "..#.#aaaabb#....",
                        "..##aaaaaabb#...",
                        "..#aaaaaaaabb#..",
                        ".#aaaaaaaaabbb#.",
                        "###aaaaaaaabb###",
                        "..#aaaaaaaabb#..",
                        "..#aaa###aabb#..",
                        "..#aaa#.#aabb#..",
                        "..#aaa#.#aabb#..",
                        "..#aaa#.#aabb#..",
                        "..#aaa#.#aabb#..",
                        "..#####.######..",
                        "................"};


void TextBrowser::setSource ( const QString & name ) {

    if ( name.left( 7 ) == "http://" || name.left( 6 ) == "ftp://" ) {
        QString command( "$BROWSER %1 &" );
        command.arg( name );
        system( command.local8Bit().data() );
    } else
        QTextBrowser::setSource( name);
}

HelpWindow::HelpWindow( const QString& home_, const QString& _path,
                        QWidget* parent, const char *name )
: QMainWindow( parent, name, WDestructiveClose ),
pathCombo( 0 ), findDialog( 0 ) {
    readHistory();
    readBookmarks();

    browser = new TextBrowser( this );

    browser->mimeSourceFactory()->setFilePath( _path );
    browser->setFrameStyle( QFrame::Panel | QFrame::Sunken );
    connect( browser, SIGNAL( sourceChanged(const QString& ) ),
             this, SLOT( sourceChanged( const QString&) ) );

    setCentralWidget( browser );

    if ( !home_.isEmpty() )
        browser->setSource( home_ );

    connect( browser, SIGNAL( highlighted( const QString&) ),
             statusBar(), SLOT( message( const QString&)) );

    resize( 640,700 );

    QPopupMenu* file = new QPopupMenu( this );
    file->insertItem( _("&New Window"), this, SLOT( newWindow() ), CTRL+Key_N );
    file->insertItem( _("&Open File"), this, SLOT( openFile() ), CTRL+Key_O );
    file->insertItem( _("&Print"), this, SLOT( print() ), CTRL+Key_P );
    file->insertSeparator();
    file->insertItem( _("&Close"), this, SLOT( close() ), CTRL+Key_Q );

    // The same icons are used twice each.
    QIconSet icon_back( QPixmap( (const char **) back) );
    QIconSet icon_forward( QPixmap((const char **) forward) );
    QIconSet icon_home( QPixmap((const char **) home) );
    QIconSet icon_find( QPixmap((const char **) findxpm) );

    QPopupMenu* go = new QPopupMenu( this );
    backwardId = go->insertItem( icon_back,
                                 _("&Backward"), browser, SLOT( backward() ),
                                 CTRL+Key_Left );
    forwardId = go->insertItem( icon_forward,
                                _("&Forward"), browser, SLOT( forward() ),
                                CTRL+Key_Right );
    go->insertItem( icon_home, _("&Home"), browser, SLOT( home() ) );
    go->insertItem( icon_find,
                    _("&Find"), this, SLOT( find() ),
                    CTRL+Key_F );

    hist = new QPopupMenu( this );
    QStringList::Iterator it = history.begin();
    for ( ; it != history.end(); ++it )
        mHistory[ hist->insertItem( *it ) ] = *it;
    connect( hist, SIGNAL( activated( int ) ),
             this, SLOT( histChosen( int ) ) );

    bookm = new QPopupMenu( this );
    bookm->insertItem( _( "Add Bookmark" ), this, SLOT( addBookmark() ) );
    bookm->insertSeparator();

    QStringList::Iterator it2 = bookmarks.begin();
    for ( ; it2 != bookmarks.end(); ++it2 )
        mBookmarks[ bookm->insertItem( *it2 ) ] = *it2;
    connect( bookm, SIGNAL( activated( int ) ),
             this, SLOT( bookmChosen( int ) ) );

    menuBar()->insertItem( _("&File"), file );
    menuBar()->insertItem( _("&Go"), go );
    menuBar()->insertItem( _( "History" ), hist );
    menuBar()->insertItem( _( "Bookmarks" ), bookm );
    menuBar()->insertSeparator();

    menuBar()->setItemEnabled( forwardId, FALSE);
    menuBar()->setItemEnabled( backwardId, FALSE);
    connect( browser, SIGNAL( backwardAvailable( bool ) ),
             this, SLOT( setBackwardAvailable( bool ) ) );
    connect( browser, SIGNAL( forwardAvailable( bool ) ),
             this, SLOT( setForwardAvailable( bool ) ) );


    QToolBar* toolbar = new QToolBar( this );
    addToolBar( toolbar, "Toolbar");
    QToolButton* button;

    button = new QToolButton( icon_back, _("Backward"), "", browser, SLOT(backward()), toolbar );
    connect( browser, SIGNAL( backwardAvailable(bool) ), button, SLOT( setEnabled(bool) ) );
    button->setEnabled( FALSE );
    button = new QToolButton( icon_forward, _("Forward"), "", browser, SLOT(forward()), toolbar );
    connect( browser, SIGNAL( forwardAvailable(bool) ), button, SLOT( setEnabled(bool) ) );
    button->setEnabled( FALSE );
    button = new QToolButton( icon_home, _("Home"), "", browser, SLOT(home()), toolbar );
    button = new QToolButton( icon_find, _("Find"), "", this, SLOT(find()), toolbar );

    toolbar->addSeparator();

    pathCombo = new QComboBox( TRUE, toolbar );
    connect( pathCombo, SIGNAL( activated( const QString & ) ),
             this, SLOT( pathSelected( const QString & ) ) );
    toolbar->setStretchableWidget( pathCombo );
    setRightJustification( TRUE );
    setDockEnabled( DockLeft, FALSE );
    setDockEnabled( DockRight, FALSE );

    pathCombo->insertItem( home_ );
    browser->setFocus();

}


void HelpWindow::setBackwardAvailable( bool b) {
    menuBar()->setItemEnabled( backwardId, b);
}

void HelpWindow::setForwardAvailable( bool b) {
    menuBar()->setItemEnabled( forwardId, b);
}


void HelpWindow::sourceChanged( const QString& url ) {
    if ( browser->documentTitle().isNull() )
        setCaption( "Helpviewer - " + url );
    else
        setCaption( "Helpviewer - " + browser->documentTitle() ) ;

    if ( !url.isEmpty() && pathCombo ) {
        bool exists = FALSE;
        int i;
        for ( i = 0; i < pathCombo->count(); ++i ) {
            if ( pathCombo->text( i ) == url ) {
                exists = TRUE;
                break;
            }
        }
        if ( !exists ) {
            pathCombo->insertItem( url, 0 );
            pathCombo->setCurrentItem( 0 );
            mHistory[ hist->insertItem( url ) ] = url;
        } else
            pathCombo->setCurrentItem( i );
    }
}

HelpWindow::~HelpWindow() {
    history =  mHistory.values();

    QFile f( QDir::currentDirPath() + "/.history" );
    f.open( IO_WriteOnly );
    QDataStream s( &f );
    s << history;
    f.close();

    bookmarks = mBookmarks.values();

    QFile f2( QDir::currentDirPath() + "/.bookmarks" );
    f2.open( IO_WriteOnly );
    QDataStream s2( &f2 );
    s2 << bookmarks;
    f2.close();
}

void HelpWindow::openFile() {
    #ifndef QT_NO_FILEDIALOG
    QString fn = QFileDialog::getOpenFileName( QString::null, QString::null, this );
    if ( !fn.isEmpty() )
        browser->setSource( fn );
    #endif
}

void HelpWindow::newWindow() {
    ( new HelpWindow(browser->source(), "qbrowser") )->show();
}

void HelpWindow::print() {
    #ifndef QT_NO_PRINTER
    QPrinter printer;
    printer.setFullPage(TRUE);
    if ( printer.setup( this ) ) {
        QPainter p( &printer );
        QPaintDeviceMetrics metrics(p.device());
        int dpix = metrics.logicalDpiX();
        int dpiy = metrics.logicalDpiY();
        const int margin = 72; // pt
        QRect body(margin*dpix/72, margin*dpiy/72,
                   metrics.width()-margin*dpix/72*2,
                   metrics.height()-margin*dpiy/72*2 );
        QSimpleRichText richText( browser->text(), QFont(), browser->context(), browser->styleSheet(),
                                  browser->mimeSourceFactory(), body.height() );
        richText.setWidth( &p, body.width() );
        QRect view( body );
        int page = 1;
        do {
            richText.draw( &p, body.left(), body.top(), view, colorGroup() );
            view.moveBy( 0, body.height() );
            p.translate( 0 , -body.height() );
            p.drawText( view.right() - p.fontMetrics().width( QString::number(page) ),
                        view.bottom() + p.fontMetrics().ascent() + 5, QString::number(page) );
            if ( view.top()  >= richText.height() )
                break;
            printer.newPage();
            page++;
        } while (TRUE);
    }
    #endif
}

void HelpWindow::pathSelected( const QString &_path ) {
    browser->setSource( _path );
    if ( mHistory.values().contains(_path) )
        mHistory[ hist->insertItem( _path ) ] = _path;
}

void HelpWindow::readHistory() {
    if ( QFile::exists( QDir::currentDirPath() + "/.history" ) ) {
        QFile f( QDir::currentDirPath() + "/.history" );
        f.open( IO_ReadOnly );
        QDataStream s( &f );
        s >> history;
        f.close();
        while ( history.count() > 20 )
            history.remove( history.begin() );
    }
}

void HelpWindow::readBookmarks() {
    if ( QFile::exists( QDir::currentDirPath() + "/.bookmarks" ) ) {
        QFile f( QDir::currentDirPath() + "/.bookmarks" );
        f.open( IO_ReadOnly );
        QDataStream s( &f );
        s >> bookmarks;
        f.close();
    }
}

void HelpWindow::histChosen( int i ) {
    if ( mHistory.contains( i ) )
        browser->setSource( mHistory[ i ] );
}

void HelpWindow::bookmChosen( int i ) {
    if ( mBookmarks.contains( i ) )
        browser->setSource( mBookmarks[ i ] );
}

void HelpWindow::addBookmark() {
    mBookmarks[ bookm->insertItem( caption() ) ] = browser->context();
}

void HelpWindow::find() {
    if ( !findDialog ) {
        findDialog = new FindDialog( this );
        findDialog->setBrowser( browser );
    }
    findDialog->comboFind->setFocus();
    findDialog->comboFind->lineEdit()->setSelection(
        0, findDialog->comboFind->lineEdit()->text().length() );
    findDialog->show();
}
