/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

public class Postler.AccountWidget : Gtk.VBox {
    Gtk.SizeGroup sizegroup;
    Gtk.Box content_area;
    Gtk.Box advanced_area;

    Elementary.Entry header;
    Elementary.Entry filter;

    Elementary.Entry realname;
    public Elementary.Entry address;
    Elementary.Entry password;
    Elementary.Entry organization;
    Gtk.TextView signature;

    Elementary.Entry receiver;
    Elementary.Entry username;
    Elementary.Entry prefix;
    Elementary.Entry sender;
    Elementary.Entry sender_port;

    bool address_changed = false;
    AccountInfo info;

    public AccountWidget (AccountInfo? account_info=null) {
        if (account_info == null) {
            info = new AccountInfo ();
            info.type = AccountType.IMAP;
            string realname = Environment.get_real_name ();
            if (realname == "Unknown")
                realname = "";
            info.realname = realname.locale_to_utf8 (-1, null, null, null);
            try {
                GLib.FileUtils.get_contents (
                    Environment.get_home_dir () + "/.signature", out info.signature);
                info.signature = info.signature.chomp ();
            } catch (GLib.FileError error) { }
        } else {
            info = account_info;
        }

        sizegroup = new Gtk.SizeGroup (Gtk.SizeGroupMode.HORIZONTAL);
        content_area = new Gtk.VBox (false, 0);

        if (info.type == AccountType.SEARCH) {
            string? query = info.path.substring (7);
            return_if_fail (query != null);
            string[] parts = query.split ("/", 2);
            return_if_fail (parts[0] != null && parts[1] != null);

            header = new Elementary.Entry ();
            header.set_text (parts[0]);
            add_label_entry (_("_Header"), header);
            filter = new Elementary.Entry ();
            filter.set_text (parts[1]);
            add_label_entry (_("_Keywords"), filter);

            content_area.show_all ();
            pack_end (content_area, true, true, 0);

            /* Empty entry since it's a public member */
            address = new Elementary.Entry ();
            return;
        }

        realname = new Elementary.Entry (_("Firstname Lastname"));
        if (info.realname != null)
            realname.set_text (info.realname);
        add_label_entry (_("_Full Name:"), realname);
        address = new Elementary.Entry (_("email@example.com"));
        address.focus_out_event.connect (on_focus_out);
        if (info.address != null)
            address.set_text (info.address);
        add_label_entry (_("_Email Address:"), address);
        password = new Elementary.Entry (_("Password"));
        password.visibility = false;
        if (info.password != null)
            password.set_text (info.password);
        add_label_entry (_("_Password:"), password);
        organization = new Elementary.Entry (_("Optional, Inc."));
        if (info.organization != null)
            organization.set_text (info.organization);
        add_label_entry (_("_Organization:"), organization);

        signature = new Gtk.TextView ();
        signature.buffer.text = info.signature ?? "";
        signature.accepts_tab = false;
        add_label_entry (_("_Signature:"), new Postler.ScrolledWindow (signature));
        content_area.pack_start (new Gtk.Label (
            _("Insert -- to render part of the signature gray.")), false, false, 4);
        (signature.parent as Gtk.ScrolledWindow).shadow_type = Gtk.ShadowType.ETCHED_OUT;

        var expander = new Gtk.Expander.with_mnemonic (_("_Advanced"));
        content_area.pack_start (expander, false, false, 4);
        advanced_area = new Gtk.VBox (false, 4);
        expander.add (advanced_area);

        receiver = new Elementary.Entry (_("imap.example.com"));
        var pop3_label = new Gtk.Label (_("POP3 is not supported."));
        pop3_label.set_no_show_all (true);
        receiver.changed.connect ((editable) => {
            Gtk.Entry entry = editable as Gtk.Entry;
            Gdk.Color? background = null;
            Gdk.Color? foreground = null;
            /* pop3 is not supported */
            if (entry.text.has_prefix ("pop3.")
             || entry.text.has_prefix ("pop.")) {
                Gdk.Color.parse ("#ef7070", out background);
                Gdk.Color.parse ("#000", out foreground);
            }
            Gtk.Widget widget = editable as Gtk.Widget;
            widget.modify_base (widget.get_state (), background);
            widget.modify_text (widget.get_state (), foreground);
            pop3_label.visible = background != null;
        });
        if (info.receive != null)
            receiver.set_text (info.receive);
        add_label_entry (_("_Receiving Server:"), receiver, true);
        advanced_area.pack_start (pop3_label, false, false, 4);
        username = new Elementary.Entry (_("Username"));
        if (info.username != null)
            username.set_text (info.username);
        add_label_entry (_("_Username:"), username, true);
        prefix = new Elementary.Entry (_("Prefix"));
        if (info.prefix != null)
            prefix.set_text (info.prefix);
        add_label_entry (_("Prefi_x:"), prefix, true);
        sender = new Elementary.Entry (_("smtp.example.com"));
        if (info.send != null)
            sender.set_text (info.send);
        sender.changed.connect ((widget) => {
            /* "stmp." is a common typo, highlight entry if that happens */
            if (sender.text.has_prefix ("stmp.")) {
                Gdk.Color background, foreground;
                Gdk.Color.parse ("#ef7070", out background);
                Gdk.Color.parse ("#000", out foreground);
                sender.modify_base (sender.get_state (), background);
                sender.modify_text (sender.get_state (), foreground);
            } else {
                sender.modify_base (sender.get_state (), null);
                sender.modify_text (sender.get_state (), null);
            }
        });
        add_label_entry (_("Sen_ding Server:"), sender, true);
        sender_port = new Elementary.Entry ("25");
        if (info.send_port != 0)
            sender_port.set_text (info.send_port.to_string ());
        add_label_entry (_("Sen_ding Port:"), sender_port, true);
        advanced_area.pack_start (new Gtk.Label (
            _("STARTTLS is used by default, use port 465 for secure SMTP.")), false, false, 4);
        if (info.address != null)
            set_servers_from_address ();
        content_area.show_all ();
        pack_end (content_area, true, true, 0);
    }

    bool on_focus_out () {
        address_changed = (info.address != address.get_text () || address_changed);
        set_servers_from_address ();
        return false;
    }

    void set_servers_from_address () {
        string domain = address.get_text ();
        if (domain == "")
            return;

        string user = domain.split ("@") [0];
        domain = domain.split ("@") [1];
        if (receiver.get_text () == "" || address_changed)
            receiver.set_text ("imap." + domain);
        if (sender.get_text () == "" || address_changed)
            sender.set_text ("smtp." + domain);
        if (username.get_text () == "" || address_changed)
            username.set_text (user);
    }

    void add_label_entry (string text, Gtk.Widget widget, bool advanced=false) {
        var editable = (widget is Gtk.Bin) ? (widget as Gtk.Bin).get_child () : widget;
        var hbox = new Gtk.HBox (false, 0);
        if (advanced)
            advanced_area.pack_start (hbox, false, false, 0);
        else
            content_area.pack_start (hbox, false, false, 4);
        var label = new Gtk.Label.with_mnemonic (text);
        label.set_mnemonic_widget (editable);
        sizegroup.add_widget (label);
        hbox.pack_start (label, false, false, 4);
        if (editable is Elementary.Entry) {
            label.xalign = 1.0f;
            (editable as Elementary.Entry).activate.connect ((editable) => {
                apply (); });
        }
        else {
            hbox.set_orientation (Gtk.Orientation.VERTICAL);
            label.xalign = 0.0f;
        }
        hbox.pack_start (widget, false, false, 4);
    }

    public void apply () {
        if (info.type == AccountType.SEARCH) {
            info.path = "search:" + header.get_text () + "/" + filter.get_text ();
            done (info);
            return;
        }

        if (info.name == null)
            info.name = address.get_text ();

        /* Reset localised folders on server changes */
        if (info.address != address.get_text ()
         || info.password != password.get_text ()
         || info.receive != null && info.receive != receiver.get_text ()
         || info.username != null && info.username != username.get_text ()
         || info.prefix != null && info.prefix != username.get_text ()
         || info.send != null && info.send != sender.get_text ()
         || info.send_port != 0 && info.send_port != sender_port.get_text ().to_int ()) {
            for (int i = 1; i < FolderType.MAX; i++)
                info.folders[i] = null;
        }

        info.realname = realname.get_text ();
        info.address = address.get_text ();
        info.password = password.get_text ();
        if (info.address != null)
            set_servers_from_address ();

        info.organization = organization.get_text () != "" ? organization.get_text () : null;
        info.signature = signature.buffer.text != "" ? signature.buffer.text : null;
        info.receive = receiver.get_text () != "" ? receiver.get_text () : null;
        info.username = username.get_text () != "" ? username.get_text () : null;
        info.prefix = prefix.get_text () != "" ? prefix.get_text () : null;
        info.send = sender.get_text () != "" ? sender.get_text () : null;
        info.send_port = sender_port.get_text () != "" ? sender_port.get_text ().to_int () : 0;
        done (info);
    }

    public signal void done (AccountInfo info);
}

public class Postler.AccountSetup : Gtk.Dialog {
    AccountWidget widget;

    public signal void done (AccountInfo? info);

    private AccountSetup (AccountInfo? info=null) {
        GLib.Object (title: _("Account Properties"));
        widget = new AccountWidget (info);
        widget.done.connect ((info) => {
            hide ();
            done (info);
            destroy ();
        });
        (get_content_area () as Gtk.Box).pack_start (widget, true, true, 0);
        widget.show ();
        response.connect (responded);
    }

    void responded (int response) {
        if (response == Gtk.ResponseType.APPLY)
            widget.apply ();
        destroy ();
    }

    public static AccountSetup new_account () {
        var setup = new AccountSetup ();
        setup.add_button (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL);
        setup.add_button (_("_Create Account"), Gtk.ResponseType.APPLY);
        setup.set_default_response (Gtk.ResponseType.APPLY);
        setup.set_response_sensitive (Gtk.ResponseType.APPLY, false);
        setup.widget.address.changed.connect ((editable) => {
             setup.set_response_sensitive (Gtk.ResponseType.APPLY,
                 setup.widget.address.get_text ().chr (-1, '@') != null);
        });
        setup.show ();
        return setup;
    }

    public static AccountSetup edit_account (AccountInfo info) {
         var setup = new AccountSetup (info);

        unowned string remove_label = _("Remove Account");
        unowned string update_label = _("Update Account");
        if (info.type == AccountType.SEARCH) {
            setup.title = _("Saved Search Properties");
            remove_label = _("Remove Saved Search");
            update_label = _("Update Saved Search");
        }

        var remove = new Gtk.Button.with_mnemonic (remove_label);
        remove.set_image (new Gtk.Image.from_stock (Gtk.STOCK_DELETE,
                                                    Gtk.IconSize.BUTTON));
        remove.relief = Gtk.ReliefStyle.NONE;
        remove.show ();
        remove.image.show ();
        (setup.action_area as Gtk.Box).pack_start (remove, false, false, 0);
        remove.clicked.connect ((button) => {
            var toplevel = setup.get_toplevel () as Gtk.Window;
            var dialog = new Gtk.MessageDialog (toplevel, 0,
                Gtk.MessageType.WARNING, Gtk.ButtonsType.NONE,
                _("You are about to remove the account \"%s\"."), info.name);
                dialog.add_buttons (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
                    remove_label, Gtk.ResponseType.OK);
                dialog.set_default_response (Gtk.ResponseType.CANCEL);
                int response = dialog.run ();
                dialog.destroy ();
                if (response == Gtk.ResponseType.OK) {
                    setup.hide ();
                    setup.done (null);
                    setup.destroy ();
                }
        });
        var apply = setup.add_button (update_label, Gtk.ResponseType.APPLY);
        (setup.action_area as Gtk.Box).set_child_packing (apply,
            true, true, 0, Gtk.PackType.END);
        setup.set_default_response (Gtk.ResponseType.APPLY);
        setup.widget.address.changed.connect ((editable) => {
             setup.set_response_sensitive (Gtk.ResponseType.APPLY,
                 setup.widget.address.get_text ().chr (-1, '@') != null);
        });
        setup.show ();
        return setup;
    }
}

