/*************************************************************************\
*   Copyright (C) 2009 by Ulf Kreissig                                    *
*   udev@gmx.net                                                          *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA            *
\*************************************************************************/

//--- LOCAL ---
#include "../config.h"
#include "accuweather_configloader.h"
#include "logger/streamlogger.h"

//--- QT4 ---
#include <QDir>
#include <QString>
#include <QStringList>
#include <QThread>
#include <QXmlStreamAttributes>
#include <QXmlStreamReader>

//--- KDE4 ---
#include <kstandarddirs.h>


AccuWeatherConfigLoader::AccuWeatherConfigLoader(QObject * parent)
      : QThread(parent)
{
	m_pMap = 0;
}

AccuWeatherConfigLoader::~AccuWeatherConfigLoader()
{
	if (m_pMap != 0)
		delete m_pMap;
}

AccuWeatherConfigMap *
AccuWeatherConfigLoader::configMap()
{
	AccuWeatherConfigMap * map = m_pMap;
	m_pMap = 0;
	return map;
}


void
AccuWeatherConfigLoader::run()
{
	readConfig();
}

QString
AccuWeatherConfigLoader::createLocationCode(
	const QString & areaCode)
{
	if( areaCode.isEmpty())
		return QString();
	return areaCode;
}
QString
AccuWeatherConfigLoader::createLocationCode(
	const QString & areaCode,
	const QString & countryCode)
{
	if( areaCode.isEmpty() || countryCode.isEmpty() )
		return QString();
	return QString("%1|%2").arg(areaCode).arg(countryCode);
}
QString
AccuWeatherConfigLoader::createLocationCode(
	const QString & areaCode,
	const QString & countryCode,
	const QString & locationCode)
{
	if( areaCode.isEmpty() || countryCode.isEmpty() || locationCode.isEmpty() )
		return QString();
	return QString("%1|%2|%3").arg(areaCode).arg(countryCode).arg(locationCode);
}


void
AccuWeatherConfigLoader::readConfig()
{
	dStartFunct();
	
	QString configPath;
	if( !getConfigPath(configPath) )
	{
		dEndFunct();
		return;
	}
	dDebug() << "Found config file: " << configPath;
	
	QFile file(configPath);
	if( !file.open(QIODevice::ReadOnly|QIODevice::Text) )
	{
		dEndFunct();
		return;
	}
	
	dDebug() << "Start config reader...";
	
	QXmlStreamReader xml( &file);
	int state = 0;
	QString areaCode = "";
	QString countryCode = "";
	
	AccuWeatherConfigMap * configMap = new AccuWeatherConfigMap();
	
	while(!xml.atEnd())
	{
		xml.readNext();
		
		if( xml.isStartElement() )
		{
			if( state == 0 && xml.name().compare("accuweather_areas") == 0 )
				state += 1;
			else if( state == 1 && xml.name().compare("accuweather_area") == 0 )
			{
				state += 1;
				
				QXmlStreamAttributes attr = xml.attributes();
				areaCode = attr.value("code").toString();
				QString map = attr.value("map").toString();

				QString key = createLocationCode( areaCode );

				if( !key.isEmpty() && !map.isEmpty() )
					configMap->insert(areaCode, map.toLatin1());
			}
			else if( state == 2 && xml.name().compare("accuweather_country") == 0 )
			{
				state += 1;

				QXmlStreamAttributes attr = xml.attributes();
				countryCode = attr.value("code").toString();
				QString map = attr.value("map").toString();

				QString key = createLocationCode(areaCode, countryCode);
				
				if( !key.isEmpty() && !map.isEmpty() )
					configMap->insert(key, map.toLatin1());
			}
			else if( state == 3 && xml.name().compare("accuweather_location") == 0 )
			{
				state += 1;
				
				QXmlStreamAttributes attr = xml.attributes();
				QString locationCode = attr.value("code").toString();
				QString map = attr.value("map").toString();

				QString key = createLocationCode(areaCode, countryCode, locationCode);

				if( !key.isEmpty() && !map.isEmpty() )
					configMap->insert(key, map.toLatin1());
			}
			else
				dWarning() << QString("Unsupported tag <%1> (Line %2, Column %3)")
					.arg(xml.name().toString()).arg(xml.lineNumber()).arg(xml.columnNumber());
		}
		if( xml.isEndElement() )
		{
			if( state == 4 && xml.name().compare("accuweather_location") == 0 )
				state -= 1;
			else if( state == 3 && xml.name().compare("accuweather_country") == 0 )
			{
				state -= 1;
				countryCode.clear();
			}
			else if( state == 2 && xml.name().compare("accuweather_area") == 0 )
			{
				state -= 1;
				areaCode.clear();
			}
			else if( state == 1 && xml.name().compare("accuweather_areas") == 0 )
				state -= 1;
		}
		if( xml.hasError() )
		{
			dWarning() << xml.errorString()
				<< QString(": XML-Tag %1 Line %2, Column %3")
					.arg(xml.name().toString()).arg(xml.lineNumber()).arg(xml.columnNumber());
			break;
		}
	}
	file.close();
	
	if( !xml.hasError() )
		m_pMap = configMap;
	
	dEndFunct();
}


bool
AccuWeatherConfigLoader::getConfigPath(QString & configPath )
{
	configPath.clear();
	
	KStandardDirs kdeDirs;
	QStringList paths =  kdeDirs.resourceDirs("data");
	bool bFound = false;
	QString configFileName("satellite_map.conf");
	
	foreach(QString path, paths)
	{
		QDir dir( QString("%1/ion_accuweather").arg(path) );
		
		if(dir.exists(configFileName))
		{
			configPath = QDir::cleanPath( dir.absoluteFilePath(configFileName) );
			bFound = true;
			break;
		}
	}
	
	return bFound;
}
