/* Copyright (c) 2001-2011, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.dicom;

import com.pixelmed.geometry.GeometryOfSlice;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * <p>A class to describe a set of frames sharing common characteristics suitable for display or analysis as an entity.</p>
 *
 * @author	dclunie
 */
class FrameSet {

	private static final String identString = "@(#) $Header: /userland/cvs/pixelmed/imgbook/com/pixelmed/dicom/FrameSet.java,v 1.3 2011/12/29 16:48:00 dclunie Exp $";
	
	private Map<AttributeTag,String> distinguishingAttributes;
	private ArrayList<Map<AttributeTag,String>> perFrameAttributes;
	
	private Map<AttributeTag,String> mapOfUsedAttributeTagsToDictionaryKeywords = new HashMap<AttributeTag,String>();
	
	private static Set<AttributeTag> distinguishingAttributeTags = new HashSet<AttributeTag>();
	{
		distinguishingAttributeTags.add(TagFromName.Rows);
		distinguishingAttributeTags.add(TagFromName.Columns);
		distinguishingAttributeTags.add(TagFromName.Modality);
		distinguishingAttributeTags.add(TagFromName.SOPClassUID);
		distinguishingAttributeTags.add(TagFromName.StudyInstanceUID);
		distinguishingAttributeTags.add(TagFromName.FrameOfReferenceUID);
		distinguishingAttributeTags.add(TagFromName.BodyPartExamined);
		distinguishingAttributeTags.add(TagFromName.ImageOrientationPatient);
		distinguishingAttributeTags.add(TagFromName.PixelSpacing);
		distinguishingAttributeTags.add(TagFromName.SliceThickness);
	}
	
	private static Set<AttributeTag> excludeFromGeneralPerFrameProcessingTags = new HashSet<AttributeTag>();
	{
		excludeFromGeneralPerFrameProcessingTags.addAll(distinguishingAttributeTags);
		excludeFromGeneralPerFrameProcessingTags.add(TagFromName.AcquisitionDateTime);
		excludeFromGeneralPerFrameProcessingTags.add(TagFromName.AcquisitionDate);
		excludeFromGeneralPerFrameProcessingTags.add(TagFromName.AcquisitionTime);
	}
	
	/**
	 * <p>Extract the attributes and values that are required to be common to all members of this frame set
	 * and for which different values will create distinct fram sets.</p>
	 *
	 * @param	list	a lists of DICOM attributes
	 * @return		a Map<AttributeTag,String> of the attributes and values required to be the same for membership in this frame set
	 */
	private Map<AttributeTag,String> getDistinguishingAttributes(AttributeList list) {
		Map<AttributeTag,String> map = new TreeMap<AttributeTag,String>();	// want to keep sorted for output as toString()		
		for (AttributeTag tag : distinguishingAttributeTags) {
			map.put(tag,Attribute.getDelimitedStringValuesOrEmptyString(list,tag));
			if (mapOfUsedAttributeTagsToDictionaryKeywords.get(tag) == null) {
				mapOfUsedAttributeTagsToDictionaryKeywords.put(tag,list.getDictionary().getNameFromTag(tag));
			}
		}
		return map;
	}
	
	/**
	 * <p>Extract the attributes and values that are expected to be different for all members of this frame set.</p>
	 *
	 * @param	list	a lists of DICOM attributes
	 * @return		a Map<AttributeTag,String> of the attributes and values that are expected to be different for each member of this frame set
	 */
	private Map<AttributeTag,String> getPerFrameAttributes(AttributeList list) {
		Map<AttributeTag,String> map = new TreeMap<AttributeTag,String>();	// want to keep sorted for output as toString()
		
		DicomDictionary dictionary = list.getDictionary();
		for (AttributeTag tag : list.keySet()) {
			if (! tag.isPrivate() && ! tag.isRepeatingGroup() && ! tag.isFileMetaInformationGroup() && ! excludeFromGeneralPerFrameProcessingTags.contains(tag)) {
				Attribute a = list.get(tag);
				if (! (a instanceof SequenceAttribute)) {
					String value = a.getDelimitedStringValuesOrEmptyString();
					map.put(tag,value);
					if (mapOfUsedAttributeTagsToDictionaryKeywords.get(tag) == null) {
						mapOfUsedAttributeTagsToDictionaryKeywords.put(tag,list.getDictionary().getNameFromTag(tag));
					}
				}
			}
		}
		
		String useAcquisitionDateTime = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.AcquisitionDateTime);
		if (useAcquisitionDateTime.length() == 0) {
			useAcquisitionDateTime = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.AcquisitionDate)		// could be zero lenght when time is present, which is not ideal :(
								   + Attribute.getSingleStringValueOrEmptyString(list,TagFromName.AcquisitionTime);		// assume hh is zero padded if less than 10, which should be true, but should check :(
		}
		map.put(TagFromName.AcquisitionDateTime,useAcquisitionDateTime);

		return map;
	}
		
	/**
	 * <p>Check to see if a single or multi-frame object is a potential member of the current frame set.</p>
	 *
	 * @param	list	a lists of DICOM attributes for the object to be checked
	 * @return		true if the attribute list matches the criteria for membership in this frame set
	 */
	boolean eligible(AttributeList list) {
		Map<AttributeTag,String> tryMap = getDistinguishingAttributes(list);
		boolean isEligible =  tryMap.equals(distinguishingAttributes);
//System.err.println("FrameSet.eligible(): "+isEligible);
		return isEligible;
	}
	
	/**
	 * <p>Insert the single or multi-frame object into the current frame set.</p>
	 *
	 * <p>It is assumed that the object has already been determined to be eligible.</p>
	 *
	 * @param	list	a lists of DICOM attributes for the object to be inserted
	 */
	void insert(AttributeList list) {
		perFrameAttributes.add(getPerFrameAttributes(list));
	}
	
	/**
	 * <p>Create a new frame set using the single or multi-frame object.</p>
	 *
	 * @param	list	a lists of DICOM attributes for the object from which the frame set is to be created
	 */
	FrameSet(AttributeList list) {
		distinguishingAttributes = getDistinguishingAttributes(list);
		perFrameAttributes = new ArrayList<Map<AttributeTag,String>>();
		insert(list);
	}
	
	/**
	 * <p>Return a String representing a Map.Entry's value.</p>
	 *
	 * @param	entry	a key-value pair from a Map
	 * @return	a string representation of the value of this object
	 */
	private String toString(Map.Entry<AttributeTag,String> entry) {
		StringBuffer strbuf = new StringBuffer();
		AttributeTag tag = entry.getKey();
		strbuf.append(tag.toString());
		strbuf.append(" ");
		strbuf.append(mapOfUsedAttributeTagsToDictionaryKeywords.get(tag));
		strbuf.append(" = ");
		strbuf.append(entry.getValue());
		return strbuf.toString();
	}
	
	/**
	 * <p>Return a String representing this object's value.</p>
	 *
	 * @return	a string representation of the value of this object
	 */
	public String toString() {
		StringBuffer strbuf = new StringBuffer();
		if (distinguishingAttributes != null) {
			strbuf.append("\tDistinguishing:\n");
			Set<Map.Entry<AttributeTag,String>> set = distinguishingAttributes.entrySet();
			for (Map.Entry<AttributeTag,String> entry : set) {
				strbuf.append("\t\t");
				strbuf.append(toString(entry));
				strbuf.append("\n");
			}
		}
		if (perFrameAttributes != null) {
			int j = 0;
			for (Map<AttributeTag,String> map : perFrameAttributes) {
				strbuf.append("\tFrame [");
				strbuf.append(Integer.toString(j));
				strbuf.append("]:\n");
				if (map != null) {
					Set<Map.Entry<AttributeTag,String>> set = map.entrySet();
					for (Map.Entry<AttributeTag,String> entry : set) {
						strbuf.append("\t\t\t");
						strbuf.append(toString(entry));
						strbuf.append("\n");
					}
				}
				++j;
			}
		}
		return strbuf.toString();
	}
}

