/* Copyright (c) 2001-2010, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.test;

import com.pixelmed.dose.*;

import junit.framework.*;

public class TestCTDoseAcquisition extends TestCase {
	
	// constructor to support adding tests to suite ...
	
	public TestCTDoseAcquisition(String name) {
		super(name);
	}
	
	// add tests to suite manually, rather than depending on default of all test...() methods
	// in order to allow adding TestCTDoseAcquisition.suite() in AllTests.suite()
	// see Johannes Link. Unit Testing in Java pp36-47
	
	public static Test suite() {
		TestSuite suite = new TestSuite("TestCTDoseAcquisition");
		
		suite.addTest(new TestCTDoseAcquisition("TestCTDoseAcquisitionConstructor_WithAllParameters_Helical"));
		suite.addTest(new TestCTDoseAcquisition("TestCTDoseAcquisitionConstructor_WithAllParameters_Axial"));
		suite.addTest(new TestCTDoseAcquisition("TestCTDoseAcquisitionConstructor_WithAllParameters_AxialOnePosition"));
		
		return suite;
	}
	
	protected void setUp() {
	}
	
	protected void tearDown() {
	}
	
	public void TestCTDoseAcquisitionConstructor_WithAllParameters_Helical() {
		
		CTDoseAcquisition ctDoseAcquisition = new CTDoseAcquisition("1",CTScanType.HELICAL,new ScanRange("I","12.750","I","602.750"),"15.51","948.89",CTPhantomType.selectFromDescription("BODY32"));
		
		String expectToString = "\tSeries=1\tHelical\tRange=I12.750-I602.750 mm\tCTDIvol=15.51 mGy\tDLP=948.89 mGycm\tPhantom=BODY32\n";
		String expectToStringPretty = "\tSeries=1\tHelical\tI12.750-I602.750 mm\t15.51 mGy\t948.89 mGycm\tBODY32\n";
		
		assertEquals("Checking SeriesNumber equality","1",ctDoseAcquisition.getSeriesNumber());
		assertEquals("Checking CTScanType equality",CTScanType.HELICAL,ctDoseAcquisition.getScanType());
		assertEquals("Checking CTScanType string equality","Helical",ctDoseAcquisition.getScanType().toString());
		assertTrue("Checking ScanRange equality",new ScanRange("I","12.750","I","602.750").equals(ctDoseAcquisition.getScanRange()));
		assertEquals("Checking CTDIvol equality","15.51",ctDoseAcquisition.getCTDIvol());
		assertEquals("Checking DLP equality","948.89",ctDoseAcquisition.getDLP());
		assertTrue("Checking CTPhantomType equality",CTPhantomType.BODY32.equals(ctDoseAcquisition.getPhantomType()));
		
		assertFalse("Checking specified DLP does not equal DLP computed from range and CTDIvol due to overscan",ctDoseAcquisition.specifiedDLPMatchesDLPFromRangeAndCTDIvol());
		
		assertEquals("Checking toString",expectToString,ctDoseAcquisition.toString());
		assertEquals("Checking toString pretty",expectToStringPretty,ctDoseAcquisition.toString(true));
	}
	
	public void TestCTDoseAcquisitionConstructor_WithAllParameters_Axial() {
		CTDoseAcquisition ctDoseAcquisition = new CTDoseAcquisition("2",CTScanType.AXIAL,new ScanRange("I","24.500","S","33.000"),"234.67","1408.00",CTPhantomType.selectFromDescription("HEAD16"));
		
		String expectToString = "\tSeries=2\tAxial\tRange=I24.500-S33.000 mm\tCTDIvol=234.67 mGy\tDLP=1408.00 mGycm\tPhantom=HEAD16\n";
		String expectToStringPretty = "\tSeries=2\tAxial\tI24.500-S33.000 mm\t234.67 mGy\t1408.00 mGycm\tHEAD16\n";
		
		assertEquals("Checking SeriesNumber equality","2",ctDoseAcquisition.getSeriesNumber());
		assertEquals("Checking CTScanType equality",CTScanType.AXIAL,ctDoseAcquisition.getScanType());
		assertEquals("Checking CTScanType string equality","Axial",ctDoseAcquisition.getScanType().toString());
		assertTrue("Checking ScanRange equality",new ScanRange("I","24.500","S","33.000").equals(ctDoseAcquisition.getScanRange()));
		assertEquals("Checking CTDIvol equality","234.67",ctDoseAcquisition.getCTDIvol());
		assertEquals("Checking DLP equality","1408.00",ctDoseAcquisition.getDLP());
		assertTrue("Checking CTPhantomType equality",CTPhantomType.HEAD16.equals(ctDoseAcquisition.getPhantomType()));
		
		assertFalse("Checking specified DLP does not equal DLP computed from range and CTDIvol due to slice thickness or spacing",ctDoseAcquisition.specifiedDLPMatchesDLPFromRangeAndCTDIvol());
		
		assertEquals("Checking toString",expectToString,ctDoseAcquisition.toString());
		assertEquals("Checking toString pretty",expectToStringPretty,ctDoseAcquisition.toString(true));
	}
	
	public void TestCTDoseAcquisitionConstructor_WithAllParameters_AxialOnePosition() {
		CTDoseAcquisition ctDoseAcquisition = new CTDoseAcquisition("200",CTScanType.AXIAL,new ScanRange("I","254.500","I","254.500"),"46.29","23.10",CTPhantomType.selectFromDescription("BODY32"));
		
		String expectToString = "\tSeries=200\tAxial\tRange=I254.500-I254.500 mm\tCTDIvol=46.29 mGy\tDLP=23.10 mGycm\tPhantom=BODY32\n";
		String expectToStringPretty = "\tSeries=200\tAxial\tI254.500-I254.500 mm\t46.29 mGy\t23.10 mGycm\tBODY32\n";
		
		assertEquals("Checking SeriesNumber equality","200",ctDoseAcquisition.getSeriesNumber());
		assertEquals("Checking CTScanType equality",CTScanType.AXIAL,ctDoseAcquisition.getScanType());
		assertEquals("Checking CTScanType string equality","Axial",ctDoseAcquisition.getScanType().toString());
		assertTrue("Checking ScanRange equality",new ScanRange("I","254.500","I","254.500").equals(ctDoseAcquisition.getScanRange()));
		assertEquals("Checking CTDIvol equality","46.29",ctDoseAcquisition.getCTDIvol());
		assertEquals("Checking DLP equality","23.10",ctDoseAcquisition.getDLP());
		assertTrue("Checking CTPhantomType equality",CTPhantomType.BODY32.equals(ctDoseAcquisition.getPhantomType()));
		
		assertFalse("Checking specified DLP does not equal DLP computed from range and CTDIvol due to slice thickness",ctDoseAcquisition.specifiedDLPMatchesDLPFromRangeAndCTDIvol());
		
		assertEquals("Checking toString",expectToString,ctDoseAcquisition.toString());
		assertEquals("Checking toString pretty",expectToStringPretty,ctDoseAcquisition.toString(true));
	}
	
}
