/* Copyright (c) 2001-2010, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.dose;

import java.io.IOException;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;

import com.pixelmed.dicom.*;

public class CTDose implements RadiationDoseStructuredReport {
	
	protected String dlpTotal;
	protected int totalNumberOfIrradiationEvents;
	protected ScopeOfDoseAccummulation scopeOfDoseAccummulation;
	protected String scopeUID;
	protected ArrayList<CTDoseAcquisition> acquisitions;
	protected CommonDoseObserverContext observerContext;
	protected CompositeInstanceContext compositeInstanceContext;
	protected String startDateTime;
	protected String endDateTime;
	protected String description;
	
	protected StructuredReport sr;
	protected AttributeList list;
	
	public CTDose(String dlpTotal,int totalNumberOfIrradiationEvents,ScopeOfDoseAccummulation scopeOfDoseAccummulation,String scopeUID,String startDateTime,String endDateTime,String description) {
		this.observerContext = null;
		this.compositeInstanceContext = null;
		this.dlpTotal = dlpTotal;
		this.totalNumberOfIrradiationEvents = totalNumberOfIrradiationEvents;
		this.scopeOfDoseAccummulation = scopeOfDoseAccummulation;
		this.scopeUID = scopeUID;
		acquisitions = new ArrayList<CTDoseAcquisition>();
		this.startDateTime = startDateTime;
		this.endDateTime = endDateTime;
		this.description = description;
	}
	
	public CTDose(ScopeOfDoseAccummulation scopeOfDoseAccummulation,String scopeUID,String startDateTime,String endDateTime,String description) {
		this.observerContext = null;
		this.compositeInstanceContext = null;
		this.dlpTotal = null;
		this.totalNumberOfIrradiationEvents = 0;
		this.scopeOfDoseAccummulation = scopeOfDoseAccummulation;
		this.scopeUID = scopeUID;
		acquisitions = new ArrayList<CTDoseAcquisition>();
		this.startDateTime = startDateTime;
		this.endDateTime = endDateTime;
		this.description = description;
	}
	
	public void addAcquisition(CTDoseAcquisition acquisition) {
		acquisitions.add(acquisition);
	}
	
	public CommonDoseObserverContext getObserverContext() { return observerContext; }
	
	public void setObserverContext(CommonDoseObserverContext observerContext) { this.observerContext = observerContext; }
	
	public CompositeInstanceContext getCompositeInstanceContext() { return compositeInstanceContext; }
	
	public void setCompositeInstanceContext(CompositeInstanceContext compositeInstanceContext) { this.compositeInstanceContext = compositeInstanceContext; }
	
	public String getDLPTotal() { return dlpTotal; }
	
	public void setDLPTotal(String dlpTotal) { this.dlpTotal = dlpTotal; }
	
	public int getTotalNumberOfIrradiationEvents() { return totalNumberOfIrradiationEvents ==  0 ? acquisitions.size() : totalNumberOfIrradiationEvents; }
	
	public ScopeOfDoseAccummulation getScopeOfDoseAccummulation() { return scopeOfDoseAccummulation; }
	
	public String getScopeUID() { return scopeUID; }
	
	public int getNumberOfAcquisitions() { return acquisitions.size(); }
	
	public CTDoseAcquisition getAcquisition(int i) { return acquisitions.get(i); }
	
	public String getDLPTotalFromAcquisitions() throws NumberFormatException {
		double dlpTotalFromAcquisitions = 0;
		for (CTDoseAcquisition a : acquisitions) {
			if (a != null) {
				String aDLP = a.getDLP();
				if (aDLP != null) {
					dlpTotalFromAcquisitions+=Double.parseDouble(aDLP);
				}
			}
		}
		java.text.DecimalFormat formatter = (java.text.DecimalFormat)(java.text.NumberFormat.getInstance());
		formatter.setMaximumFractionDigits(2);
		formatter.setMinimumFractionDigits(2);
		formatter.setDecimalSeparatorAlwaysShown(true);		// i.e., a period even if fraction is zero
		formatter.setGroupingUsed(false);					// i.e., no comma at thousands
		String formatted = formatter.format(dlpTotalFromAcquisitions);
//System.err.println("CTDose.getDLPTotalFromAcquisitions(): returns formatted string "+formatted+" for "+Double.toString(dlpTotalFromAcquisitions));
		return formatted;
	}
	
	public boolean specifiedDLPTotalMatchesDLPTotalFromAcquisitions() {
		return (dlpTotal != null && dlpTotal.equals(getDLPTotalFromAcquisitions())) || (dlpTotal == null && getNumberOfAcquisitions() == 0);	// could check "0.00".equals()
	}
	
	public String getStartDateTime() { return startDateTime; }

	public String getEndDateTime() { return endDateTime; }

	public String getDescription() { return description; }

	public String toString() {
		return toString(true,false);
	}
	
	public String toString(boolean detail,boolean pretty) {
		StringBuffer buffer = new StringBuffer();
		buffer.append("Dose");
		if (detail || startDateTime != null) {
			buffer.append("\t");
			if (!pretty) {
				buffer.append("Start=");
			}
			if (pretty && startDateTime != null && startDateTime.length() > 0) {
				try {
					java.util.Date dateTime = new java.text.SimpleDateFormat("yyyyMMddHHmmss").parse(startDateTime);
					String formattedDate = new java.text.SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(dateTime);
					buffer.append(formattedDate);
				}
				catch (java.text.ParseException e) {
					e.printStackTrace(System.err);
				}
			}
			else {
				buffer.append(startDateTime);
			}
		}
		if (detail && !pretty) {
			buffer.append("\tEnd=");
			buffer.append(endDateTime);
		}
		buffer.append("\t");
		if (!pretty) {
			buffer.append("Modality=");
		}
		buffer.append("CT");
		buffer.append("\t");
		if (!pretty) {
			buffer.append("Description=");
		}
		buffer.append(description);
		if (detail && !pretty) {
			buffer.append("\tScope=");
			buffer.append(scopeOfDoseAccummulation);
		}
		if (detail && !pretty) {
			buffer.append("\tUID=");
			buffer.append(scopeUID);
		}
		if (detail && !pretty) {
			buffer.append("\tEvents=");
			buffer.append(Integer.toString(getTotalNumberOfIrradiationEvents()));
		}
		buffer.append("\tDLP Total=");
		buffer.append(dlpTotal);
		buffer.append(" mGycm");
		buffer.append("\n");
		if (detail) {
			for (int i=0; i<acquisitions.size(); ++i) {
				buffer.append(acquisitions.get(i).toString(pretty));
			}
		}
		return buffer.toString();
	}

	public StructuredReport getStructuredReport() throws DicomException {
		if (sr == null) {
			ContentItemFactory cif = new ContentItemFactory();
			ContentItem root = cif.new ContainerContentItem(
				null/*no parent since root*/,null/*no relationshipType since root*/,
				new CodedSequenceItem("113701","DCM","X-Ray Radiation Dose Report"),
				true/*continuityOfContentIsSeparate*/,
				"DCMR","10011");
			ContentItem procedureReported = cif.new CodeContentItem(root,"HAS CONCEPT MOD",new CodedSequenceItem("121058","DCM","Procedure reported"),new CodedSequenceItem("P5-08000","SRT","Computed Tomography X-Ray"));
			cif.new CodeContentItem(procedureReported,"HAS CONCEPT MOD",new CodedSequenceItem("G-C0E8","SRT","Has Intent"),new CodedSequenceItem("R-408C3","SRT","Diagnostic Intent"));
		
			if (observerContext != null) {
				Map<RecordingDeviceObserverContext.Key,ContentItem> cimap = observerContext.getRecordingDeviceObserverContext().getStructuredReportFragment();
				Iterator<RecordingDeviceObserverContext.Key> i = cimap.keySet().iterator();
				while (i.hasNext()) {
					root.addChild(cimap.get(i.next()));
				}
			}
		
			if (startDateTime != null) {
				cif.new DateTimeContentItem(root,"HAS OBS CONTEXT",new CodedSequenceItem("113809","DCM","Start of X-Ray Irradiation"),startDateTime);
			}
			if (endDateTime != null) {
				cif.new DateTimeContentItem(root,"HAS OBS CONTEXT",new CodedSequenceItem("113810","DCM","End of X-Ray Irradiation"),endDateTime);
			}
		
			ContentItem ctAccumulatedDoseData = cif.new ContainerContentItem(root,"CONTAINS",new CodedSequenceItem("113811","DCM","CT Accumulated Dose Data"),true,"DCMR","10012");
			cif.new NumericContentItem(ctAccumulatedDoseData,"CONTAINS",new CodedSequenceItem("113812","DCM","Total Number of Irradiation Events"),Integer.toString(acquisitions.size()),new CodedSequenceItem("{events}","UCUM","1.8","events"));
			cif.new NumericContentItem(ctAccumulatedDoseData,"CONTAINS",new CodedSequenceItem("113813","DCM","CT Dose Length Product Total"),
				(dlpTotal == null ? getDLPTotalFromAcquisitions() : dlpTotal),
				new CodedSequenceItem("mGycm","UCUM","1.8","mGycm"));

			for (CTDoseAcquisition a : acquisitions) {
				if (a != null) {
					ContentItem aci = a.getStructuredReportFragment(root);
					if (aci != null && observerContext != null) {
						DeviceParticipant dp = observerContext.getDeviceParticipant();
						if (dp != null) {
							aci.addChild(dp.getStructuredReportFragment());
						}
						PersonParticipant padmin = observerContext.getPersonParticipantAdministering();
						if (padmin != null) {
							aci.addChild(padmin.getStructuredReportFragment());
						}
					}
				}
			}

			ContentItem scope = cif.new CodeContentItem(root,"HAS OBS CONTEXT",new CodedSequenceItem("113705","DCM","Scope of Accumulation"),scopeOfDoseAccummulation.getCodedSequenceItemForScopeConcept());
			cif.new UIDContentItem(scope,"HAS PROPERTIES",scopeOfDoseAccummulation.getCodedSequenceItemForUIDConcept(),scopeUID);
		
			cif.new CodeContentItem(root,"CONTAINS",new CodedSequenceItem("113854","DCM","Source of Dose Information"),new CodedSequenceItem("113856","DCM","Automated Data Collection"));
			
			if (observerContext != null) {
				PersonParticipant pauth = observerContext.getPersonParticipantAuthorizing();
				if (pauth != null) {
					root.addChild(pauth.getStructuredReportFragment());
				}
			}

			sr = new StructuredReport(root);
			list = null;	// any list previously populated is invalidated by newly generated SR tree; fluche cached version
		}
//System.err.println("CTDose.getStructuredReport():  sr =\n"+sr);
		return sr;
	}
	
	public AttributeList getAttributeList() throws DicomException {
//System.err.println("CTDose.getAttributeList(): compositeInstanceContext.getAttributeList() =\n"+compositeInstanceContext.getAttributeList());
		if (list == null) {
			getStructuredReport();
			list = sr.getAttributeList();
			if (compositeInstanceContext != null) {
//System.err.println("CTDose.getAttributeList(): compositeInstanceContext.getAttributeList() =\n"+compositeInstanceContext.getAttributeList());
				list.putAll(compositeInstanceContext.getAttributeList());
			}
			{ Attribute a = new UniqueIdentifierAttribute(TagFromName.SOPClassUID); a.addValue(SOPClass.XRayRadiationDoseSRStorage); list.put(a); }
			{ Attribute a = new CodeStringAttribute(TagFromName.Modality); a.addValue("SR"); list.put(a); }
		}
//System.err.println("CTDose.getStructuredReport(): AttributeList =\n"+list);
		return list;
	}
	
	public void write(String filename) throws DicomException, IOException {
		getAttributeList();
		{
			java.util.Date currentDateTime = new java.util.Date();
			{ Attribute a = new DateAttribute(TagFromName.InstanceCreationDate); a.addValue(new java.text.SimpleDateFormat("yyyyMMdd").format(currentDateTime)); list.put(a); }
			{ Attribute a = new TimeAttribute(TagFromName.InstanceCreationTime); a.addValue(new java.text.SimpleDateFormat("HHmmss.SSS").format(currentDateTime)); list.put(a); }
			{ Attribute a = new UniqueIdentifierAttribute(TagFromName.InstanceCreatorUID); a.addValue(VersionAndConstants.instanceCreatorUID); list.put(a); }
			
		}
		list.insertSuitableSpecificCharacterSetForAllStringValues();
		list.removeMetaInformationHeaderAttributes();
		FileMetaInformation.addFileMetaInformation(list,TransferSyntax.ExplicitVRLittleEndian,"OURAETITLE");
        list.write(filename);
	}

}
