#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef WIN32
#include <windows.h>
#include <winreg.h>
#endif

#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <gdk/gdk.h>

#include "Socket.h"

#include "callbacks.h"
#include "interface.h"
#include "support.h"

#include "Connection.h"
#include "History.h"

#include "CommandInterpreter.h"
#include "MUDConnectorList.h"
#include "MUD.h"
#include "MUDSelector.h"
#include "PluginHandler.h"
#include "SystemColour.h"
#include "Prefs.h"
#include "color.h"
#include "mudclient.h"
#include "BaseWindow.h"
#include "About.h"
#include "EntityHandler.h"

extern EntityHandler * entities;
extern PluginHandler * phandler;
extern Prefs * preferences;
extern GtkWidget * input_entry;
extern History * history;
extern GtkWidget * connection_dialog_port;
extern GtkWidget * connection_dialog_host;
extern MUDSelector * mud_selector;
extern SystemColour * syscolour;
extern BaseWindow * mainWindow;

GtkWidget * editor = (GtkWidget *)NULL;
GtkWidget * connection_dialog = (GtkWidget *)NULL;

#ifdef TCL
#include "MyTcl.h"
extern Tcl * tcl;
#endif

void history_complete(void);

extern GdkFont * font;

CommandInterpreter * commandInterpreter = (CommandInterpreter *)NULL;

#define UP_ARROW 65362
#define DOWN_ARROW 65364
#define TAB 65289
#define LEFT_CTL 65507
#define RIGHT_CTL 65508
#define ENTER 65293
#define KP_ENTER 65421
#define BACKSPACE 65288
#define ESCAPE 65307
#define F1 65470
#define F2 65471
#define F3 65472
#define F4 65473
#define F5 65474
#define F6 65475
#define F7 65476
#define F8 65477
#define F9 65478
#define F10 65479
#define F11 65480
#define F12 65481

gint papaya_keypress(GtkWidget *, GdkEventKey *);
gint on_window_keypress (GtkWidget * widget, GdkEventKey * event) {

  Connection * c = mainWindow->getCurrentConnection();

  switch(event->keyval) {
  case TAB:

    // Can we find out which widget has focus?
    if (GTK_WIDGET_HAS_FOCUS(c->getVT()->getOutput()))
      return papaya_keypress(widget, event);
    
    if (GTK_WIDGET_HAS_FOCUS(c->getVT()->getInput())) {
      history_complete();
      return 1;
    }
    
    break;
    
  case ESCAPE:

    if (GTK_WIDGET_HAS_FOCUS(c->getVT()->getOutput()))
      return papaya_keypress(widget, event);

    if (c && c->getVT() && c->getVT()->getInput())
      gtk_entry_set_text(GTK_ENTRY(c->getVT()->getInput()), "");
    break;

  case F1:
  case F2:
  case F3:
  case F4:
  case F5:
  case F6:
  case F7:
  case F8:
  case F9:
  case F10:
  case F11:
  case F12:
    mainWindow->switchPage(event->keyval - 65470);
    return 1;
    break;
    
  default:

    if (!c) // Moved to here because it's not needed for the other functions
      return 1;

    if (!commandInterpreter)
      commandInterpreter = new CommandInterpreter();
    if (commandInterpreter->checkMacros(c, event)) {
      return 1;
    }

  }

  return 0;
}

void
on_quit_button_clicked              (GtkButton       *button,
                                        gpointer         user_data, gint i)
{
  exit(1);
}

int initialise_network_mud(MUD *);

void
on_reconnect_button_clicked              (GtkButton       *button,
					   gpointer         user_data, gint i) {
  Connection * c = mainWindow->getCurrentConnection();

  // Need similar check, but to see if dialog is open.
  //  if (!c || c->isClosed())
  //  return;

  if (c)
    c->reconnect();
  else if (Connection::getLastMUD())
    initialise_network_mud(Connection::getLastMUD());
}


void
on_close_window              (GtkButton       *button,
			      gpointer         user_data, gint i)
{
  mainWindow->closeWindow();
}


void
on_disconnect_button_clicked              (GtkButton       *button,
                                        gpointer         user_data, gint i)
{
  Connection * conn = mainWindow->getCurrentConnection();
  if (!conn || conn->isClosed())
    return;

  delete conn;
}

void create_player_trigger_dialog();

void on_entities_button_clicked(GtkButton * button, gpointer user_data, gint i) {
  entities->edit();
}

void
on_swallow              (GtkButton       *button,
			 gpointer         user_data, gint i) {
#ifndef WIN32
  mainWindow->swallow();
#endif
}


void
on_copyover_button_clicked              (GtkButton       *button,
                                        gpointer         user_data, gint i)
{
  copyover();
}
void
on_connect_button_clicked              (GtkButton       *button,
                                        gpointer         user_data, gint i)
{

  if (connection_dialog) {
    gtk_widget_show(connection_dialog);
    gtk_window_present((GtkWindow *)connection_dialog);
    return;
  }

  connection_dialog = create_connection_dialog();
  gtk_widget_show(connection_dialog);
  gtk_widget_grab_focus(connection_dialog_host);

}



gint
on_connect_dialog_connect_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{

  char host[80];
  int port;

  if (strlen(gtk_entry_get_text(GTK_ENTRY(connection_dialog_host))) == 0) {
    // No host specified.
    new Message("Error", "You must specify a host name to connect to.", false);
    return 1;
  }
  
  if (strlen(gtk_entry_get_text(GTK_ENTRY(connection_dialog_port))) == 0) {
    // No port specified.
    new Message("Error", "You must specify a port number to connect to.", false);
    return 1;
  }

  port = atoi(gtk_entry_get_text(GTK_ENTRY(connection_dialog_port)));
  sprintf(host, "%s", gtk_entry_get_text(GTK_ENTRY(connection_dialog_host)));

  gtk_widget_hide(connection_dialog);
  gtk_widget_destroy(connection_dialog);
  connection_dialog = (GtkWidget *)NULL;

  initialise_network(host, port);
  return 1;
}


gint on_connect_dialog_cancel_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{

  gtk_widget_hide(connection_dialog);
  gtk_widget_destroy(connection_dialog);
  connection_dialog = (GtkWidget *)NULL;
  return 1;
}

gint on_connect_dialog_delete_event(GtkWidget * window, GdkEvent * event, gpointer data) {
  return on_connect_dialog_cancel_button_clicked((GtkButton *)window, data);
}

void setFontSamples();

void on_help_userguide_clicked(GtkButton * button, gpointer user_data, gint i) {
  char url[16384];

#if defined(WIN32) && !defined(__MINGW32__)
  snprintf(url, 16384, "%s/share/papaya/manual.pdf", getPrefix());

  // Sanitise the URL.
  char * pc = url;
  while (*pc != '\0') {
	if (*pc == '\\')
      *pc = '/';
	pc++;
  }

#else
  snprintf(url, 16384, "file://%s/share/papaya/manual.pdf", getPrefix());
#endif
  browser_open(url);
}

void
on_help_plugins_clicked (GtkButton       *button,
                         gpointer         user_data, gint i) {

  phandler->displayAvailablePlugins();
}


void on_connection_dialog_port_activate(GtkEditable     *editable,
                                        gpointer         user_data) {

  // Select OK pressed.
  on_connect_dialog_connect_button_clicked((GtkButton *)NULL, (gpointer)NULL);

};

void history_complete(void) {
  GtkWidget * input;
  Connection * cnect;
  VT * vt;
  History * history;
  char * target;
  char * match;
  int position;

  if ((cnect = mainWindow->getCurrentConnection()) == NULL)
    return;

  if ((vt = cnect->getVT()) == NULL)
    return;

  if ((history = cnect->getHistory()) == NULL)
    return;

  if ((input = vt->getInput()) == NULL)
    return;

  position = gtk_editable_get_position(GTK_EDITABLE(input));

  target = gtk_editable_get_chars(GTK_EDITABLE(input), 0, position);
  
  if (target == NULL)
    return;

  char * tmp = cnect->queryPreferences()->getPreference("HistoryHonourNull");
  int history_honour_null = atoi(tmp);

  match = history->findMatch(target, (bool) history_honour_null);
  
  if (match != NULL)
  {
    gtk_entry_set_text(GTK_ENTRY(input), match);
    gtk_editable_set_position(GTK_EDITABLE(input), position);
    gtk_editable_select_region(GTK_EDITABLE(input), -1, position);
  } else {
    gtk_entry_set_text(GTK_ENTRY(input), target);
    gtk_editable_set_position(GTK_EDITABLE(input), position);
  }

  g_free(target);

  return;
}

void on_about_clicked(GtkMenuItem *, gpointer) {
  new About();
}

GtkWidget * gtk_button_new_with_label_and_pixmap(GtkWidget * parent, gchar * text, gchar * filename) {

  GtkWidget *box1;
  GtkWidget *label;
  GtkWidget *pixmapwid = NULL;
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  GtkStyle *style;
  
  box1 = gtk_vbox_new(FALSE, 2);

  gtk_container_set_border_width (GTK_CONTAINER (box1), 2);

  /* Get the style of the button to get the
   * background color. */
  style = gtk_widget_get_style(parent);
  
  /* Now on to the xpm stuff */
  pixmap = gdk_pixmap_create_from_xpm (parent->window, &mask,
				       &style->bg[GTK_STATE_NORMAL],
				       filename);
  if (pixmap)
    pixmapwid = gtk_pixmap_new (pixmap, mask);
  
  /* Create a label for the button */
  label = gtk_label_new (text);
  
  /* Pack the pixmap and label into the box */
  if (pixmap)
    gtk_box_pack_start (GTK_BOX (box1), pixmapwid, FALSE, FALSE, 3);
  gtk_box_pack_start (GTK_BOX (box1), label, FALSE, FALSE, 3);
  
  // Display the widgets.
  if (pixmap)
    gtk_widget_show(pixmapwid);
  gtk_widget_show(label);
  gtk_widget_show(box1);
  
  GtkWidget * button = gtk_button_new();
  gtk_container_add(GTK_CONTAINER(button), box1);

  return button;
}

void toggle_other_sensitive(GtkToggleButton * widget, gpointer data) {
  GtkWidget * target = (GtkWidget *) data;

  if (GTK_WIDGET_SENSITIVE(target))
    gtk_widget_set_sensitive(target, FALSE);
  else
    gtk_widget_set_sensitive(target, TRUE);
}

char * getPrefix() {
#ifdef WIN32
  HKEY hKey;

#define PATH "SOFTWARE\\GNU\\Papaya"
#define KEY "root"

  int result = RegOpenKeyEx(HKEY_CURRENT_USER, PATH, 0, KEY_READ, &hKey);
  if (result == ERROR_SUCCESS) { // Success
    DWORD type;
    static BYTE data[1024];
    DWORD data_len = 1024;
    
    if (RegQueryValueEx(hKey, KEY, 0, &type, data, &data_len) == ERROR_SUCCESS) {
	  RegCloseKey(hKey);
      return (char *)data;
    }  
  } else {
	RegCloseKey(hKey);
    int result = RegOpenKeyEx(HKEY_LOCAL_MACHINE, PATH, 0, KEY_READ, &hKey);
    if (result == ERROR_SUCCESS) { // Success
	  DWORD type;
      static BYTE data[1024];
      DWORD data_len = 1024;
      
      if (RegQueryValueEx(hKey, KEY, 0, &type, data, &data_len) == ERROR_SUCCESS) {
		RegCloseKey(hKey);
    	return (char *)data;
      }
    }
  }

  RegCloseKey(hKey);

//#  if defined(__MINGW32__)
//  return PREFIX;
//#  else
  return "../";
//#  endif

#else
  return PREFIX;
#endif //Win32
}
