#include "mycrypt.h"

#ifdef MDH

/* format of encrypt message

offset    | size  |  description
---------------------------------
   0      |   1   |  cipher ID
   1      |   1   |  hash ID
   2      |   4   |  length of DH public key
   6      |   c   |  DH public key
   6+c    |   d   |  Cipher CTR IV value [size determined by block cipher chosen]
   6+c+d  |   4   |  Length of ciphertext in bytes
   10+c+d |   e   |  ciphertext
--------------------------------- 
*/

int dh_encrypt(const unsigned char *in,  unsigned long len, 
                     unsigned char *out, unsigned long *outlen,
                     prng_state *prng, int wprng, int cipher, int hash, 
                     dh_key *key)
{
    unsigned char pub_expt[1536], dh_shared[1536], IV[MAXBLOCKSIZE], skey[MAXBLOCKSIZE];
    dh_key pubkey;
    unsigned long x, y, z, hashsize, blocksize, pubkeysize;
    int keysize;
    symmetric_CTR ctr;

    /* check that wprng/cipher/hash are not invalid */
    if (prng_is_valid(wprng) == CRYPT_ERROR ||
        hash_is_valid(hash)  == CRYPT_ERROR ||
        cipher_is_valid(cipher) == CRYPT_ERROR) {
       return CRYPT_ERROR;
    }

    /* make a random key and export the public copy */
    if (dh_make_key(dh_get_size(key), prng, wprng, &pubkey) == CRYPT_ERROR) {
       return CRYPT_ERROR;
    }

    pubkeysize = sizeof(pub_expt);
    if (dh_export(pub_expt, &pubkeysize, PK_PUBLIC, &pubkey) == CRYPT_ERROR) {
       dh_free(&pubkey);
       return CRYPT_ERROR;
    }
    
    /* now check if the out buffer is big enough */
    if (*outlen < (13 + pubkeysize + 
                   cipher_descriptor[cipher].block_length + len)) {
       crypt_error = "Buffer overflow in dh_encrypt().";
       dh_free(&pubkey);
       return CRYPT_ERROR;
    }

    /* make random key */
    blocksize = cipher_descriptor[cipher].block_length;
    hashsize  = hash_descriptor[hash].hashsize;
    keysize = hashsize;
    if (cipher_descriptor[cipher].keysize(&keysize) == CRYPT_ERROR) {
       crypt_error = "Invalid cipher and hash combination in dh_encrypt().";
       dh_free(&pubkey);
       return CRYPT_ERROR;
    }

    x = sizeof(dh_shared);
    if (dh_shared_secret(&pubkey, key, dh_shared, &x) == CRYPT_ERROR) {
       dh_free(&pubkey);
       return CRYPT_ERROR;
    }
    dh_free(&pubkey);

    z = sizeof(skey);
    if (hash_memory(hash, dh_shared, x, skey, &z) == CRYPT_ERROR) {
       return CRYPT_ERROR;
    }

    /* make up IV */
    if (prng_descriptor[wprng].read(IV, cipher_descriptor[cipher].block_length, prng) != 
        cipher_descriptor[cipher].block_length) {
       crypt_error = "Error reading PRNG in dh_encrypt().";
       return CRYPT_ERROR;
    }

    /* setup CTR mode */
    if (ctr_start(cipher, IV, skey, keysize, 0, &ctr) == CRYPT_ERROR) 
       return CRYPT_ERROR;

    /* output header */
    y = PACKET_SIZE;
 
    /* size of cipher name and the name itself */
    out[y++] = cipher_descriptor[cipher].ID;

    /* size of hash name and the name itself */
    out[y++] = hash_descriptor[hash].ID;

    /* length of DH pubkey and the key itself */
    STORE32L(pubkeysize, out+y);
    y += 4;
    for (x = 0; x < pubkeysize; x++, y++) {
        out[y] = pub_expt[x];
    }

    /* cipher IV */
    for (x = 0; x < blocksize; x++, y++) {
        out[y] = IV[x];
    }

    /* length of ciphertext */
    STORE32L(len, out+y);
    y += 4;

    /* encrypt the message */
    ctr_encrypt(in, out+y, len, &ctr);
    y += len;

    /* store header */
    packet_store_header(out, PACKET_SECT_DH, PACKET_SUB_ENCRYPTED, y);

    /* clean up */
    zeromem(pub_expt, sizeof(pub_expt));
    zeromem(dh_shared, sizeof(dh_shared));
    zeromem(skey, sizeof(skey));
    zeromem(IV, sizeof(IV));
    zeromem(&ctr, sizeof(ctr));
    *outlen = y;
    return CRYPT_OK;
}

int dh_decrypt(const unsigned char *in,  unsigned long len, 
                     unsigned char *out, unsigned long *outlen, 
                     dh_key *key)
{
   unsigned char shared_secret[1536], skey[MAXBLOCKSIZE];
   unsigned long x, y, z, res, hashsize, blocksize;
   int hash, cipher, keysize;
   dh_key pubkey;
   symmetric_CTR ctr;

   /* right key type? */
   if (key->type != PK_PRIVATE) {
      crypt_error = "Cannot decrypt with public key in dh_decrypt().";
      return CRYPT_ERROR;
   }

   /* is header correct? */
   if (packet_valid_header((unsigned char *)in, PACKET_SECT_DH, PACKET_SUB_ENCRYPTED) == CRYPT_ERROR) {
      crypt_error = "Invalid packet for dh_decrypt().";
      return CRYPT_ERROR;
   }

   /* now lets get the cipher name */
   y = PACKET_SIZE;
   cipher = find_cipher_id(in[y++]);
   if (cipher == -1) {
      crypt_error = "Cipher not found in descriptor table in dh_decrypt().";
      return CRYPT_ERROR;
   }

   /* now lets get the hash name */
   hash = find_hash_id(in[y++]);
   if (hash == -1) {
      crypt_error = "hash not found in descriptor table in dh_decrypt().";
      return CRYPT_ERROR;
   }

   /* common values */
   blocksize = cipher_descriptor[cipher].block_length;
   hashsize  = hash_descriptor[hash].hashsize;
   keysize = hashsize;
   if (cipher_descriptor[cipher].keysize(&keysize) == CRYPT_ERROR) {
      crypt_error = "Invalid cipher and hash combination in dh_decrypt().";
      return CRYPT_ERROR;
   }

   /* get public key */
   LOAD32L(x, in+y);
   y += 4;
   if (dh_import(in+y, &pubkey) == CRYPT_ERROR) {
      dh_free(&pubkey);
      return CRYPT_ERROR;
   }
   y += x;

   /* make shared key */
   x = sizeof(shared_secret);
   if (dh_shared_secret(key, &pubkey, shared_secret, &x) == CRYPT_ERROR) {
      dh_free(&pubkey);
      return CRYPT_ERROR;
   }
   dh_free(&pubkey);

   z = sizeof(skey);
   if (hash_memory(hash, shared_secret, x, skey, &z) == CRYPT_ERROR) {
      return CRYPT_ERROR;
   }

   /* setup CTR mode */
   if (ctr_start(cipher, in+y, skey, keysize, 0, &ctr) == CRYPT_ERROR) {
      res = CRYPT_ERROR;
      goto done;
   }

   /* skip over the IV */
   y += blocksize;

   /* get length */
   LOAD32L(len,in+y);
   y += 4;

   /* buffer overflow? */
   if (len > *outlen) {
      crypt_error = "Buffer overrun in dh_decrypt().";
      res = CRYPT_ERROR;
      goto done;
   }

   /* decrypt message */
   ctr_decrypt(in+y, out, len, &ctr);
   *outlen = len;
   
   res = CRYPT_OK;
done:
   zeromem(shared_secret, sizeof(shared_secret));
   zeromem(skey, sizeof(skey));
   zeromem(&ctr, sizeof(ctr));
   return res;
}

int dh_sign(const unsigned char *in,  unsigned long inlen,
                  unsigned char *out, unsigned long *outlen,
                  int hash, prng_state *prng, int wprng,
                  dh_key *key)
{
   mp_int a, b, k, m, g, p, p1, tmp;
   unsigned char buf[1536], md[MAXBLOCKSIZE];
   unsigned long x, y, z;
   int res;

   /* check parameters */
   if (key->type != PK_PRIVATE) {
      crypt_error = "Cannot sign with public key in dh_sign().";
      return CRYPT_ERROR;
   }

    if (prng_is_valid(wprng) == CRYPT_ERROR ||
        hash_is_valid(hash)  == CRYPT_ERROR) {
       return CRYPT_ERROR;
    }

   /* hash the message */
   z = sizeof(md) - 1;
   md[0] = 0;
   if (hash_memory(hash, in, inlen, md+1, &z) == CRYPT_ERROR) {
      return CRYPT_ERROR;
   }

   /* make up a random value k,
    * since the order of the group is prime
    * we need not check if gcd(k, r) is 1 
    */
   buf[0] = 0;
   if (prng_descriptor[wprng].read(buf+1, sets[key->idx].size-1, prng) != 
       (unsigned long)(sets[key->idx].size-1)) {
      crypt_error = "Error reading PRNG in dh_sign().";
      return CRYPT_ERROR;
   }

   /* init bignums */
   if (mp_init_multi(&a, &b, &k, &m, &p, &g, &p1, &tmp, NULL) != MP_OKAY) { 
      crypt_error = "Out of memory in dh_sign()."; 
      return CRYPT_ERROR;
   }

   /* load k and m */
   if (mp_read_raw(&m, md,  1+hash_descriptor[hash].hashsize) != MP_OKAY) { goto error; }
   if (mp_read_raw(&k, buf, sets[key->idx].size) != MP_OKAY) { goto error; }

   /* load g, p and p1 */
   if (mp_read_radix(&g, sets[key->idx].base, 10) != MP_OKAY) { goto error; }
   if (mp_read_radix(&p, sets[key->idx].prime, 10) != MP_OKAY) { goto error; }
   if (mp_sub_d(&p, 1, &p1) != MP_OKAY) { goto error; }
   if (mp_div_2(&p1, &p1) != MP_OKAY) { goto error; }                   /* p1 = (p-1)/2 */

   /* now get a = g^k mod p */
   if (mp_exptmod(&g, &k, &p, &a) != MP_OKAY) { goto error; }

   /* now find M = xa + kb mod p1 or just b = (M - xa)/k mod p1 */
   if (mp_invmod(&k, &p1, &k) != MP_OKAY) { goto error; }               /* k = 1/k mod p1 */
   if (mp_mulmod(&a, &key->x, &p1, &tmp) != MP_OKAY) { goto error; }    /* tmp = xa */
   if (mp_submod(&m, &tmp, &p1, &tmp) != MP_OKAY) { goto error; }       /* tmp = M - xa */
   if (mp_mulmod(&k, &tmp, &p1, &b) != MP_OKAY) { goto error; }         /* b = (M - xa)/k */

   /* store header  */
   y = PACKET_SIZE;

   /* store length and name of hash */
   buf[y++] = hash_descriptor[hash].ID;

   /* now store them both (a,b) */
   x = mp_raw_size(&a);
   STORE32L(x, buf+y);  y += 4;
   mp_toraw(&a, buf+y); y += x;

   x = mp_raw_size(&b);
   STORE32L(x, buf+y);  y += 4;
   mp_toraw(&b, buf+y); y += x;

   /* check if size too big */
   if (*outlen < y) {
      crypt_error = "Buffer overflow in dh_sign().";
      res = CRYPT_ERROR;
      goto done;
   }

   /* store header */
   packet_store_header(buf, PACKET_SECT_DH, PACKET_SUB_SIGNED, y);

   /* store it */
   memcpy(out, buf, y);
   zeromem(md, sizeof(md));
   zeromem(buf, sizeof(buf));
   *outlen = y;

   res = CRYPT_OK;
   goto done;
error:
   res = CRYPT_ERROR;
   crypt_error = "Out of memory in dh_sign().";
done:
   mp_clear_multi(&tmp, &p1, &g, &p, &m, &k, &b, &a, NULL);
   return res;
}

int dh_verify(const unsigned char *sig, const unsigned char *msg, 
                    unsigned long inlen, int *stat, 
                    dh_key *key)
{
   mp_int a, b, p, g, m, tmp;
   unsigned char md[MAXBLOCKSIZE];
   unsigned long x, y, z;
   int hash, res;

   /* default to invalid */
   *stat = 0;

   /* header ok? */
   if (packet_valid_header((unsigned char *)sig, PACKET_SECT_DH, PACKET_SUB_SIGNED) == CRYPT_ERROR) {
      crypt_error = "Invalid packet for dh_verify().";
      return CRYPT_ERROR;
   }

   /* get hash out of packet */
   y = PACKET_SIZE;
   hash = find_hash_id(sig[y++]);

   if (hash == -1) {
      crypt_error = "Invalid hash found in packet in dh_verify().";
      return CRYPT_ERROR;
   }

   /* hash the message */
   md[0] = 0;
   z = sizeof(md) - 1;
   if (hash_memory(hash, msg, inlen, md+1, &z) == CRYPT_ERROR) {
      return CRYPT_ERROR;
   }

   /* init all bignums */
   if (mp_init_multi(&a, &p, &b, &g, &m, &tmp, NULL) != MP_OKAY) { 
      crypt_error = "Out of memory in dh_verify()."; 
      return CRYPT_ERROR;
   }

   /* load a and b */
   LOAD32L(x, sig+y);
   y += 4;
   if (mp_read_raw(&a, (unsigned char *)sig+y, x) != MP_OKAY) { goto error; }
   y += x;

   LOAD32L(x, sig+y);
   y += 4;
   if (mp_read_raw(&b, (unsigned char *)sig+y, x) != MP_OKAY) { goto error; }
   y += x;

   /* load p and g */
   if (mp_read_radix(&p, sets[key->idx].prime, 10) != MP_OKAY) { goto error; }
   if (mp_read_radix(&g, sets[key->idx].base, 10) != MP_OKAY) { goto error; }

   /* load m */
   if (mp_read_raw(&m, md, hash_descriptor[hash].hashsize + 1) != MP_OKAY) { goto error; }

   /* find g^m mod p */
   if (mp_exptmod(&g, &m, &p, &m) != MP_OKAY) { goto error; }       /* m = g^m mod p */

   /* find y^a * a^b */
   if (mp_exptmod(&key->y, &a, &p, &tmp) != MP_OKAY) { goto error; }/* tmp = y^a mod p */
   if (mp_exptmod(&a, &b, &p, &a) != MP_OKAY) { goto error; }       /* a = a^b mod p */
   if (mp_mulmod(&a, &tmp, &p, &a) != MP_OKAY) { goto error; }      /* a = y^a * a^b mod p */

   /* y^a * a^b == g^m ??? */
   if (mp_cmp(&a, &m) == 0)
      *stat = 1;

   /* clean up */
   res = CRYPT_OK;
   goto done;
error:
   res = CRYPT_ERROR;
   crypt_error = "Out of memory in dh_verify().";
done:
   mp_clear_multi(&tmp, &m, &g, &p, &b, &a, NULL);
   zeromem(md, sizeof(md));
   return res;
}

int dh_encrypt_key(const unsigned char *inkey, unsigned long keylen,
                         unsigned char *out,  unsigned long *len,
                         prng_state *prng, int wprng, int hash,
                         dh_key *key)
{
    unsigned char pub_expt[1536], dh_shared[1536], skey[MAXBLOCKSIZE];
    dh_key pubkey;
    unsigned long x, y, z, hashsize, pubkeysize;

    /* check that wprng/hash are not invalid */
    if (prng_is_valid(wprng) == CRYPT_ERROR ||
        hash_is_valid(hash)  == CRYPT_ERROR) {
       return CRYPT_ERROR;
    }

    if (keylen > hash_descriptor[hash].hashsize)  {
        crypt_error = "Key size too big in dh_encrypt_key().";
        return CRYPT_ERROR;
    }

    /* make a random key and export the public copy */
    if (dh_make_key(dh_get_size(key), prng, wprng, &pubkey) == CRYPT_ERROR) {
       return CRYPT_ERROR;
    }

    pubkeysize = sizeof(pub_expt);
    if (dh_export(pub_expt, &pubkeysize, PK_PUBLIC, &pubkey) == CRYPT_ERROR) {
       dh_free(&pubkey);
       return CRYPT_ERROR;
    }

    /* now check if the out buffer is big enough */
    if (*len < (12 + pubkeysize + keylen)) {
       crypt_error = "Buffer overflow in dh_encrypt_key().";
       dh_free(&pubkey);
       return CRYPT_ERROR;
    }

    /* make random key */
    hashsize  = hash_descriptor[hash].hashsize;

    x = sizeof(dh_shared);
    if (dh_shared_secret(&pubkey, key, dh_shared, &x) == CRYPT_ERROR) {
       dh_free(&pubkey);
       return CRYPT_ERROR;
    }
    dh_free(&pubkey);

    z = sizeof(skey);
    if (hash_memory(hash, dh_shared, x, skey, &z) == CRYPT_ERROR) {
       return CRYPT_ERROR;
    }

    /* output header */
    y = PACKET_SIZE;

    /* size of hash name and the name itself */
    out[y++] = hash_descriptor[hash].ID;

    /* length of DH pubkey and the key itself */
    STORE32L(pubkeysize, out+y);
    y += 4;
    for (x = 0; x < pubkeysize; x++, y++)
        out[y] = pub_expt[x];

    /* Store the encrypted key */
    STORE32L(keylen, out+y);
    y += 4;

    for (x = 0; x < keylen; x++, y++)
      out[y] = skey[x] ^ inkey[x];

    /* store header */
    packet_store_header(out, PACKET_SECT_DH, PACKET_SUB_ENC_KEY, y);

    /* clean up */
    zeromem(pub_expt, sizeof(pub_expt));
    zeromem(dh_shared, sizeof(dh_shared));
    zeromem(skey, sizeof(skey));
    *len = y;
    return CRYPT_OK;
}

int dh_decrypt_key(const unsigned char *in,  unsigned long len,
                   unsigned char *outkey, unsigned long *keylen,
                   dh_key *key)
{
   unsigned char shared_secret[1536], skey[MAXBLOCKSIZE];
   unsigned long x, y, z, res, hashsize, keysize;
   int hash;
   dh_key pubkey;

   /* right key type? */
   if (key->type != PK_PRIVATE) {
      crypt_error = "Cannot decrypt with public key in dh_decrypt_key().";
      return CRYPT_ERROR;
   }

   /* is header correct? */
   if (packet_valid_header((unsigned char *)in, PACKET_SECT_DH, PACKET_SUB_ENC_KEY) == CRYPT_ERROR)  {
      crypt_error = "Invalid packet for dh_decrypt().";
      return CRYPT_ERROR;
   }

   /* now lets get the hash name */
   y = PACKET_SIZE;
   hash = find_hash_id(in[y++]);
   if (hash == -1) {
      crypt_error = "hash not found in descriptor table in dh_decrypt().";
      return CRYPT_ERROR;
   }

   /* common values */
   hashsize  = hash_descriptor[hash].hashsize;

   /* get public key */
   LOAD32L(x, in+y);
   y += 4;
   if (dh_import(in+y, &pubkey) == CRYPT_ERROR) {
      dh_free(&pubkey);
      return CRYPT_ERROR;
   }
   y += x;

   /* make shared key */
   x = sizeof(shared_secret);
   if (dh_shared_secret(key, &pubkey, shared_secret, &x) == CRYPT_ERROR) {
      dh_free(&pubkey);
      return CRYPT_ERROR;
   }
   dh_free(&pubkey);

   z = sizeof(skey);
   if (hash_memory(hash, shared_secret, x, skey, &z) == CRYPT_ERROR) {
      return CRYPT_ERROR;
   }

   /* load in the encrypted key */
   LOAD32L(keysize, in+y);
   if (keysize > *keylen)
     {
       crypt_error = "Buffer overrun in dh_decrypt_key().";
       res = CRYPT_ERROR;
       goto done;
     }
   y += 4;

   *keylen = keysize;

   for (x = 0; x < keysize; x++, y++)
     outkey[x] = skey[x] ^ in[y];

   res = CRYPT_OK;
done:
   zeromem(shared_secret, sizeof(shared_secret));
   zeromem(skey, sizeof(skey));
   return res;
}


#endif

static char *ID2_TAG = "dh_sys.c"; 
 
