///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/ViewportPanel.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/input/ViewportInputHandler.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/data/DataSetManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/actions/ActionManager.h>
#include <core/rendering/RenderSettings.h>

namespace Core {

/******************************************************************************
* The constructor of the viewport panel class.
******************************************************************************/
ViewportPanel::ViewportPanel(QWidget* parent) : Window3DContainer(parent)
{
	// The mouse cursor must updated each time when a new input mode becomes active.
	connect(&VIEWPORT_INPUT_MANAGER, SIGNAL(inputModeChanged(ViewportInputHandler*, ViewportInputHandler*)), this, SLOT(updateViewportCursor()));

	// Repaint the viewport borders if the animation mode has been activated.
	connect(&ANIM_MANAGER, SIGNAL(animationModeChanged(bool)), this, SLOT(update()));

	// Active the new viewport layout as soon as a new scene file is loaded.
	connect(&DATASET_MANAGER, SIGNAL(dataSetReset(DataSet*)), this, SLOT(reset(DataSet*)));

	// Layout viewports if the render settings have changed (to reflect new aspect ratio).
	connect(&_renderSettingsListener, SIGNAL(notificationMessage(RefTargetMessage*)), this, SLOT(renderSettingsChanged(RefTargetMessage*)));
}

/******************************************************************************
* Creates a new viewport.
******************************************************************************/
Viewport* ViewportPanel::addViewport()
{
	Viewport* vp = new Viewport(this);
	_viewports.push_back(vp);
	return vp;
}

/******************************************************************************
* Removes and destroys a viewport.
******************************************************************************/
void ViewportPanel::removeViewport(Viewport* vp)
{
	CHECK_POINTER(vp);
	OVITO_ASSERT(_viewports.contains(vp));
	_viewports.remove(_viewports.indexOf(vp));
	delete vp;
}

/******************************************************************************
* Resets the viewport panel to the state saved in the current data set.
******************************************************************************/
void ViewportPanel::reset(DataSet* newDataSet)
{
	if(newDataSet) {
		newDataSet->viewportConfig()->restoreConfiguration();
		layoutViewports();

		// Listen for changes of the render settings.
		_renderSettingsListener.setTarget(newDataSet->renderSettings());
	}
}

/******************************************************************************
* Updates the cursor for each viewport.
* The cursor is taken from the active viewport input handler.
******************************************************************************/
void ViewportPanel::updateViewportCursor()
{
	ViewportInputHandler* handler = VIEWPORT_INPUT_MANAGER.currentHandler();
	if(handler && handler->temporaryNavigationMode()) handler = handler->temporaryNavigationMode();
	if(!handler)
		unsetCursor();
	else
		setCursor(handler->getCursor());
}

/******************************************************************************
* Is called when the render settings have been changed.
******************************************************************************/
void ViewportPanel::renderSettingsChanged(RefTargetMessage* msg)
{
	if(msg->type() == REFTARGET_CHANGED) {
		layoutViewports();
	}
}

/******************************************************************************
* Renders the borders of the viewports.
******************************************************************************/
void ViewportPanel::paintEvent(QPaintEvent* event)
{
	// Render border of active viewport.
	Viewport* vp = VIEWPORT_MANAGER.activeViewport();
	if(!vp || vp->isHidden()) return;

	QPainter painter(this);
	// Choose a color for the viewport border.
	ColorA borderColor;
	if(ANIM_MANAGER.animationMode())
		borderColor = Viewport::getVPColor(Viewport::COLOR_ANIMATION_MODE);
	else
		borderColor = Viewport::getVPColor(Viewport::COLOR_ACTIVE_VIEWPORT_BORDER);

	painter.setPen(borderColor);
	QRect rect = vp->geometry();
	rect.adjust(-1, -1, 0, 0);
	painter.drawRect(rect);
	rect.adjust(-1, -1, 1, 1);
	painter.drawRect(rect);
}

/******************************************************************************
* Handles size event for the window.
* Does the actual calculation of its children's positions and sizes.
******************************************************************************/
void ViewportPanel::resizeEvent(QResizeEvent* event)
{
	layoutViewports();
	VIEWPORT_MANAGER.updateViewports(true);
}

/******************************************************************************
* Performs the layout of the viewports.
* Does the actual calculation of its children's positions and sizes.
******************************************************************************/
void ViewportPanel::layoutViewports()
{
	// Count the number of visible window.
	int nvisible = 0;
	Q_FOREACH(Viewport* viewport, viewports()) {
		if(!viewport->isHidden()) nvisible++;
	}
	if(nvisible == 0) return;

	// Compute number of rows/columns
	int rows = (int)(sqrt((double)nvisible) + 0.5);
	int columns = (nvisible+rows-1) / rows;

	// Get client rect.
	QRect clientRect = rect();

	// Position items.
	int count = 0;
	Q_FOREACH(Viewport* viewport, viewports()) {
		if(viewport->isHidden()) continue;

		int x = count%columns;
		int y = count/columns;
		QRect rect(clientRect.topLeft(), QSize(0,0));
		rect.translate(clientRect.width() * x / columns, clientRect.height() * y / rows);
		rect.setWidth((clientRect.width() * (x+1) / columns) - rect.x());
		rect.setHeight((clientRect.height() * (y+1) / rows) - rect.y());
		rect.adjust(2,2,-2,-2);

		if(viewport->settings()->renderFrameShown()) {
			// Setup a viewport rectangle that has the same
			// aspect ratio as the rendering image.
			RenderSettings* renderSettings = DATASET_MANAGER.currentSet()->renderSettings();
			if(renderSettings && rect.width() > 0) {
				FloatType renderAspectRatio = (FloatType)renderSettings->imageHeight() / (FloatType)renderSettings->imageWidth();
				FloatType windowAspectRatio = (FloatType)rect.height() / (FloatType)rect.width();
				if(renderAspectRatio < windowAspectRatio) {
					int frameHeight = max((int)(rect.width() * renderAspectRatio), 1);
					rect = QRect(rect.x(), rect.y() + (rect.height() - frameHeight) / 2, rect.width(), frameHeight);
				}
				else {
					int frameWidth = max((int)(rect.height() / renderAspectRatio), 1);
					rect = QRect(rect.x() + (rect.width() - frameWidth) / 2, rect.y(), frameWidth, rect.height());
				}
			}
		}

		viewport->setGeometry(rect);

		count++;
	}
}


};
