///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file ParameterUI.h 
 * \brief Contains the definition of the Core::ParameterUI class and some derived classes. 
 */

#ifndef __OVITO_PARAMETER_UI_H
#define __OVITO_PARAMETER_UI_H

#include <core/Core.h>
#include <core/reference/RefTarget.h>

namespace Core {

/**
 * \brief Base class for UI components that allow the user to edit a parameter
 *        of a RefTarget derived object in the PropertiesEditor.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT ParameterUI : public RefMaker
{
public:
	/// \brief Constructor.
	/// \param parentEditor The editor in which this parameter UI is used. This becomes the parent of this object.
	///
	/// The parameter UI is automatically deleted when the editor is deleted. 
	ParameterUI(PropertiesEditor* parentEditor);

	/// \brief Destructor.
	/// \note The destructor must clear all references since this UI object
	///       might have been deleted via the delete operator and not via
	///       the PluginClass::autoDeleteObject() method which should normaly be used for PluginClass derived classes.
	virtual ~ParameterUI() { clearAllReferences(); }	
	
	/// \brief Gets the object whose parameter is being edited/shown in this parameter UI.
	/// \return The current object being edited.
	/// \sa setEditObject()
	RefTarget* editObject() const { return _editObject; }
	
	/// \brief Returns a pointer to the properties editor this parameter UI belongs to.
	/// \return Th editor in which this parameter UI is used.
	PropertiesEditor* editor() const { return (PropertiesEditor*)this->parent(); }

	/// \brief Returns the enabled state of the UI.
	/// \return \c true if this parameter's value can be changed by the user;
	///         \c false otherwise.
	/// \sa setEnabled()
	bool isEnabled() const { return _enabled; }

public:	
	
	Q_PROPERTY(RefTarget editObject READ editObject)		
	Q_PROPERTY(bool isEnabled READ isEnabled WRITE setEnabled)
		
public Q_SLOTS:	
		
	/// \brief This method is called when a new editable object has been assigned to the properties owner 
	///       this parameter UI belongs to.
	/// 
	/// The parameter UI should react to this change appropriately and
	/// show the properties value for the new edit object in the UI. The default implementation
	/// of this method just calls updateUI() to reflect the change.
	///
	/// \sa setEditObject()
	virtual void resetUI() { updateUI(); }
	
	/// \brief This method updates the displayed value of the parameter UI.
	/// 
	/// This method should be overriden by derived classes.
	virtual void updateUI() {}

	/// \brief Sets the enabled state of the UI.
	/// \param enabled Controls whether may change the parameter's value or not.
	/// \sa isEnabled()
	virtual void setEnabled(bool enabled) { _enabled = enabled; }

private Q_SLOTS:

	/// \brief Sets the object whose property is being displayed in this parameter UI.
	/// \sa editObject()
	void setEditObject(RefTarget* newObject) {
		_editObject = newObject;
		resetUI();
	}
	
private:

	/// The object whose parameter is being edited.
	ReferenceField<RefTarget> _editObject;

	/// Stores whether this UI is enabled.
	bool _enabled;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(ParameterUI)
	DECLARE_REFERENCE_FIELD(_editObject)
};


/**
 * \brief Base class for UI components that allow the user to edit a RefTarget based
 *        parameter of a RefTarget in the properties editor.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT ReferenceParameterUI : public ParameterUI
{
public:
	
	/// brief Constructor.
	/// \param parentEditor The editor in which this parameter UI is used. This becomes the parent of this object.
	/// \param refField Specifies the reference field that contains the sub-object to be edited by this parameter UI.
	ReferenceParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField);

	/// \brief Destructor.
	/// \note The destructor must clear all references since this UI object
	///       might have been deleted via the delete operator and not via
	///       the PluginClass::autoDeleteObject() method which should normaly be used for PluginClass derived classes.
	virtual ~ReferenceParameterUI() { clearAllReferences(); }	

	/// \brief The reference field that specifies the parameter or sub-object of the edited object that
	///        is bound to this parameter UI.
	const PropertyFieldDescriptor& referenceField() const { return _refField; }

	/// \brief Returns the sub-object that is bound to this parameter UI.
	/// \return The object stored in the reference field. This may be \c NULL either when there is no editable object selected in the parent editor
	///         or if the editable object's reference field is currently empty.
	RefTarget* parameterObject() const { return _parameterObject; }	
	
	/// \brief This method is called when parameter object has been assigned to the reference field of the editable object 
	/// this parameter UI is bound to.
	///
	/// It is also called when the editable object itself has
	/// been replaced in the editor. The parameter UI should react to this change appropriately and
	/// show the properties value for the new edit object in the UI. New implementations of this
	/// method must call the base implementation before any other action is taken.
	virtual void resetUI();
	
public:
	
	Q_PROPERTY(RefTarget parameterObject READ parameterObject)			
	
protected:

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

private:

	/// The controller or value object.
	ReferenceField<RefTarget> _parameterObject;

	/// The reference field that specifies the parameter of the edited object that
	/// is bound to this parameter UI.
	const PropertyFieldDescriptor& _refField;

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(ReferenceParameterUI)
	DECLARE_REFERENCE_FIELD(_parameterObject)
};

/**
 * \brief Base class for UI components that allow the user to edit a property of
 *        an QObject in an editor.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT PropertyParameterUI : public ParameterUI
{
public:
	
	/// \brief Constructor for a Qt property.
	/// \param parentEditor The editor in which this parameter UI is used. This becomes the parent of this object.
	/// \param propertyName The name of the property that has been defined using the \c Q_PROPERTY macro. 
	PropertyParameterUI(PropertiesEditor* parentEditor, const char* propertyName);

	/// \brief Constructor for a PropertyField property.
	/// \param parentEditor The editor in which this parameter UI is used. This becomes the parent of this object.
	/// \param propField The non-animatable property field.
	PropertyParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField);
	
	/// \brief Returns the property being edited in this parameter UI.
	/// \return The name of the QObject property this UI is bound to or \c
	///         NULL if this PropertyUI is bound to a PropertyField.
	/// \sa propertyField()
	const char* propertyName() const { return _propertyName; }

	/// \brief Returns the property field being edited.
	/// \return A pointer to the descriptor of the PropertyField being edited or
	///         \c NULL if this PropertyUI is bound to a normal Qt property.
	/// \sa propertyName()
	const PropertyFieldDescriptor* propertyField() const { return _propField; }
	
public:
	
	Q_PROPERTY(const char* propertyName READ propertyName)
	
protected:

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

private:
	
	/// The name of the property being edited or NULL.
	const char* _propertyName;

	/// The property field being edited or NULL.
	const PropertyFieldDescriptor* _propField;
	
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(PropertyParameterUI)
};

};

#endif // __OVITO_PARAMETER_UI_H
