///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/utilities/ProgressIndicator.h>
#include "LAMMPSBinaryDumpWriter.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(LAMMPSBinaryDumpWriter, LAMMPSDumpWriter)

/// This method looks for a special argument on the command line
/// that enables the use of an extended binary LAMMPS dump format
/// that can handle non-orthogonal simulation boxes.
static bool isModifiedBinaryDumpFormatEnabled()
{
	Q_FOREACH(QString s, QCoreApplication::arguments()) {
		if(s == "--extended-dump-format")
			return true;
	}
	return false;
}

/******************************************************************************
* Writes the output file.
******************************************************************************/
bool LAMMPSBinaryDumpWriter::writeAtomsFile(const QString& filepath, DataSet* dataset, const QVector<TimeTicks>& exportFrames, bool suppressDialogs)
{
	MsgLogger() << "Opening binary LAMMPS dump file" << filepath << "for writing." << endl;

	QFile stream(filepath);
	if(!stream.open(QIODevice::WriteOnly))
		throw Exception(tr("Failed to open the file %1 for writing: %2").arg(filepath, stream.errorString()));

	ProgressIndicator progress(QString(), exportFrames.size() * 100, suppressDialogs);

	Q_FOREACH(TimeTicks time, exportFrames) {
		int frame = time / dataset->animationSettings()->ticksPerFrame();
		progress.setLabelText(tr("Writing binary LAMMPS dump file (frame %1)").arg(frame));
		if(progress.isCanceled()) return false;

		// Extract the atoms to be exported from the scene.
		PipelineFlowState flowState = retrieveAtoms(dataset, time);
		AtomsObject* atoms = dynamic_object_cast<AtomsObject>(flowState.result());
		if(atoms == NULL)
			throw Exception(tr("The scene does not contain any atoms that could be exported (at animation frame %1).").arg(frame));

		DataRecordWriterHelper helper(&channelMapping(), atoms);

		int natoms = atoms->atomsCount();
		int ntimestep, nColumns, nchunk;
		double xlo, xhi, ylo, yhi, zlo, zhi, xy, xz, yz;

		ntimestep = frame;
		nColumns = helper.actualColumnCount();
		nchunk = 1;

		TimeInterval interval;
		AffineTransformation simCell = atoms->simulationCell()->cellMatrix();

		xlo = simCell.getTranslation().X;
		ylo = simCell.getTranslation().Y;
		zlo = simCell.getTranslation().Z;
		xhi = simCell.column(0).X + xlo;
		yhi = simCell.column(1).Y + ylo;
		zhi = simCell.column(2).Z + zlo;
		xy = simCell.column(1).X;
		xz = simCell.column(2).X;
		yz = simCell.column(2).Y;

		stream.write((char*)&ntimestep,sizeof(int));
		stream.write((char*)&natoms,sizeof(int));
		stream.write((char*)&xlo,sizeof(double));
		stream.write((char*)&xhi,sizeof(double));
		stream.write((char*)&ylo,sizeof(double));
		stream.write((char*)&yhi,sizeof(double));
		stream.write((char*)&zlo,sizeof(double));
		stream.write((char*)&zhi,sizeof(double));
		if(isModifiedBinaryDumpFormatEnabled()) {
			stream.write((char*)&xy,sizeof(double));
			stream.write((char*)&xz,sizeof(double));
			stream.write((char*)&yz,sizeof(double));
		}
		stream.write((char*)&nColumns,sizeof(int));
		stream.write((char*)&nchunk,sizeof(int));

		int n = natoms * nColumns;
		stream.write((char*)&n,sizeof(int));

		QVector<double> dataValues(n);
		double* iter = dataValues.data();

		int progressStartValue = progress.value();
		for(int i=0; i<natoms; i++) {

			// Update progress indicator.
			if((i % 1000) == 0) {
				progress.setValue(i * 100 / natoms + progressStartValue);
				progress.isCanceled();
			}

			// Write mapped columns.
			helper.writeAtom(i, iter);
			iter += helper.actualColumnCount();
		}

		if(stream.write((const char*)dataValues.constData(), n * sizeof(double)) != n * sizeof(double))
			throw Exception(tr("Could not write all bytes to the output file."));
	}
	return true;
}

};	// End of namespace AtomViz
