///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file PositionDataChannel.h
 * \brief Contains the definition of the AtomViz::PositionDataChannel class.
 */

#ifndef __POSITION_DATA_CHANNEL_H
#define __POSITION_DATA_CHANNEL_H

#include <atomviz/AtomViz.h>

#include "DataChannel.h"
#include <atomviz/atoms/AtomsRenderer.h>

#include <core/scene/animation/controller/Controller.h>


namespace AtomViz {

/**
 * \brief The data channel stores the atomic positions as well as several important parameters that affect
 *        the display of atoms.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT PositionDataChannel : public DataChannel
{
	Q_OBJECT

public:

	/// \brief Serialization constructor that creates an empty data channel.
	///
	/// \note This constructor is only used when a data channel is loaded from a scene file.
	///       It must not be used to create a new data channel.
	PositionDataChannel(bool isLoading = false);

	/// \brief Constructor that creates a standard data channel.
	/// \param which Specifies which standard data channel should be created.
	///              This should be DataChannelIdentifier::PositionChannel.
	///
	/// Data type, component count and name are automatically chosen by this constructor.
	PositionDataChannel(DataChannelIdentifier which);

	/// \brief Returns the global scaling factor applied to all atom radii.
	/// \return This scaling for is applied on all atom radii before rendering. The default is 1.
	FloatType globalAtomRadiusScale() const { return _globalAtomRadiusScale ? _globalAtomRadiusScale->getCurrentValue() : (FloatType)1.0; }

	/// \brief Sets the global scaling factor applied to all atom radii.
	/// \param scale The new scaling factor.
	/// \undoable
	void setGlobalAtomRadiusScale(FloatType scale) { if(_globalAtomRadiusScale) _globalAtomRadiusScale->setCurrentValue(scale); }

	/// \brief Returns the controller for the global scaling factor applied to all atom radii.
	/// \return The controller for the radius scaling factor.
	FloatController* globalAtomRadiusScaleController() const { return _globalAtomRadiusScale; }

	/// \brief Sets the controller for the global scaling factor applied to all atom radii.
	/// \param ctrl The new controller.
	/// \undoable
	void setGlobalAtomRadiusScaleController(const FloatController::SmartPtr& ctrl) { _globalAtomRadiusScale = ctrl; }

	/// \brief Returns whether the atoms are rendered in high-quality in the viewports.
	/// \return \c true if the high-quality mode is active.
	/// \sa setHighQualityRenderingInViewports()
	bool highQualityRenderingInViewports() const { return _useHighQualityRenderingInViewports; }

	/// \brief Sets whether the atoms are rendered in high-quality in the viewports.
	/// \param enable Specifies whether HQ mode will be enabled.
	/// \undoable
	/// \sa highQualityRenderingInViewports()
	void setHighQualityRenderingInViewports(bool enable) { _useHighQualityRenderingInViewports = enable; }

	/// \brief Returns whether the atoms are rendered without shading.
	/// \return \c true if the flat rendering mode is active.
	/// \sa setFlatAtomRendering()
	bool flatAtomRendering() const { return _flatAtomRendering; }

	/// \brief Sets whether the atoms are rendered without shading.
	/// \param enable Specifies whether flat rendering mode will be enabled.
	/// \undoable
	/// \sa flatAtomRendering()
	void setFlatAtomRendering(bool enable) { _flatAtomRendering = enable; }

	/// \brief Renders the channel' contents in a viewport.
	/// \param time The current animation time.
	/// \param vp The viewport into which the channel should be rendered.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param contextNode The scene object the AtomsObject belongs to.
	virtual void render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode);

	/// \brief Renders the channel's contents in high-quality mode to an offscreen buffer.
	/// \param time The current animation time.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param view Describes the projection parameters.
	/// \param contextNode The scene object the AtomsObject belongs to.
	/// \param imageWidth The width of the image buffer in pixels.
	/// \param imageHeight The height of the image buffer in pixels.
	/// \param glcontext The OpenGL rendering context.
	virtual void renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext);

	/// \brief Returns the bounding box of the channel's geometry when rendered in the viewports.
	/// \param time The current animation time.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param contextNode The scene object the AtomsObject belongs to.
	/// \param validityInterval This is used to return the validity interval during which the bounding box doesn't change.
	/// \return The bounding box of the rendered geometry or an empty box if the channel is not rendered in the viewports.
	virtual Box3 boundingBox(TimeTicks time, AtomsObject* atoms, ObjectNode* contextNode, TimeInterval& validityInterval);

	/// \brief Lets the channel clear all its internal caches.
	///
	/// This method is automatically invoked for each DataChannel by the AtomsObject::invalidate()
	/// method. It informs the data channel that the AtomsObject it belongs to has
	/// changed in some way.
	virtual void clearCaches();

public:

	Q_PROPERTY(FloatType globalAtomRadiusScale READ globalAtomRadiusScale WRITE setGlobalAtomRadiusScale)
	Q_PROPERTY(bool highQualityRenderingInViewports READ highQualityRenderingInViewports WRITE setHighQualityRenderingInViewports)
	Q_PROPERTY(bool flatAtomRendering READ flatAtomRendering WRITE setFlatAtomRendering)

private:

	/// Fills a render buffer with atom data.
	bool fillRenderBuffer(TimeTicks time, AtomsObject* atoms, AtomsRenderer& renderBuffer, TimeInterval& renderBufferValidity);

	/// Controls a global scaling factor applied to all atom radii.
	ReferenceField<FloatController> _globalAtomRadiusScale;

	/// Controls whether high-quality rendering of atoms is used in the viewports.
	PropertyField<bool> _useHighQualityRenderingInViewports;

	/// Controls whether atoms are rendered flat without shading.
	PropertyField<bool> _flatAtomRendering;

	/// The internal buffer that is used for rendering of atoms.
	AtomsRenderer renderBuffer;

	/// Indicates if the atoms stored in the render buffer have to be updated.
	TimeInterval renderBufferValidity;

	DECLARE_SERIALIZABLE_PLUGIN_CLASS(PositionDataChannel)
	DECLARE_REFERENCE_FIELD(_globalAtomRadiusScale)
	DECLARE_PROPERTY_FIELD(_useHighQualityRenderingInViewports)
	DECLARE_PROPERTY_FIELD(_flatAtomRendering)
};

/**
 * \brief A properties editor for the PositionDataChannel class.
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT PositionDataChannelEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(PositionDataChannelEditor)
};

};	// End of namespace AtomViz

#endif // __POSITION_DATA_CHANNEL_H
