
#include "config.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <orc/orcprogram.h>
#include <orc/orcdebug.h>

/**
 * SECTION:orcexecutor
 * @title: OrcExecutor
 * @short_description: Running Orc programs
 */


OrcExecutor *
orc_executor_new (OrcProgram *program)
{
  OrcExecutor *ex;

  ex = malloc(sizeof(OrcExecutor));
  memset(ex,0,sizeof(OrcExecutor));

  ex->program = program;

  return ex;
}

void
orc_executor_free (OrcExecutor *ex)
{
  free (ex);
}

void
orc_executor_run (OrcExecutor *ex)
{
  void (*func) (OrcExecutor *);

  func = ex->program->code_exec;
  if (func) {
    func (ex);
    //ORC_ERROR("counters %d %d %d", ex->counter1, ex->counter2, ex->counter3);
  } else {
    orc_executor_emulate (ex);
  }
}

void
orc_executor_set_program (OrcExecutor *ex, OrcProgram *program)
{
  ex->program = program;
  if (program->code_exec) {
    ex->arrays[ORC_VAR_A1] = (void *)program->code_exec;
  } else {
    ex->arrays[ORC_VAR_A1] = (void *)orc_executor_emulate;
  }
}

void
orc_executor_set_array (OrcExecutor *ex, int var, void *ptr)
{
  ex->arrays[var] = ptr;
}

void
orc_executor_set_stride (OrcExecutor *ex, int var, int stride)
{
  ex->params[var] = stride;
}

void
orc_executor_set_array_str (OrcExecutor *ex, const char *name, void *ptr)
{
  int var;
  var = orc_program_find_var_by_name (ex->program, name);
  ex->arrays[var] = ptr;
}

void
orc_executor_set_param (OrcExecutor *ex, int var, int value)
{
  ex->params[var] = value;
}

void
orc_executor_set_param_str (OrcExecutor *ex, const char *name, int value)
{
  int var;
  var = orc_program_find_var_by_name (ex->program, name);
  ex->params[var] = value;
}

int
orc_executor_get_accumulator (OrcExecutor *ex, int var)
{
  return ex->accumulators[var - ORC_VAR_A1];
}

int
orc_executor_get_accumulator_str (OrcExecutor *ex, const char *name)
{
  int var;
  var = orc_program_find_var_by_name (ex->program, name);
  return ex->accumulators[var];
}

void
orc_executor_set_n (OrcExecutor *ex, int n)
{
  ex->n = n;
}

void
orc_executor_set_m (OrcExecutor *ex, int m)
{
  ORC_EXECUTOR_M(ex) = m;
}

void
orc_executor_emulate (OrcExecutor *ex)
{
  int i;
  int j;
  int k;
  int m, m_index;
  OrcProgram *program = ex->program;
  OrcInstruction *insn;
  OrcStaticOpcode *opcode;
  OrcOpcodeExecutor opcode_ex;

  ex->accumulators[0] = 0;
  ex->accumulators[1] = 0;
  ex->accumulators[2] = 0;
  ex->accumulators[3] = 0;

  ORC_DEBUG("emulating");

  memset (&opcode_ex, 0, sizeof(opcode_ex));

  if (program->is_2d) {
    m = ORC_EXECUTOR_M(ex);
  } else {
    m = 1;
  }
  ORC_DEBUG("src ptr %p stride %d", ex->arrays[ORC_VAR_S1], ex->params[ORC_VAR_S1]);
  for(m_index=0;m_index<m;m_index++){
    ORC_DEBUG("m_index %d m %d", m_index, m);
    for(i=0;i<ex->n;i++){
      for(j=0;j<program->n_insns;j++){
        insn = program->insns + j;
        opcode = insn->opcode;

        /* set up args */
        for(k=0;k<ORC_STATIC_OPCODE_N_SRC;k++) {
          OrcVariable *var = program->vars + insn->src_args[k];

          if (opcode->src_size[k] == 0) continue;

          if (var->vartype == ORC_VAR_TYPE_CONST) {
            opcode_ex.src_values[k] = var->value;
          } else if (var->vartype == ORC_VAR_TYPE_PARAM) {
            opcode_ex.src_values[k] = ex->params[insn->src_args[k]];
          } else if (var->vartype == ORC_VAR_TYPE_TEMP) {
            /* FIXME shouldn't store executor stuff in program */
            opcode_ex.src_values[k] = var->value;
          } else if (var->vartype == ORC_VAR_TYPE_SRC ||
              var->vartype == ORC_VAR_TYPE_DEST) {
            void *ptr = ORC_PTR_OFFSET(ex->arrays[insn->src_args[k]],
                var->size*i + ex->params[insn->src_args[k]]*m_index);

            switch (var->size) {
              case 1:
                opcode_ex.src_values[k] = *(orc_int8 *)ptr;
                break;
              case 2:
                opcode_ex.src_values[k] = *(orc_int16 *)ptr;
                break;
              case 4:
                opcode_ex.src_values[k] = *(orc_int32 *)ptr;
                break;
              case 8:
                opcode_ex.src_values[k] = *(orc_int64 *)ptr;
                break;
              default:
                ORC_ERROR("unhandled size %d", program->vars[insn->src_args[k]].size);
            }
          } else {
            ORC_ERROR("shouldn't be reached (%d)", var->vartype);
          }
        }

        opcode->emulate (&opcode_ex, opcode->emulate_user);

        for(k=0;k<ORC_STATIC_OPCODE_N_DEST;k++){
          OrcVariable *var = program->vars + insn->dest_args[k];

          if (opcode->dest_size[k] == 0) continue;

          if (var->vartype == ORC_VAR_TYPE_TEMP) {
            /* FIXME shouldn't store executor stuff in program */
            var->value = opcode_ex.dest_values[k];
          } else if (var->vartype == ORC_VAR_TYPE_DEST) {
            void *ptr = ORC_PTR_OFFSET(ex->arrays[insn->dest_args[k]],
                var->size*i + ex->params[insn->dest_args[k]]*m_index);

            switch (var->size) {
              case 1:
                *(orc_int8 *)ptr = opcode_ex.dest_values[k];
                break;
              case 2:
                *(orc_int16 *)ptr = opcode_ex.dest_values[k];
                break;
              case 4:
                *(orc_int32 *)ptr = opcode_ex.dest_values[k];
                break;
              case 8:
                *(orc_int64 *)ptr = opcode_ex.dest_values[k];
                break;
              default:
                ORC_ERROR("unhandled size %d", program->vars[insn->dest_args[k]].size);
            }
          } else if (var->vartype == ORC_VAR_TYPE_ACCUMULATOR) {
            switch (var->size) {
              case 2:
                ex->accumulators[insn->dest_args[k] - ORC_VAR_A1] +=
                  opcode_ex.dest_values[k];
                ex->accumulators[insn->dest_args[k] - ORC_VAR_A1] &= 0xffff;
                break;
              case 4:
                ex->accumulators[insn->dest_args[k] - ORC_VAR_A1] +=
                  opcode_ex.dest_values[k];
                break;
              default:
                ORC_ERROR("unhandled size %d", program->vars[insn->dest_args[k]].size);
            }
          } else {
            ORC_ERROR("shouldn't be reached (%d)", var->vartype);
          }
        }
      }
    }
  }
}


