/* OpenVAS-Client
 * $Id$
 * Description: Create user install packages from rpm packages by calling alien.
 *
 * Authors:
 * Felix Wolfsteller <felix.wolfsteller@intevation.de>
 *
 * Copyright:
 * Copyright (C) 2009 Greenbone Networks GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * or, at your option, any later version as published by the Free
 * Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * @file
 * 
 * The OpenVAS Credentials Manager allows the user to create "accounts" that
 * are used to perform local checks.
 * 
 * For *nix- systems RPM packages can be created that install the public key of
 * the 'account' (@see openvas_lsc_user_rpm).
 * 
 * Using the tool "alien" these packages can be transformed into debian (.deb)
 * packages.
 */

#include <glib/gstdio.h>

#include "openvas_ssh_login.h"
#include "file_utils.h"
#include "openvas_i18n.h"
#include "error_dlg.h"
#include "openvas_lsc_target_prep.h"
#include "openvas_lsc_user_deb.h"
#include "openvas_lsc_user_rpm.h" /* for openvas_lsc_user_rpm_filename */

/**
 * @brief Execute alien to create a deb packages from an rpm package.
 * @param rpmfile .rpm file to transform with alien to a .deb.
 */
static gboolean
execute_alien (const gchar* rpmfile)
{
  gchar** cmd;
  gint exit_status;

  cmd = (gchar **) g_malloc (6 * sizeof (gchar *));

  cmd[0] = g_strdup ("fakeroot");
  cmd[1] = g_strdup ("--");
  cmd[2] = g_strdup ("alien");
  cmd[3] = g_strdup ("--scripts");
  cmd[4] = g_strdup (rpmfile);
  cmd[5] = NULL;
  printf ("--- executing alien.\n");
  if (g_spawn_sync (".",
                    cmd,
                    NULL, // env
                    G_SPAWN_SEARCH_PATH,
                    NULL, // setup func
                    NULL,
                    NULL,
                    NULL,
                    &exit_status,
                    NULL                 ) == FALSE
      || exit_status != 0)
    {
      show_error(_("Error (%d) creating the deb package.\n"
                   "For further information consult your shell."), exit_status);
      exit_status = -1;
    }

  g_free (cmd[0]);
  g_free (cmd[1]);
  g_free (cmd[2]);
  g_free (cmd[3]);
  g_free (cmd[4]);
  g_free (cmd[5]);
  g_free (cmd);

  printf ("--- alien returned %d.\n", exit_status);
  return (exit_status == 0);
}

/**
 * @brief Returns whether alien could be found in the path.
 * 
 * The check itself will only be done once.
 * 
 * @return true if alien could be found in the path, false otherwise.
 */
gboolean
openvas_lsc_user_deb_alien_found ()
{
  static gboolean searched = FALSE;
  static gboolean found    = FALSE;

  if (searched == FALSE)
    {
      // Check if alien is found in path
      gchar* alien_path = g_find_program_in_path ("alien");
      if (alien_path != NULL)
        {
          found = TRUE;
          g_free (alien_path);
        }
      searched = TRUE;
    }

  return found;
}

/**
 * @brief Returns the path to debian package 'installer' for a given
 * @brief openvas_ssh_login.
 *
 * Installers will be placed to or read from this location. It equals the path
 * to the public key but with a different file extension (.deb).
 *
 * @return Path to installer for given openvas_ssh_login, has to be freed using
 *         g_free, NULL in case of errors.
 */
gchar*
openvas_lsc_user_deb_filename (const char* accountname)
{
  return openvas_lsc_target_prep_filename (accountname, "deb");
}


/**
 * @brief Attempts creation of a deb packages from an rpm package.
 * 
 * @param loginfo openvas_ssh_login struct to create rpm for.
 * 
 * @return TRUE if successful, FALSE otherwise.
 * @TODO Test (and likely correct) behaviour.
 */
gboolean
openvas_lsc_user_deb (openvas_ssh_login* loginfo, const gchar* to_filename)
{
  gboolean success   = FALSE;
  gchar* debfilename = g_strdup_printf ("openvas-lsc-target-%s_0.5-2_all.deb", loginfo->username);
  gchar* rpmfile = openvas_lsc_user_rpm_filename (loginfo->name);

  openvas_lsc_user_deb_alien_found ();

  execute_alien (rpmfile);
  file_utils_move_file (debfilename, to_filename);

  g_free (debfilename);
  g_free (rpmfile);

  return success;
}
