//                                               -*- C++ -*-
/**
 *  @file  Logistic.cxx
 *  @brief The Logistic distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: Logistic.cxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#include <cmath>
#include "Logistic.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Func::SpecFunc        SpecFunc;
      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Logistic);

      static Base::Common::Factory<Logistic> RegisteredFactory("Logistic");

      /* Default constructor */
      Logistic::Logistic()
        : NonEllipticalDistribution("Logistic"),
          alpha_(0.0), beta_(1.0)
      {
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      Logistic::Logistic(const NumericalScalar alpha,
                         const NumericalScalar beta)
        /* throw (InvalidArgumentException) */
        : NonEllipticalDistribution("Logistic"),
          alpha_(alpha), beta_(0.0)
      {
        // This call set also the range.
        setBeta(beta);
        setDimension( 1 );
      }

      /* Comparison operator */
      Bool Logistic::operator ==(const Logistic & other) const
      {
        if (this == &other) return true;
        return (alpha_ == other.alpha_) && (beta_ == other.beta_);
      }

      /* String converter */
      String Logistic::__repr__() const
      {
        OSS oss;
        oss << "class=" << Logistic::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " alpha=" << alpha_
            << " beta=" << beta_;
        return oss;
      }

      String Logistic::__str__(const String & offset) const
      {
        OSS oss;
        oss << offset << getClassName() << "(alpha = " << alpha_ << ", beta = " << beta_ << ")";
        return oss;
      }

      /* Virtual constructor */
      Logistic * Logistic::clone() const
      {
        return new Logistic(*this);
      }

      /* Get one realization of the distribution */
      Logistic::NumericalPoint Logistic::getRealization() const
      {
        NumericalScalar prob(RandomGenerator::Generate());
        return NumericalPoint(1, alpha_ + beta_ * log(prob / (1.0 - prob)));
      }


      /* Get the DDF of the distribution */
      Logistic::NumericalPoint Logistic::computeDDF(const NumericalPoint & point) const
      {
        NumericalScalar expX(exp((point[0] - alpha_) / beta_));
        NumericalScalar betaExpX(beta_ * (1.0 + expX));
        return NumericalPoint(1, beta_ * expX * (1.0 - expX) / (betaExpX * betaExpX * betaExpX));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Logistic::computePDF(const NumericalPoint & point) const
      {
        const NumericalScalar x((point[0] - alpha_) / beta_);
        if ((x <= 0.5 * SpecFunc::LogMinNumericalScalar) || (x >= 0.5 * SpecFunc::LogMaxNumericalScalar)) return 0.0;
        NumericalScalar expX(exp(x));
        NumericalScalar betaExpX(beta_ * (1.0 + expX));
        return beta_ * expX / (betaExpX * betaExpX);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Logistic::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        const NumericalScalar expX(exp((point[0] - alpha_) / beta_));
        if (tail) return 1.0 / (1.0 + expX);
        return expX / (1.0 + expX);
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex Logistic::computeCharacteristicFunction(const NumericalScalar x,
                                                               const Bool logScale) const
      {
        const NumericalScalar piBetaU(M_PI * beta_ * x);
        if (logScale) return NumericalComplex(0.0, x * alpha_) + log(piBetaU) - log(sinh(piBetaU));
        return exp(NumericalComplex(0.0, x * alpha_)) * piBetaU / sinh(piBetaU);
      }

      /* Get the PDFGradient of the distribution */
      Logistic::NumericalPoint Logistic::computePDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar x((point[0] - alpha_) / beta_);
        NumericalScalar expX(exp(x));
        NumericalScalar betaExpX(beta_ * (1.0 + expX));
        NumericalPoint pdfGradient(2);
        pdfGradient[0] = beta_ * expX * (expX - 1.0) / (betaExpX * betaExpX * betaExpX);
        pdfGradient[1] = pdfGradient[0] * x - expX / (betaExpX * betaExpX);
        return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Logistic::NumericalPoint Logistic::computeCDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar x((point[0] - alpha_) / beta_);
        NumericalScalar expX(exp(x));
        NumericalScalar betaExpX(beta_ * (1.0 + expX));
        NumericalPoint cdfGradient(2);
        cdfGradient[0] = -beta_ * expX / (betaExpX * betaExpX);
        cdfGradient[1] = cdfGradient[0] * x;
        return cdfGradient;
      }

      /* Get the quantile of the distribution */
      NumericalScalar Logistic::computeScalarQuantile(const NumericalScalar prob,
                                                      const Bool tail,
                                                      const NumericalScalar precision) const
      {
        if (tail) return alpha_ + beta_ * log((1.0 - prob) / prob);
        return alpha_ + beta_ * log(prob / (1.0 - prob));
      }

      /* Get the roughness, i.e. the L2-norm of the PDF */
      NumericalScalar Logistic::getRoughness() const
      {
        // 0.1666666666666666666666667 = 1 / 6
        return 0.1666666666666666666666667 / beta_;
      }

      /* Compute the mean of the distribution */
      void Logistic::computeMean() const
      {
        mean_ = NumericalPoint(1, alpha_);
        isAlreadyComputedMean_ = true;
      }

      /* Get the standard deviation of the distribution */
      Logistic::NumericalPoint Logistic::getStandardDeviation() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, SpecFunc::PI_SQRT3 * beta_);
      }

      /* Get the skewness of the distribution */
      Logistic::NumericalPoint Logistic::getSkewness() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, 0.0);
      }

      /* Get the kurtosis of the distribution */
      Logistic::NumericalPoint Logistic::getKurtosis() const /* throw(NotDefinedException) */
      {
        // 3+6/5 = 4.2
        return NumericalPoint(1, 4.2);
      }

      /* Compute the covariance of the distribution */
      void Logistic::computeCovariance() const
      {
        covariance_ = CovarianceMatrix(1);
        covariance_(0, 0) = SpecFunc::PI2_3 * beta_ * beta_;
        isAlreadyComputedCovariance_ = true;
      }

      /* Get the moments of the standardized distribution */
      Logistic::NumericalPoint Logistic::getStandardMoment(const UnsignedLong n) const
      {
        if (n % 2 == 1) return NumericalPoint(1, 0.0);
        if (n ==  2) return NumericalPoint(1, 3.2898681336964528730e00);
        if (n ==  4) return NumericalPoint(1, 4.5457575815867804043e01);
        if (n ==  6) return NumericalPoint(1, 1.4191935714683065499e03);
        if (n ==  8) return NumericalPoint(1, 8.0336229269397526593e04);
        if (n == 10) return NumericalPoint(1, 7.2506291303452157147e06);
        if (n == 12) return NumericalPoint(1, 9.5777106159240860718e08);
        if (n == 14) return NumericalPoint(1, 1.7434597634941644924e11);
        if (n == 16) return NumericalPoint(1, 4.1844942225739403485e13);
        if (n == 18) return NumericalPoint(1, 1.2804698598089257967e16);
        if (n == 20) return NumericalPoint(1, 4.8657993773520842256e18);
        if (n == 22) return NumericalPoint(1, 2.2480009196614112639e21);
        if (n == 24) return NumericalPoint(1, 1.2408967295076549681e24);
        if (n == 26) return NumericalPoint(1, 8.0658291023450627114e26);
        if (n == 28) return NumericalPoint(1, 6.0977668695185918278e29);
        if (n == 30) return NumericalPoint(1, 5.3050571913031274079e32);
        if (n == 32) return NumericalPoint(1, 5.2626167374485750005e35);
        if (n == 34) return NumericalPoint(1, 5.9046559804483869578e38);
        if (n == 36) return NumericalPoint(1, 7.4398665356897600917e41);
        if (n == 38) return NumericalPoint(1, 1.0460452349293967338e45);
        if (n == 40) return NumericalPoint(1, 1.6318305664943113275e48);
        if (n == 42) return NumericalPoint(1, 2.8100122355051208743e51);
        if (n == 44) return NumericalPoint(1, 5.3165431495765953256e54);
        if (n == 46) return NumericalPoint(1, 1.1005244319624021505e58);
        if (n == 48) return NumericalPoint(1, 2.4827831185072057135e61);
        if (n == 50) return NumericalPoint(1, 6.0828186403426702060e64);
        return Logistic(0.0, 1.0).getMoment(n);
      }

      /* Parameters value and description accessor */
      Logistic::NumericalPointWithDescriptionCollection Logistic::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(2);
        Description description(point.getDimension());
        point[0] = alpha_;
        point[1] = beta_;
        description[0] = "alpha";
        description[1] = "beta";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void Logistic::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = Logistic(parametersCollection[0][0], parametersCollection[0][1]);
      }

      /* Alpha accessor */
      void Logistic::setAlpha(const NumericalScalar alpha)
      {
        if (alpha != alpha_)
          {
            alpha_ = alpha;
            isAlreadyComputedMean_ = false;
            // The covariancedoes not depend on alpha
            computeRange();
          }
      }

      NumericalScalar Logistic::getAlpha() const
      {
        return alpha_;
      }


      /* Beta accessor */
      void Logistic::setBeta(const NumericalScalar beta)
      /* throw(InvalidArgumentException) */
      {
        if (beta <= 0.) throw InvalidArgumentException(HERE) << "Beta MUST be positive, here beta=" << beta;
        if (beta != beta_)
          {
            beta_ = beta;
            isAlreadyComputedCovariance_ = false;
            // The mean does not depend on beta
            computeRange();
          }
      }

      NumericalScalar Logistic::getBeta() const
      {
        return beta_;
      }

      /* Method save() stores the object through the StorageManager */
      void Logistic::save(StorageManager::Advocate & adv) const
      {
        NonEllipticalDistribution::save(adv);
        adv.saveAttribute( "alpha_", alpha_ );
        adv.saveAttribute( "beta_", beta_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void Logistic::load(StorageManager::Advocate & adv)
      {
        NonEllipticalDistribution::load(adv);
        adv.loadAttribute( "alpha_", alpha_ );
        adv.loadAttribute( "beta_", beta_ );
        computeRange();
      }



    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
