//                                               -*- C++ -*-
/**
 *  @file  Exponential.cxx
 *  @brief The Exponential distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: Exponential.cxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#include <cmath>
#include "Exponential.hxx"
#include "SpecFunc.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Func::SpecFunc        SpecFunc;
      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Exponential);

      static Base::Common::Factory<Exponential> RegisteredFactory("Exponential");

      /* Default constructor */
      Exponential::Exponential()
        : NonEllipticalDistribution("Exponential"),
          lambda_(1.0),
          gamma_(0.0)
      {
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      Exponential::Exponential(const NumericalScalar lambda,
                               const NumericalScalar gamma)
        /* throw(InvalidArgumentException) */
        : NonEllipticalDistribution("Exponential"),
          lambda_(0.0),
          gamma_(gamma)
      {
        // We set the dimension of the Exponential distribution
        setDimension( 1 );
        // This call check lambda and set also the range.
        setLambda(lambda);
      }

      /* Comparison operator */
      Bool Exponential::operator ==(const Exponential & other) const
      {
        if (this == &other) return true;
        return (lambda_ == other.lambda_) && (gamma_ == other.gamma_);
      }

      /* String converter */
      String Exponential::__repr__() const
      {
        OSS oss;
        oss << "class=" << Exponential::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " lambda=" << lambda_
            << " gamma=" << gamma_;
        return oss;
      }

      String Exponential::__str__(const String & offset) const
      {
        OSS oss;
        oss << offset << getClassName() << "(lambda = " << lambda_ << ", gamma = " << gamma_ << ")";
        return oss;
      }

      /* Virtual constructor */
      Exponential * Exponential::clone() const
      {
        return new Exponential(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Exponential::computeRange()
      {
        const NumericalPoint lowerBound(1, gamma_);
        const NumericalPoint upperBound(computeUpperBound());
        const Interval::BoolCollection finiteLowerBound(1, true);
        const Interval::BoolCollection finiteUpperBound(1, false);
        setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /* Get one realization of the distribution */
      Exponential::NumericalPoint Exponential::getRealization() const
      {
        return NumericalPoint(1, gamma_ - log(RandomGenerator::Generate()) / lambda_);
      }


      /* Get the DDF of the distribution */
      Exponential::NumericalPoint Exponential::computeDDF(const NumericalPoint & point) const
      {
        if (point[0] < gamma_) return NumericalPoint(1, 0.0);
        return NumericalPoint(1, -lambda_ * computePDF(point));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Exponential::computePDF(const NumericalPoint & point) const
      {
        NumericalScalar x(point[0] - gamma_);
        if (x < 0.0) return 0.0;
        return lambda_ * exp(-lambda_ * x);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Exponential::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        NumericalScalar x(point[0] - gamma_);
        if (x < 0.0) return (tail ? 1.0 : 0.0);
        if (tail) return exp(-lambda_ * x);
        return 1.0 - exp(-lambda_ * x);
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex Exponential::computeCharacteristicFunction(const NumericalScalar x,
                                                                  const Bool logScale) const
      {
        if (logScale) return NumericalComplex(0.0, x * gamma_) - log(NumericalComplex(1.0, - x / lambda_));
        return exp(NumericalComplex(0.0, x * gamma_)) / NumericalComplex(1.0, -x / lambda_);
      }

      /* Get the PDFGradient of the distribution */
      Exponential::NumericalPoint Exponential::computePDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar x(point[0] - gamma_);
        NumericalPoint pdfGradient(2, 0.0);
        if (x < 0.0) return pdfGradient;
        NumericalScalar expX(exp(-lambda_ * x));
        pdfGradient[0] = (1.0 - lambda_ * x) * expX;
        pdfGradient[1] = lambda_ * lambda_ * expX;
        return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Exponential::NumericalPoint Exponential::computeCDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar x(point[0] - gamma_);
        NumericalPoint cdfGradient(2, 0.0);
        if (x < 0.0) return cdfGradient;
        NumericalScalar expX(exp(-lambda_ * x));
        cdfGradient[0] = x * expX;
        cdfGradient[1] = -lambda_ * expX;
        return cdfGradient;
      }

      /* Get the quantile of the distribution */
      NumericalScalar Exponential::computeScalarQuantile(const NumericalScalar prob,
                                                         const Bool tail,
                                                         const NumericalScalar precision) const
      {
        if (tail) return gamma_ - log(prob) / lambda_;
        return gamma_ - log(1.0 - prob) / lambda_;
      }

      /* Compute the mean of the distribution */
      void Exponential::computeMean() const
      {
        mean_ = NumericalPoint(1, gamma_ + 1.0 / lambda_);
        isAlreadyComputedMean_ = true;
      }

      /* Get the standard deviation of the distribution */
      Exponential::NumericalPoint Exponential::getStandardDeviation() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, 1.0 / lambda_);
      }

      /* Get the skewness of the distribution */
      Exponential::NumericalPoint Exponential::getSkewness() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, 2.0);
      }

      /* Get the kurtosis of the distribution */
      Exponential::NumericalPoint Exponential::getKurtosis() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, 9.0);
      }

      /* Compute the covariance of the distribution */
      void Exponential::computeCovariance() const
      {
        covariance_ = CovarianceMatrix(1);
        covariance_(0, 0) = 1.0 / (lambda_ * lambda_);
        isAlreadyComputedCovariance_ = true;
      }

      /* Get the moments of the standardized distribution */
      Exponential::NumericalPoint Exponential::getStandardMoment(const UnsignedLong n) const
      {
        if (gamma_ == 0.0) return NumericalPoint(1, SpecFunc::Gamma(n + 1));
        const NumericalScalar expGamma(exp(gamma_));
        return NumericalPoint(1, expGamma * (SpecFunc::Gamma(n + 1) - pow(gamma_, n + 1) * SpecFunc::HyperGeom_1_1(n + 1, n + 2, -gamma_) / (n + 1)));
      }

      /* Parameters value and description accessor */
      Exponential::NumericalPointWithDescriptionCollection Exponential::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(2);
        Description description(point.getDimension());
        point[0] = lambda_;
        point[1] = gamma_;
        description[0] = "lambda";
        description[1] = "gamma";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void Exponential::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = Exponential(parametersCollection[0][0], parametersCollection[0][1]);
      }

      /* Lambda accessor */
      void Exponential::setLambda(const NumericalScalar lambda)
      /* throw(InvalidArgumentException) */
      {
        if (lambda <= 0.) throw InvalidArgumentException(HERE) << "Lambda MUST be positive";
        if (lambda != lambda_)
          {
            lambda_ = lambda;
            isAlreadyComputedMean_ = false;
            isAlreadyComputedCovariance_ = false;
            computeRange();
          }
      }

      /* Lambda accessor */
      NumericalScalar Exponential::getLambda() const
      {
        return lambda_;
      }


      /* Gamma accessor */
      void Exponential::setGamma(const NumericalScalar gamma)
      {
        if (gamma != gamma_)
          {
            gamma_ = gamma;
            isAlreadyComputedMean_ = false;
            // The covariance does not depend on gamma
            computeRange();
          }
      }

      /* Gamma accessor */
      NumericalScalar Exponential::getGamma() const
      {
        return gamma_;
      }

      /* Method save() stores the object through the StorageManager */
      void Exponential::save(StorageManager::Advocate & adv) const
      {
        NonEllipticalDistribution::save(adv);
        adv.saveAttribute( "lambda_", lambda_ );
        adv.saveAttribute( "gamma_", gamma_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void Exponential::load(StorageManager::Advocate & adv)
      {
        NonEllipticalDistribution::load(adv);
        adv.loadAttribute( "lambda_", lambda_ );
        adv.loadAttribute( "gamma_", gamma_ );
        computeRange();
      }




    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
