//                                               -*- C++ -*-
/**
 *  @file  SquareMatrix.cxx
 *  @brief SquareMatrix implements the classical mathematical square matrix
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-06-30 10:19:34 +0200 (Thu, 30 Jun 2011) $
 *  Id:      $Id: SquareMatrix.cxx 1972 2011-06-30 08:19:34Z schueller $
 */
#include "SquareMatrix.hxx"
#include "SymmetricMatrix.hxx"
#include "IdentityMatrix.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      CLASSNAMEINIT(SquareMatrix);

      /* Default constructor */
      SquareMatrix::SquareMatrix()
        : Matrix(0, 0)
      {
        // Nothing to do
      }


      /* Constructor with size (dim, which is the same for nbRows_ and nbColumns_ )*/
      SquareMatrix::SquareMatrix(const UnsignedLong dim)
        : Matrix(dim, dim)
      {
        // Nothing to do
      }

      /* Constructor from external collection */
      /* If the dimensions of the matrix and of the collection */
      /* do not match, either the collection is truncated */
      /* or the rest of the matrix is filled with zeros */
      SquareMatrix::SquareMatrix(const UnsignedLong dim,
                                 const Collection<NumericalScalar> & elementsValues)
        : Matrix(dim, dim, elementsValues)
      {
        // Nothing to do
      }


      /* Constructor with implementation */
      SquareMatrix::SquareMatrix(const Implementation & i)
        : Matrix(i)
      {
      }


      /* String converter */
      String SquareMatrix::__repr__() const
      {
        return OSS() << "class=" << getClassName()
                     << " dimension=" << this->getDimension()
                     << " implementation=" << getImplementation()->__repr__();
      }

      /* Get the dimension of the matrix */
      const UnsignedLong SquareMatrix::getDimension() const
      {
        return getImplementation()->getDimension();
      }

      /* SquareMatrix transpose */
      SquareMatrix SquareMatrix::transpose () const
      {
        // Quick return for empty or scalar SquareMatrix
        if (getDimension() <= 1) return (*this);
        return SquareMatrix(new MatrixImplementation(getImplementation()->transpose()));
      }

      /* Matrix additions (must have the same dimensions) */
      SquareMatrix SquareMatrix::operator+ (const SquareMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return SquareMatrix(new MatrixImplementation(*getImplementation() + *(m.getImplementation()) ));
      }

      SquareMatrix SquareMatrix::operator+ (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */
      {
        (m.getImplementation())->symmetrize();
        return SquareMatrix(new MatrixImplementation(*getImplementation() + *(m.getImplementation()) ));
      }

      /* Matrix substractions (must have the same dimensions) */
      SquareMatrix SquareMatrix::operator- (const SquareMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return SquareMatrix(new MatrixImplementation(*getImplementation() - *(m.getImplementation()) ));
      }

      SquareMatrix SquareMatrix::operator- (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */
      {
        (m.getImplementation())->symmetrize();
        return SquareMatrix(new MatrixImplementation(*getImplementation() - *(m.getImplementation()) ));
      }

      /* Matrix multiplications (must have consistent dimensions) */
      SquareMatrix SquareMatrix::operator* (const SquareMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return SquareMatrix(new MatrixImplementation(getImplementation()->genProd(*(m.getImplementation())) ));
      }

      SquareMatrix SquareMatrix::operator* (const SymmetricMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return SquareMatrix(new MatrixImplementation(getImplementation()->symProd(*(m.getImplementation()), 'R') ));
      }

      SquareMatrix SquareMatrix::operator* (const IdentityMatrix & m) const /* throw(InvalidDimensionException) */
      {
        return *this;
      }


      /* Multiplication with a NumericalPoint (must have consistent dimensions) */
      NumericalPoint SquareMatrix::operator * (const NumericalPoint & pt) const /* throw(InvalidDimensionException) */
      {
        return getImplementation()->genVectProd(pt) ;
      }


      /* Multiplication with a NumericalScalar */
      SquareMatrix SquareMatrix::operator * (const NumericalScalar s) const
      {
        return SquareMatrix(new MatrixImplementation(*getImplementation() * s ));
      }

      /* Division by a NumericalScalar*/
      SquareMatrix SquareMatrix::operator / (const NumericalScalar s) const /* throw(InvalidArgumentException) */
      {
        return SquareMatrix(new MatrixImplementation(*getImplementation() / s ));
      }

      /* SquareMatrix integer power */
      SquareMatrix SquareMatrix::power(const UnsignedLong n) const
      {
        return SquareMatrix(new MatrixImplementation(getImplementation()->genPower(n)));
      }

      /* Resolution of a linear system */
      NumericalPoint SquareMatrix::solveLinearSystem(const NumericalPoint & b,
						     const Bool keepIntact)
      {
        return getImplementation()->solveLinearSystemSquare(b, keepIntact);
      }

      Matrix SquareMatrix::solveLinearSystem(const Matrix & b,
					     const Bool keepIntact)
      {
        return getImplementation()->solveLinearSystemSquare(*b.getImplementation(), keepIntact);
      }

      /* Compute determinant */
      NumericalScalar SquareMatrix::computeDeterminant(const Bool keepIntact)
      {
        return getImplementation()->computeDeterminant(keepIntact);
      }

      /* Compute eigenvalues */
      SquareMatrix::NumericalComplexCollection SquareMatrix::computeEigenValues(const Bool keepIntact)
      {
        return getImplementation()->computeEigenValuesSquare(keepIntact);
      }


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
