//                                               -*- C++ -*-
/**
 *  @file  Cobyla.cxx
 *  @brief Cobyla is an actual implementation for
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-07-28 18:31:43 +0200 (Thu, 28 Jul 2011) $
 *  Id:      $Id: Cobyla.cxx 2046 2011-07-28 16:31:43Z schueller $
 */
#include "Cobyla.hxx"
#include "algocobyla.h"
#include "NumericalPoint.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Optim
    {

      typedef OT::Base::Type::NumericalPoint NumericalPoint;

      CLASSNAMEINIT(Cobyla);

      /* Default constructor */
      Cobyla::Cobyla():
        NearestPointAlgorithmImplementation()
      {
        // Nothing to do
      }

      /*
       * @brief  Standard constructor: the problem is defined by a scalar valued function  (in fact, a 1-D vector valued function)
       *         and a level value
       */
      Cobyla::Cobyla(const NumericalMathFunction & levelFunction,
                     const Bool verbose):
        NearestPointAlgorithmImplementation(levelFunction, verbose),
        specificParameters_()
      {
        // Nothing to do
      }

      /*
       * @brief  Standard constructor: the problem is defined by a scalar valued function  (in vact, a 1-D vector valued fnction)
       *         and a level value
       */
      Cobyla::Cobyla(const CobylaSpecificParameters & specificParameters,
                     const NumericalMathFunction & levelFunction,
                     const Bool verbose):
        NearestPointAlgorithmImplementation(levelFunction, verbose),
        specificParameters_(specificParameters)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      Cobyla * Cobyla::clone() const
      {
        return new Cobyla(*this);
      }

      /* Performs the actual computation by calling the Cobyla algorithm
       */
      void Cobyla::run()
      /* throw(InternalException) */
      {
        int n(getStartingPoint().getDimension());
        int m(2);
        NumericalPoint x(getStartingPoint());
        NumericalScalar rhoBeg(specificParameters_.getRhoBeg());
        NumericalScalar rhoEnd(getMaximumAbsoluteError());
        int maxFun(getMaximumIterationsNumber() * x.getDimension());
        cobyla_message message((getVerbose() ? COBYLA_MSG_INFO : COBYLA_MSG_NONE));

        /*
         * cobyla : minimize a function subject to constraints
         *
         * n         : number of variables (>=0)
         * m         : number of constraints (>=0)
         * x         : on input, initial estimate ; on output, the solution
         * rhobeg    : a reasonable initial change to the variables
         * rhoend    : the required accuracy for the variables
         * message   : see the cobyla_message enum
         * maxfun    : on input, the maximum number of function evaluations
         *             on output, the number of function evaluations done
         * calcfc    : the function to minimize (see cobyla_function)
         * state     : used by function (see cobyla_function)
         *
         * The cobyla function returns a code defined in the cobyla_rc enum.
         *
         * extern int cobyla(int n, int m, double *x, double rhobeg, double rhoend,
         *  int message, int *maxfun, cobyla_function *calcfc, void *state);
         */
        int returnCode(cobyla(n, m, &x[0], rhoBeg, rhoEnd, message, &maxFun, Cobyla::ComputeObjectiveAndConstraint, (void*) this));
        /* Strore the result */
        NearestPointAlgorithmImplementation::Result myResult(x, maxFun / x.getDimension(), rhoEnd, rhoEnd / x.norm(), -1.0, -1.0);
        setResult(myResult);
        if (returnCode != 0)
          {
	    LOGWARN(OSS() << "Warning! The Cobyla algorithm failed to converge. The error message is " << cobyla_rc_string[returnCode - COBYLA_MINRC]);
          }
      }

      /* Specific parameters accessor */
      Cobyla::CobylaSpecificParameters Cobyla::getSpecificParameters() const
      {
        return specificParameters_;
      }

      /* Specific parameters accessor */
      void Cobyla::setSpecificParameters(const CobylaSpecificParameters & specificParameters)
      {
        specificParameters_ = specificParameters;
      }

      /* String converter */
      String Cobyla::__repr__() const
      {
        OSS oss;
        oss << "class=" << Cobyla::GetClassName()
            << " " << NearestPointAlgorithmImplementation::__repr__()
            << " specificParameters=" << getSpecificParameters();
        return oss;
      }

      /*
       * Wrapper of the NumericalMathFunction operator() compatible with
       * cobyla signature
       */
      int Cobyla::ComputeObjectiveAndConstraint(int n,
                                                int m,
                                                double *x,
                                                double *f,
                                                double *con,
                                                void *state)
      {
        NearestPointAlgorithmImplementation *algorithm = static_cast<NearestPointAlgorithmImplementation *>(state);

        /* Convert the input vector in OpenTURNS format */
        NumericalPoint inPoint((UnsignedLong)n);
        for(UnsignedLong index = 0; index < (UnsignedLong)(n); ++index) inPoint[index] = x[index];
        /* Compute the level function at inPoint */
        *f = 0.5 * inPoint.norm2();
        /* Compute the constraints at inPoint */
        NumericalPoint constraintValue;
        try
          {
            constraintValue = algorithm->getLevelFunction().operator()(inPoint);
          }
        catch (NumericalMathFunction::InternalException & ex)
          {
            return 1;
          }
        catch (NumericalMathFunction::InvalidArgumentException & ex)
          {
            return 1;
          }
        con[0] = constraintValue[0] + algorithm->getMaximumConstraintError() - algorithm->getLevelValue();
        con[1] = algorithm->getLevelValue() + algorithm->getMaximumConstraintError() - constraintValue[0];

        return 0;
      }

    } /* namespace Optim */
  } /* namespace Base */
} /* namespace OpenTURNS */
