//                                               -*- C++ -*-
/**
 *  @file  Catalog.hxx
 *  @brief Catalog records all the factories into a dictionnary
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: Catalog.hxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#ifndef OPENTURNS_CATALOG_HXX
#define OPENTURNS_CATALOG_HXX

#include <map>         // for std::map
#include "OTprivate.hxx"


namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      /** Forward declaration */
      class PersistentObjectFactory;

      /**
       * @class Catalog
       *
       * @brief %Catalog records all the object's factories into a dictionnary.
       *
       * %Catalog implements the 'singleton' strategy to share an unique dictionnary.
       * The catalog is readonly for common usage but writable only for factories
       * that record themselves within their constructor.
       */

      class Catalog
      {
      private:

        /** These methods allocate and free storage */
        static void Release();
        static void Initialize();


      public:

        /** Return the catalog as a singleton */
        static Catalog & GetInstance();

        /** Return the catalog as a singleton */
        static void Add(const String & factoryName, const PersistentObjectFactory * p_factory);

        /** Get the factory by name */
        static const PersistentObjectFactory & Get(const String & factoryName);


        /** Destructor */
        ~Catalog();

        /** @copydoc Object::__repr__() const */
        String __repr__() const;

      private:

        typedef std::map<String, const PersistentObjectFactory *> CatalogType;

        /** The dictionnary of factories */
        CatalogType catalog_;

        /** Default constructor is private so Catalog can not be instanciated */
        Catalog();

        /** Add a new factory to the catalog */
        void add(const String & factoryName, const PersistentObjectFactory * p_factory);

        /** Get the factory by name */
        const PersistentObjectFactory & get(const String & factoryName) const;

        friend class Catalog_init; /* friendship for static member initialization */
      }; /* end class Catalog */

      /** This struct initializes all static members of Catalog */
      struct Catalog_init {
        Catalog_init();
      }; /* end class Catalog_init */

      static Catalog_init __Catalog_initializer;



    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_CATALOG_HXX */
