//                                               -*- C++ -*-
/**
 *  @file  PenalizedLeastSquaresAlgorithm.cxx
 *  @brief This Penalized Least Squares Algorithm as a functor class
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "PenalizedLeastSquaresAlgorithm.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "Matrix.hxx"
#include "IdentityMatrix.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Optim
    {

      CLASSNAMEINIT(PenalizedLeastSquaresAlgorithm);

      static Base::Common::Factory<PenalizedLeastSquaresAlgorithm> RegisteredFactory("PenalizedLeastSquaresAlgorithm");

      typedef Type::Matrix                                  Matrix;
      typedef Type::IdentityMatrix                          IdentityMatrix;
      typedef Common::InvalidArgumentException              InvalidArgumentException;
      typedef Common::NotSymmetricDefinitePositiveException NotSymmetricDefinitePositiveException;

      /* Default constructor */
      PenalizedLeastSquaresAlgorithm::PenalizedLeastSquaresAlgorithm()
        : Base::Common::PersistentObject()
      {
        // Nothing to do
      }

      /* Parameters constructor */
      PenalizedLeastSquaresAlgorithm::PenalizedLeastSquaresAlgorithm(const NumericalSample & inputSample,
                                                                     const NumericalSample & outputSample,
                                                                     const NumericalMathFunctionCollection & basis,
                                                                     const NumericalScalar penalizationFactor):
        inputSample_(inputSample),
        outputSample_(outputSample),
        weight_(inputSample.getSize(), 1.0),
        basis_(basis),
        penalizationFactor_(penalizationFactor),
        penalizationMatrix_(0),
        coefficients_(0),
        residual_(0.0),
        isAlreadyComputedCoefficients_(false)
      {
        const UnsignedLong size(inputSample.getSize());
        if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation based on an empty sample.";
        if (size != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation based on samples of different size.";
        if (outputSample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: cannot perform penalizedleast squares approximation based on output sample of dimension different from 1.";
        const UnsignedLong basisSize(basis.getSize());
        for (UnsignedLong i = 0; i < basisSize; ++i)
          if (basis[i].getOutputDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can only use basis functions with 1D values.";
        // If the penalization factor is strictly positive, use the identity matrix as a penalization term
        if (penalizationFactor > 0.0) penalizationMatrix_ = IdentityMatrix(basisSize);
      }

      /* Parameters constructor */
      PenalizedLeastSquaresAlgorithm::PenalizedLeastSquaresAlgorithm(const NumericalSample & inputSample,
                                                                     const NumericalSample & outputSample,
                                                                     const NumericalPoint & weight,
                                                                     const NumericalMathFunctionCollection & basis,
                                                                     const NumericalScalar penalizationFactor):
        inputSample_(inputSample),
        outputSample_(outputSample),
        weight_(weight),
        basis_(basis),
        penalizationFactor_(penalizationFactor),
        penalizationMatrix_(0),
        coefficients_(0),
        residual_(0.0),
        isAlreadyComputedCoefficients_(false)
      {
        const UnsignedLong dataSize(inputSample.getSize());
        if (dataSize == 0) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation based on an empty sample.";
        if (dataSize != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation based on samples of different size.";
        if (outputSample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation based on output sample of dimension different from 1.";
        if (dataSize != weight.getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation with weight of improper dimension.";
        for (UnsignedLong i = 0; i < dataSize; ++i)
          if (weight[i] < 0.0) throw InvalidArgumentException(HERE) << "Error: can only use positive weight.";
        const UnsignedLong basisSize(basis.getSize());
        for (UnsignedLong i = 0; i < basisSize; ++i)
          if (basis[i].getOutputDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can only use basis functions with 1D values.";
        // If the penalization factor is strictly positive, use the identity matrix as a penalization term
        if (penalizationFactor > 0.0) penalizationMatrix_ = IdentityMatrix(basisSize);
      }

      /* Parameters constructor */
      PenalizedLeastSquaresAlgorithm::PenalizedLeastSquaresAlgorithm(const NumericalSample & inputSample,
                                                                     const NumericalSample & outputSample,
                                                                     const NumericalPoint & weight,
                                                                     const NumericalMathFunctionCollection & basis,
                                                                     const NumericalScalar penalizationFactor,
                                                                     const CovarianceMatrix & penalizationMatrix):
        inputSample_(inputSample),
        outputSample_(outputSample),
        weight_(weight),
        basis_(basis),
        penalizationFactor_(penalizationFactor),
        penalizationMatrix_(penalizationMatrix),
        coefficients_(0),
        residual_(0.0),
        isAlreadyComputedCoefficients_(false)
      {
        const UnsignedLong dataSize(inputSample.getSize());
        if (dataSize == 0) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation based on an empty sample.";
        if (dataSize != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation based on samples of different size.";
        if (outputSample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation based on output sample of dimension different from 1.";
        if (dataSize != weight.getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot perform penalized least squares approximation with weight of improper dimension.";
        for (UnsignedLong i = 0; i < dataSize; ++i)
          if (weight[i] < 0.0) throw InvalidArgumentException(HERE) << "Error: can only use positive weight.";
        const UnsignedLong basisSize(basis.getSize());
        for (UnsignedLong i = 0; i < basisSize; ++i)
          if (basis[i].getOutputDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can only use basis functions with 1D values.";
        // Check if the penalization matrix has the proper dimension
        if (penalizationMatrix.getDimension() != basisSize) throw InvalidArgumentException(HERE) << "Error: the given penalization matrix has an improper dimension.";
        if (!penalizationMatrix.isPositiveDefinite()) throw NotSymmetricDefinitePositiveException(HERE) << "Error: the given penalization matrix is not strictly definite positive.";
      }

      /* Coefficients accessor */
      PenalizedLeastSquaresAlgorithm::NumericalPoint PenalizedLeastSquaresAlgorithm::getCoefficients()
      {
        if (!isAlreadyComputedCoefficients_) run();
        return coefficients_;
      }

      /* Residual accessor */
      NumericalScalar PenalizedLeastSquaresAlgorithm::getResidual()
      {
        if (!isAlreadyComputedCoefficients_) run();
        return residual_;
      }

      /* Perform the optimization */
      void PenalizedLeastSquaresAlgorithm::run()
      {
        const UnsignedLong basisDimension(basis_.getSize());
        const UnsignedLong sampleSize(inputSample_.getSize());
        Matrix basisMatrix;
        NumericalPoint rightHandSide;
        // If there is a penalization term, augment the dimension of the matrix and the right hand side
        if (penalizationFactor_ > 0.0)
          {
            basisMatrix = Matrix(sampleSize + basisDimension, basisDimension);
            rightHandSide = NumericalPoint(sampleSize + basisDimension, 0.0);
          }
        else
          {
            basisMatrix = Matrix(sampleSize, basisDimension);
            rightHandSide = NumericalPoint(sampleSize, 0.0);
          }
        NumericalPoint weightSquareRoot(sampleSize);
        // Build the right-hand side
        for (UnsignedLong rowIndex = 0; rowIndex < sampleSize; ++rowIndex)
          {
            weightSquareRoot[rowIndex] = sqrt(weight_[rowIndex]);
            rightHandSide[rowIndex] = weightSquareRoot[rowIndex] * outputSample_[rowIndex][0];
          }
        // Build the matrix column-wise
        for (UnsignedLong functionIndex = 0; functionIndex < basisDimension; ++functionIndex)
          {
            NumericalSample functionSample(basis_[functionIndex](inputSample_));
            // Fill-in the matrix column
            for (UnsignedLong rowIndex = 0; rowIndex < sampleSize; ++rowIndex)
              {
                basisMatrix(rowIndex, functionIndex) = functionSample[rowIndex][0] * weightSquareRoot[rowIndex];
              }
          }
        // If there is a penalization term, add the penalization matrix to the basis matrix
        if (penalizationFactor_ > 0.0)
          {
	    const Matrix transposedSquareRootPenalizationMatrix(sqrt(penalizationFactor_) * penalizationMatrix_.computeCholesky());
            for (UnsignedLong i = 0; i < basisDimension; ++i)
	      {
		// The cholesky factor has to be transposed, thus we fill only the upper triangular part of the trailing block
              for (UnsignedLong j = i; j < basisDimension; ++j)
                basisMatrix(sampleSize + i, j) = transposedSquareRootPenalizationMatrix(j, i);
	      }
          }
        coefficients_ = basisMatrix.solveLinearSystem(rightHandSide);
	// The residual is the mean L2 norm of the fitting
        residual_ = (basisMatrix * coefficients_ - rightHandSide).norm() / sampleSize;
        isAlreadyComputedCoefficients_ = true;
      }


      /* Virtual constructor */
      PenalizedLeastSquaresAlgorithm * PenalizedLeastSquaresAlgorithm::clone() const
      {
        return new PenalizedLeastSquaresAlgorithm(*this);
      }


      /* String converter */
      String PenalizedLeastSquaresAlgorithm::__repr__() const
      {
        return OSS() << "class=" << getClassName()
                     << " inputSample=" << inputSample_
                     << " outputSample=" << outputSample_
                     << " weight=" << weight_
                     << " basis=" << basis_
                     << " penalization factor=" << penalizationFactor_
                     << " penalization matrix=" << penalizationMatrix_;
      }


      /* Method save() stores the object through the StorageManager */
      void PenalizedLeastSquaresAlgorithm::save(StorageManager::Advocate & adv) const
      {
        PersistentObject::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void PenalizedLeastSquaresAlgorithm::load(StorageManager::Advocate & adv)
      {
        PersistentObject::load(adv);
      }



    } /* namespace Optim */
  } /* namespace Base */
} /* namespace OpenTURNS */
