/*
 * Copyright 2001-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * $Id: SAMLDateTime.h 453 2005-01-29 00:16:10Z cantor $
 */

/*
 * This is mostly copied from Xerces-C, but they don't seem inclined to produce a usable
 * class, so I had to incorporate my own version of it for now. I can't inherit it
 * since the fields I need are private.
 */

#ifndef _XML_DATETIME_H
#define _XML_DATETIME_H

#include <xercesc/util/XMLDateTime.hpp>

namespace saml
{
    class SAML_EXPORTS SAMLDateTime
    {
    public:
        SAMLDateTime();
        SAMLDateTime(const XMLCh* const);
        SAMLDateTime(time_t epoch);
        SAMLDateTime(const SAMLDateTime&);
        SAMLDateTime& operator=(const SAMLDateTime&);
        ~SAMLDateTime();
    
        inline void setBuffer(const XMLCh* const);
    
        const XMLCh* getRawData() const;
        const XMLCh* getFormattedString() const;
        int getSign() const;
    
        XMLCh* getDateTimeCanonicalRepresentation() const;
        XMLCh* getTimeCanonicalRepresentation() const;
    
        void parseDateTime();
        void parseDate();
        void parseTime();
        void parseDay();
        void parseMonth();
        void parseYear();
        void parseMonthDay();
        void parseYearMonth();
        void parseDuration();
    
        static int compare(const SAMLDateTime* const, const SAMLDateTime* const);
        static int compare(const SAMLDateTime* const, const SAMLDateTime* const, bool);
        static int compareOrder(const SAMLDateTime* const, const SAMLDateTime* const);                                    
    
        int getYear() const {return fValue[CentYear];}
        int getMonth() const {return fValue[Month];}
        int getDay() const {return fValue[Day];}
        int getHour() const {return fValue[Hour];}
        int getMinute() const {return fValue[Minute];}
        int getSecond() const {return fValue[Second];}
        time_t getEpoch() const;
    
    private:
        enum valueIndex {
            CentYear   = 0,
            Month      ,
            Day        ,
            Hour       ,
            Minute     ,
            Second     ,
            MiliSecond ,  //not to be used directly
            utc        ,
            TOTAL_SIZE
        };
    
        enum utcType {
            UTC_UNKNOWN = 0,
            UTC_STD        ,          // set in parse() or normalize()
            UTC_POS        ,          // set in parse()
            UTC_NEG                   // set in parse()
        };
    
        enum timezoneIndex {
            hh = 0,
            mm ,
            TIMEZONE_ARRAYSIZE
        };
    
        static int compareResult(int, int, bool);
        static void addDuration(SAMLDateTime* pDuration, const SAMLDateTime* const pBaseDate, int index);
        static int compareResult(const SAMLDateTime* const, const SAMLDateTime* const, bool, int);
        static inline int getRetVal(int, int);
    
        inline void reset();
        //inline void assertBuffer() const;
        inline void copy(const SAMLDateTime&);
        
        inline void initParser();
        inline bool isNormalized() const;
    
        void getDate();
        void getTime();
        void getYearMonth();
        void getTimeZone(const int);
        void parseTimeZone();
    
        int findUTCSign(const int start);
        int indexOf(const int start, const int end, const XMLCh ch) const;
        int parseInt(const int start, const int end) const;
        int parseIntYear(const int end) const;
        double parseMiliSecond(const int start, const int end) const;
    
        void validateDateTime() const;
        void normalize();
        void fillString(XMLCh*& ptr, valueIndex ind, int expLen) const;
        int  fillYearString(XMLCh*& ptr, valueIndex ind) const;
        void searchMiliSeconds(XMLCh*& miliStartPtr, XMLCh*& miliEndPtr) const;
    
    	bool operator==(const SAMLDateTime& toCompare) const;
    
        static const int DATETIMES[][TOTAL_SIZE];
        int fValue[TOTAL_SIZE];
        int fTimeZone[TIMEZONE_ARRAYSIZE];
        int fStart;
        int fEnd;
        int fBufferMaxLen;
        XMLCh* fBuffer;
    
        double fMiliSecond;
        bool fHasTime;
    };

    inline void SAMLDateTime::setBuffer(const XMLCh* const aString)
    {
        reset();
        fEnd = XMLString::stringLen(aString);
        if (fEnd > 0) {
            if (fEnd > fBufferMaxLen) {
                delete[] fBuffer;
                fBufferMaxLen = fEnd + 8;
                fBuffer = new XMLCh[fBufferMaxLen+1];
            }
            memcpy(fBuffer, aString, (fEnd+1) * sizeof(XMLCh));
        }
    }
    
    inline void SAMLDateTime::reset()
    {
        for ( int i=0; i < XMLDateTime::TOTAL_SIZE; i++ )
            fValue[i] = 0;
    
        fMiliSecond   = 0;
        fHasTime      = false;
        fTimeZone[hh] = fTimeZone[mm] = 0;
        fStart = fEnd = 0;
    
        if (fBuffer)
            *fBuffer = 0;
    }
    
    inline void SAMLDateTime::copy(const SAMLDateTime& rhs)
    {
        for ( int i = 0; i < XMLDateTime::TOTAL_SIZE; i++ )
            fValue[i] = rhs.fValue[i];
    
        fMiliSecond   = rhs.fMiliSecond;
        fHasTime      = rhs.fHasTime;
        fTimeZone[hh] = rhs.fTimeZone[hh];
        fTimeZone[mm] = rhs.fTimeZone[mm];
        fStart = rhs.fStart;
        fEnd   = rhs.fEnd;
    
        if (fEnd > 0) {
            if (fEnd > fBufferMaxLen) {
                delete[] fBuffer;
                fBufferMaxLen = rhs.fBufferMaxLen;
                fBuffer = new XMLCh[fBufferMaxLen+1];
            }
            memcpy(fBuffer, rhs.fBuffer, (fEnd+1) * sizeof(XMLCh));
        }
    }
    
    inline void SAMLDateTime::initParser()
    {
        fStart = 0;   // to ensure scan from the very first beginning
                      // in case the pointer is updated accidentally by someone else.
    }
    
    inline bool SAMLDateTime::isNormalized() const
    {
        return (fValue[XMLDateTime::utc] == XMLDateTime::UTC_STD ? true : false);
    }
    
    inline int SAMLDateTime::getRetVal(int c1, int c2)
    {
        if ((c1 == XMLDateTime::LESS_THAN && c2 == XMLDateTime::GREATER_THAN) ||
            (c1 == XMLDateTime::GREATER_THAN && c2 == XMLDateTime::LESS_THAN))
            return XMLDateTime::INDETERMINATE;
    
        return (c1 != XMLDateTime::INDETERMINATE) ? c1 : c2;
    }

}

#endif
