/*******************************************************************************
*                                                                              *
*   Project BASTOOL for the OpenOffice.org Testtool Environment                *
*                                                                              *
*   (C) Joerg Skottke, 2003                                                    *
*                                                                              *
*   License: GPL, use at own risk                                              *
*                                                                              *
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* The custom functions used in this program */
void exec( char* , int );
void log_write( int, char* );
void get_config();
void print_help();
    
/* Some variables used throughout the program / multiple functions */
int test_id = 0;
char log_file[ 200 ];
char unopkg_dir[ 200 ]; 
char unopkg_userdir[ 200 ];
char unopkg_UserInstallation[ 200 ];
char unopkg_logfile[ 200 ];
char unopkg_LogFile[ 200 ];
int is_remote;

/* Some definitions for better readability */
#define OK 0
#define FAIL 256

#define RETURN_NO_ERR 0
#define RETURN_BAD_ENV 1
#define RETURN_WRONG_USAGE 2
#define RETURN_UNOPKG_CRASH 3
#define RETURN_UNOPKG_MISSING 4

#define LOGFILE "unopkg-test.log"

/*******************************************************************************
*                                                                              *
*   Execute ./unopkg with the given parameters and test against the expected   *
*                                                                              *
*   returnvalue.                                                               *
*                                                                              *
*******************************************************************************/

int main( int argc, char *argv[] )
{

    /* Local variables */
    char sub_command[ 200 ];
    char env_option[ 200 ];
    char command[ 200 ];
    char extension_path[ 300 ];
    char extension_name[ 50 ];
    int iCurrentExtension;
    int iCurrentParameter;
    int iCurrentOption;
    int iCurrentCommand;
    int status;
    
    /* If no parameters are provided: Print the help and exit */
    if ( argc != 3 ) { print_help(); }
    
    /* First argument should be the path to the extension */
    strncpy( extension_path , argv[ 1 ] , 299 );
    printf( "\n Using extension path: %s" , extension_path );
    
    /* Second argument should be the name of the extension itself */
    strncpy( extension_name , argv[ 2 ] , 49 );
    printf( "\n Using extension name: %s" , extension_name );
    
    /* Get the config - read all environmant variables and set defaults */
    get_config();
    
    /* Print some leading info to the logfile */
    log_write( 1 , "Log begin" );

    /* See what's in the user's directory */
    log_write( 0 , "Get the content of the userdirectory" );
    system( "find $HOME/.staroffice8 -name \'*\' >> $HOME/unopkg-test.log" );
    log_write( 0 , "Done reading user directory" );
        
    /* Some extension-independnet calls that are supposed to work (rc = 0) */
    printf( "\n\n Working calls:" );
    exec( "" , OK );
    exec( "--help" , OK );
    exec( "-h" , OK );
    exec( "-V" , OK );
    exec( "list" , OK );
    exec( "list --shared" , OK );
    exec( "list -v" , OK );
    exec( "list --shared -v" , OK );
    exec( "list -env:UserInstallation=file:///tmp" , OK );
    exec( "list -v -env:UserInstallation=file:///tmp" , OK );
    exec( "list --shared -env:UserInstallation=file:///tmp" , OK );
    exec( "list --shared -v -env:UserInstallation=file:///tmp" , OK );    

    /* See what's in the user's directory */
    log_write( 0 , "Get the content of the userdirectory" );
    system( "find $HOME/.staroffice8 -name \'*\' >> $HOME/unopkg-test.log" );
    log_write( 0 , "Done reading user directory" );

    
    /* Some extension independent calls that are supposed to fail (rc = 256) */
    printf( "\n\n Non-working calls:" );
    exec( "list -V" , FAIL );
    exec( "non-existing-command" , FAIL );
    exec( "add non-existing-extension.oxt" , FAIL );
    exec( "add --shared non-existing-extension.oxt" , FAIL );
    exec( "remove non-existing-extension.oxt" , FAIL );
    exec( "remove --shared non-existing-extension.oxt" , FAIL );
    exec( "reinstall non-existing-extension.oxt" , FAIL );
    exec( "reinstall --shared non-existing-extension.oxt" , FAIL );
    exec( "help" , FAIL );    
    exec( "called with a whole bunch of invalid parameters" , FAIL );    

    /* See what's in the user's directory */
    log_write( 0 , "Get the content of the userdirectory" );
    system( "find $HOME/.staroffice8 -name \'*\' >> $HOME/unopkg-test.log" );
    log_write( 0 , "Done reading user directory" );
    
    /* calls which require a working extension add, remove, reinstall 
    Note that i hope that all these tests work ok (they all should return 0) */
    /* These are the most commonly used parameters and modifiers (output)*/
    printf( "\n\n Work with an extension - add, reinstall, remove etc." );
    for ( iCurrentParameter = 1 ; iCurrentParameter <= 9 ; iCurrentParameter++ )
    {
        switch( iCurrentParameter )
        {
            case  1 : strcpy( sub_command, "" ); break;
            case  2 : strcpy( sub_command, "-f " ); break;
            case  3 : strcpy( sub_command, "--force " ); break;
            case  4 : strcpy( sub_command, "-v " ); break;
            case  5 : strcpy( sub_command, "--verbose " ); break;
            case  6 : strcpy( sub_command, "-v -f " ); break;
            case  7 : strcpy( sub_command, "--verbose -f " ); break;
            case  8 : strcpy( sub_command, "--verbose --force " ); break;
            case  9 : strcpy( sub_command, "-v --force " ); break;
        }
        
        /* Some modifiers that alter the behavior (paths) of unopkg */
        for ( iCurrentOption = 1 ; iCurrentOption <= 4 ; iCurrentOption++ )
        {
            switch( iCurrentOption )
            {
                case  1 : strcpy( env_option , "" ); break;
                case  2 : strcpy( env_option , " --shared " ); break;
                case  3 : strcpy( env_option , unopkg_UserInstallation );
                          strcat( env_option , " --shared " ); 
                          break;
                case  4 : strcpy( env_option , unopkg_UserInstallation );
                          strcat( env_option , " " );
                          strcat( env_option , unopkg_LogFile );
                          strcat( env_option , " --shared " );
                          break;
            }
        
            /* the three commands unopkg knows about (according to the spec, which doesn't mean much these days) */
            for ( iCurrentCommand = 1 ; iCurrentCommand <= 8 ; iCurrentCommand++ )
            {
                switch( iCurrentCommand )
                {
                    case 1 : strcpy( command , "add " ); break;
                    case 2 : strcpy( command , "add " ); break;
                    case 3 : strcpy( command , "reinstall " ); break;
                    case 4 : strcpy( command , "reinstall " ); break;
                    case 5 : strcpy( command , "remove " ); break;
                    case 6 : strcpy( command , "remove " ); break;
                    case 7 : strcpy( command , "add " ); break;
                    case 8 : strcpy( command , "remove " ); break;
                }
                
                /* Here the parts of the command line are concatenated and executed */
                strcat( command , sub_command );
                strcat( command , env_option  );
                
                /* When removing an extension we cannot use the full path, only the extension name */
                switch( iCurrentCommand )
                {
                    case 1 : strcat( command , extension_path ); break;
                    case 2 : strcat( command , extension_path ); break;
                    case 3 : strcat( command , extension_path ); break;
                    case 4 : strcat( command , extension_path ); break;
                    case 5 : break;
                    case 6 : break;
                    case 7 : strcat( command , extension_path ); break;
                    case 8 : break;
                }
                strcat( command , extension_name );
                
                /* The commands are executed twice, adding and removing must fail, reinstall is supposed to work */
                switch( iCurrentCommand )
                {
                    case 1 : exec( command ,  OK  ); break;
                    case 2 : exec( command , FAIL ); break;
                    case 3 : exec( command ,  OK  ); break;
                    case 4 : exec( command ,  OK  ); break;
                    case 5 : exec( command ,  OK  ); break;
                    case 6 : exec( command , FAIL ); break;
                    case 7 : exec( command ,  OK  ); break;
                    case 8 : exec( command ,  OK  ); break;
                }
            }
        }

    /* See what's in the user's directory */
    log_write( 0 , "Get the content of the userdirectory" );
    system( "find $HOME/.staroffice8 -name \'*\' >> $HOME/unopkg-test.log" );
    log_write( 0 , "Done reading user directory" );

    }
    
    /* Write a last fancy comment to the log, flush the buffer and return something nice */
    log_write( 0 , "Test end" );
    printf( "\n" );
    fflush( stdout );
    return( RETURN_NO_ERR );
}

/*******************************************************************************
*                                                                              *
*   Execute ./unopkg with the given parameters and test against the expected   *
*                                                                              *
*   returnvalue. Write the output to the log, the result to the screen.        *
*                                                                              *
*******************************************************************************/

void exec( char *command, int rc_expected )
{
    int rc;
    char full_command[ 500 ];
    
    FILE *log;
    
    test_id++;
    full_command[ 0 ] = '\0';
    
    if ( is_remote ) {
        strcpy( full_command , unopkg_dir );
        strcat( full_command , "unopkg " );
    }
    else {
        strcat( full_command , "./unopkg " );
    }
    
    log_write( 0 , command );
    
    /* Build the command line */
    strcat( full_command , command  );
    strcat( full_command , " >>"    );
    strcat( full_command , log_file );
    strcat( full_command , " 2>&1"  );
    
    /* Print out the full command line for debugging */
    printf( "\n Test case %5i: ", test_id );
    fflush( stdout );
    
    /* Execute the command */
    rc = system( full_command );
    
    /* Handle a number of known returnvalues - some come from unopkg, others are provided 
       by the OS */
    switch ( rc )
    {
        case 0 :    if ( rc == rc_expected ) {
                        printf( "rc = %5i ( ok ) for unopkg parameter(s): %s" , rc, command );
                        log_write( 0 , "success" );
                    }
                    else {
                        printf( "rc = %5i (fail) for unopkg parameter(s): %s" , rc, command );
                        log_write( 0 , "success" );
                    }
                    fflush( stdout );
                    break;
        case 256 :  if ( rc == rc_expected ) {
                        printf( "rc = %5i ( ok ) for unopkg parameter(s): %s" , rc, command );
                        log_write( 0 , "success" );
                    }
                    else {
                        printf( "rc = %5i (fail) for unopkg parameter(s): %s" , rc, command );
                        log_write( 0 , "success" );
                    }
                    fflush( stdout );
                    break;
        case 512 :  printf( "\nIt looks as if unopkg could not be found. Aborting" );
                    exit( RETURN_UNOPKG_MISSING );
                    break;
        case 35584 :
                printf( "\n[ Error - unopkg crashed, aborting test ]\n" );
                fflush( stdout );
                exit( RETURN_UNOPKG_CRASH );
                break;
        default  :
                printf( "Unknown rc = %5i (fail) for unopkg parameter(s): %s" , rc, command );
                fflush( stdout );
                log_write( 0 , "Unknown rc - failure" );
    }
       
}

/*******************************************************************************
*                                                                              *
*   Simple logic to write logfiles, append or delete                           *
*                                                                              *
*                                                                              *
*                                                                              *
*******************************************************************************/


void log_write( int imode, char *comment )
{

    FILE *log;
    
    switch ( imode )
    {
        case 0: log = fopen( log_file , "a" ); break; /* Append to log */
        case 1: log = fopen( log_file , "w" ); break; /* Delete log */
    }
    
    /* Some fancy formatting */
    if ( log ) { 
    
        switch( imode )
        {
            case 0: fprintf( log , " -----------------------------------------------------------------------------------\n" );
                    fprintf( log , " Action ID: %5i : %s\n" , test_id , comment ); 
                    fprintf( log , " -----------------------------------------------------------------------------------\n" );
                    break;
            case 1: fprintf( log , " -----------------------------------------------------------------------------------\n" );
                    fprintf( log , " Action: %s\n" , comment ); 
                    fprintf( log , " -----------------------------------------------------------------------------------\n" );
                    break;
        }
        fclose( log );
    } 
    
}   

/*******************************************************************************
*                                                                              *
*   A few parameters of the program can be adjusted via environment variables  *
*                                                                              *
*                                                                              *
*                                                                              *
*******************************************************************************/

void get_config()
{
    /* Note that all of the variables in this function are global ones. Evil 
       but very simple and convenient. */

    /* Find out where unopkg is - if not in local dir UNOPKG_DIR must be set */
    is_remote = 0;
    if ( getenv( "UNOPKGDIR" ) ) { 
        strcpy( unopkg_dir , getenv( "UNOPKGDIR" ) );
        is_remote = 1;
        }
    else { 
        strcpy( unopkg_dir , "/opt/staroffice8/program/" ); 
        is_remote = 1;
        }
        
    /* This is the place the unopkg-test.log file should be written to */
    if ( getenv( "HOME" ) ) { 
        strcpy( log_file , getenv( "HOME" ) ); 
        if ( strstr( log_file, "/" ) ) { strcat( log_file , "/" ); }
        else { strcat( log_file , "\\" ); }
        strcat( log_file , LOGFILE );
    }
    else { 
        printf( "\nCannot determine homedir, set $HOME" ); 
        exit( RETURN_BAD_ENV );
    }
    
    /* Get the UserInstallation directory for -env:UserInstallation... */
    if ( getenv( "UNOPKG_USERDIR" ) ) {
        strcpy( unopkg_userdir  , getenv( "UNOPKG_USERDIR" ) );
    }
    else {
        strcpy( unopkg_userdir , "file:///tmp/unopkg-test" );
    }
    
    /* Build the string */
    strcpy( unopkg_UserInstallation , "-env:UserInstallation=" );
    strcat( unopkg_UserInstallation , unopkg_userdir );    
    
    /* Find out where to write the unopkg-logfile */
    if ( getenv( "UNOPKG_LOGFILE" ) ) {
        strcpy( unopkg_logfile  , getenv( "UNOPKG_LOGFILE" ) );
    }
    else {
        strcpy( unopkg_logfile , "/tmp/unopkg.log" );
    }
    
    /* Build the string */
    strcpy( unopkg_LogFile , "--log-file=" );
    strcat( unopkg_LogFile , unopkg_logfile );        
}

/*******************************************************************************
*                                                                              *
*   A few parameters of the program can be adjusted via environment variables  *
*                                                                              *
*   Describe them here in the help                                             *
*                                                                              *
*******************************************************************************/

void print_help()
{
    printf( "\n" );
    printf( "\n  Usage:" );
    printf( "\n" );
    printf( "\n    unopkg-test <path to extension> <extension.oxt>" );
    printf( "\n" );
    printf( "\n  Use environment variables to change the behavior of the program:" );
    printf( "\n" );
    printf( "\n    HOME           = Where to put the logfile (the one from unopkg-test)" );
    printf( "\n    UNOPKG_USERDIR = The directory to use for the temporary user directory" );
    printf( "\n                     Use a file-URL like file:///tmp/myuser" );
    printf( "\n                     Default: file:///tmp/unopkg-test" );    
    printf( "\n                     The subdirectory \"user\" will be created" );
    printf( "\n                     The .lock file is created beside the user-dir" );
    printf( "\n    UNOPKG_LOGFILE = Where unopkg puts its logfile" );
    printf( "\n                     Default: /tmp/unopkg.log" );
    printf( "\n    UNOPKG_DIR     = Path to the directory containing unopkg" );
    printf( "\n                     The path must contain the trailing pathseparator." );
    printf( "\n                     Default is: /opt/staroffice8/program" );
    printf( "\n" );
    printf( "\n  Hints:" );
    printf( "\n" );
    printf( "\n    * You need to run this tool as root as the shared layer would ńot be" );
    printf( "\n      accessible otherwise. Use with extreme caution." );
    printf( "\n    * HOME must be set, the other variables are optional." );
    printf( "\n    * If not run from the OOo/program dir you need to set UNOPKG_DIR." );
    printf( "\n    * Currently unopkg only knows two returnvalues: 0 and 256." );
    printf( "\n    * Other returnvalues are undefined" );
    printf( "\n\n" );
    exit( RETURN_WRONG_USAGE );
} 
