/*
 * conference.h
 *
 * Conferencing functions for a simple MCU
 *
 * Copyright (c) 2000 Equivalence Pty. Ltd.
 * Copyright (c) 2004 Post Increment
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.x
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of ths code were written by by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Stonevoice, slc. http://www.stonevoice.com
 *
 * Portions of this code were written by Post Increment (http://www.postincrement.com) 
 * with the assistance of funding from Citron Networks (http://www.citron.com.tw)
 *
 * Contributor(s): Derek J Smithies (derek@indranet.co.nz)
 *                 Craig Southeren (craig@postincrement.com)
 *
 * $Log: conference.h,v $
 * Revision 2.15  2006/08/07 06:18:22  csoutheren
 * Fix detection of MCU members
 * Add auto-removal of conferences only containing MCU members
 *
 * Revision 2.14  2006/07/21 08:01:40  csoutheren
 * Fixed conference member detect
 * Re-factored video mixer code slightly
 *
 * Revision 2.13  2006/07/21 05:52:51  csoutheren
 * Add flag to indicate if remote endpoint is an MCU
 *
 * Revision 2.12  2006/07/21 05:08:03  csoutheren
 * Stability fixes and more inline documentation
 * Thanks to Paolo Amadini of Stonevoice
 *
 * Revision 2.11  2006/06/22 04:15:00  csoutheren
 * Fixed for new pwlib
 *
 * Revision 2.10  2006/06/21 06:11:36  csoutheren
 * Fixes for latest pwlib
 *
 * Revision 2.9  2006/06/11 00:14:43  csoutheren
 * Fixed compile error on 64 bit systems
 *
 * Revision 2.8  2006/06/09 04:39:59  csoutheren
 * Migrated VideoBranch to main trunk
 *
 * Revision 2.7.2.17  2006/06/06 08:24:16  csoutheren
 * Added support for echo test room
 * Fix problem with using high frame rates
 *
 * Revision 2.7.2.16  2006/05/31 08:48:02  csoutheren
 * Fixed crash on second call when entry/exit files do not exist
 * Fix crash when Cisco HOLD used. Thanks to Christian Bongiovanni of Stonevoice
 *
 * Revision 2.7.2.15  2006/04/26 15:03:16  csoutheren
 * Added better override for creating conferenc
 *
 * Revision 2.7.2.14  2006/04/26 13:09:08  csoutheren
 * Fix problem when connecting file not available
 * Add optional time limit for rooms
 *
 * Revision 2.7.2.13  2006/04/19 09:26:16  csoutheren
 * Fixes for gcc 4.1.0
 *
 * Revision 2.7.2.12  2006/04/18 03:05:09  csoutheren
 * Fix video mix problem with > 3 members
 * Add test room
 *
 * Revision 2.7.2.11  2006/04/12 04:54:18  csoutheren
 * Fix more problems with audio-only members
 *
 * Revision 2.7.2.10  2006/04/06 08:20:29  csoutheren
 * Retyped conference member identifier to epxlicit type
 * Added support for H.245 terminal added and terminal left
 *
 * Revision 2.7.2.9  2006/04/06 01:11:16  csoutheren
 * Latest sources include
 *   - premedia blanking and optional image display
 *   - ablity to defer conference join for authentication if required
 *   - more bulletproofing on conference join
 *   - new video copy/fill functions
 *
 * Revision 2.7.2.8  2006/04/06 00:50:29  csoutheren
 * Latest changes (more to come)
 *
 * Revision 2.7.2.7  2006/03/28 05:13:38  csoutheren
 * Normalised file headers
 * Fixed problem with QCIF video
 * Seperated H.323 and MCU process functions into seperate files
 *
 * Revision 2.7.2.6  2006/03/27 09:14:14  csoutheren
 * Fixed Unix compile warnings and errors
 * Added VideoTxQuality value
 *
 * Revision 2.7.2.5  2006/03/24 22:49:24  csoutheren
 * Video now working
 *
 * Revision 2.7.2.4  2006/03/21 14:02:49  csoutheren
 * More video mixiing.
 * Still some crashes in video mixer code when members exit conference
 *
 * Revision 2.7.2.3  2006/03/17 07:00:24  csoutheren
 * More video implementation
 * Video mixing now working except copying subimage UV components causes overrun
 * Y is OK, which is why the images are grey. Also need to work out flipping requirements
 *
 * Revision 2.7.2.2  2006/03/14 08:02:50  csoutheren
 * More implemenrtation of video
 * Video mixing infrastructure implemented but not fully working
 *
 * Revision 2.7.2.1  2006/03/06 05:48:15  csoutheren
 * Start of implementation for video. Still in progress
 *
 * Revision 2.6  2006/02/08 06:54:44  csoutheren
 * Added extra information to monitoring page
 *
 * Revision 2.5  2005/06/21 12:45:21  rjongbloed
 * Disabled video so will compile - for now.
 *
 * Revision 2.4  2005/03/11 00:25:24  csoutheren
 * Fixed problems with flags
 *
 * Revision 2.3  2004/05/26 06:54:31  csoutheren
 * Changed to be a PHTTPServiceProcess
 * Added ability to play WAV files on member entry and exit
 * Added additional documentation on all classes
 * Preparation for re-introducing video
 *
 * Revision 2.2  2004/03/31 03:36:38  csoutheren
 * Fixed problem with user indication messages
 * Fixed problems with room listener and unlisten
 *
 * Revision 2.1  2004/03/11 20:49:44  csoutheren
 * Removed warnings
 *
 * Revision 2.0  2004/03/08 02:06:24  csoutheren
 * Totally rewritten to use new connection locking mecahnism
 * Added ability to monitor conferences
 * Added initial support for H.323 MCU messages
 * Thanks to Citron Networks for supporting this work
 *
 */

#ifndef _OpenMCU_CONFERENCE_H
#define _OpenMCU_CONFERENCE_H

#ifdef _WIN32
#pragma warning(disable:4786)
#pragma warning(disable:4100)
#endif

#include "config.h"

#include <ptlib/sound.h>
#include <ptlib/video.h>
#include <ptlib/vconvert.h>
#include <ptclib/delaychan.h>

#include <set>
#include <map>

#define CIF_WIDTH     352
#define CIF_HEIGHT    288
#define CIF_SIZE      (CIF_WIDTH*CIF_HEIGHT*3/2)

#define QCIF_WIDTH    (CIF_WIDTH / 2)
#define QCIF_HEIGHT   (CIF_HEIGHT / 2)
#define QCIF_SIZE     (QCIF_WIDTH*QCIF_HEIGHT*3/2)

typedef void * ConferenceMemberId;

////////////////////////////////////////////////////

class MCULock : public PObject
{
  PCLASSINFO(MCULock, PObject);
  public:
    MCULock();
    BOOL Wait(BOOL hard = FALSE);
    void Signal(BOOL hard = FALSE);
    void WaitForClose();
  protected:
    PMutex mutex;
    BOOL closing;
    int count;
    PSyncPoint closeSync;
};

#if OPENMCU_VIDEO

////////////////////////////////////////////////////

class VideoFrameStoreList {
  public:
    class FrameStore {
      public:
        FrameStore(int _w, int _h)
          : valid(FALSE), width(_w), height(_h)
        { valid = FALSE; PAssert(_w != 0 && _h != 0, "Cannot create zero size framestore"); data.SetSize(_w * _h * 3 / 2); }

      BOOL valid;
      int width;
      int height;
      PBYTEArray data;
    };

    inline unsigned WidthHeightToKey(int width, int height)
    { return width << 16 | height; }

    inline void KeyToWidthHeight(unsigned key, int & width, int & height)
    { width = (key >> 16) & 0xffff; height = (key & 0xffff); }

    ~VideoFrameStoreList();
    FrameStore & AddFrameStore(int width, int height);
    FrameStore & GetFrameStore(int width, int height);
    FrameStore & GetNearestFrameStore(int width, int height, BOOL & found);
    void InvalidateExcept(int w, int h);

    typedef std::map<unsigned, FrameStore *> VideoFrameStoreListMapType;
    VideoFrameStoreListMapType videoFrameStoreList;
};

////////////////////////////////////////////////////

class ConferenceMember;

class MCUVideoMixer
{
  public:
    class VideoMixPosition {
      public:
        VideoMixPosition(ConferenceMemberId _id, 
                         ConferenceMember & mbr, 
                                  int _x = 0, 
                                  int _y = 0, 
                                  int _w = 0, 
                                  int _h = 0);

        virtual ~VideoMixPosition()
        { }

        virtual BOOL WriteSubFrame(MCUVideoMixer & parent, const void * buffer, int width, int height, PINDEX amount)
        { return parent.WriteSubFrame(*this, buffer, width, height, amount); }

        ConferenceMemberId id;
        int xpos;
        int ypos;
        int width;
        int height;
        unsigned audioLevel;
    };

    virtual ~MCUVideoMixer()
    { }

    typedef std::map<ConferenceMemberId, VideoMixPosition *> VideoMixPositionMap;
    VideoMixPositionMap videoPositions;

    int rows;
    int cols;
    int subImageWidth;
    int subImageHeight;

    MCUVideoMixer()
      : subImageWidth(0), subImageHeight(0)
    { }

    virtual MCUVideoMixer * Clone() const = 0;
    virtual BOOL ReadFrame(ConferenceMember & mbr, void * buffer, int width, int height, PINDEX & amount) = 0;
    virtual BOOL WriteFrame(ConferenceMemberId id, const void * buffer, int width, int height, PINDEX amount) = 0;
    virtual void SetAudioLevel(ConferenceMemberId id, unsigned audioLevel) = 0;

    virtual BOOL WriteSubFrame(VideoMixPosition & vmp, const void * buffer, int width, int height, PINDEX amount) = 0;
    virtual void SetSubFrameLevel(VideoMixPosition & vmp, unsigned audioLevel) = 0;

    virtual BOOL AddVideoSource(ConferenceMemberId id, ConferenceMember & mbr) = 0;
    virtual void RemoveVideoSource(ConferenceMemberId id, ConferenceMember & mbr) = 0;

    virtual VideoMixPosition * CreateVideoMixPosition(ConferenceMemberId _id, 
                                          ConferenceMember & _mbr, 
                                                         int _x, 
                                                         int _y,
                                                         int _w, 
                                                         int _h)
    { return new VideoMixPosition(_id, _mbr, _x, _y, _w, _h); }

    static void ConvertRGBToYUV(BYTE R, BYTE G, BYTE B, BYTE & Y, BYTE & U, BYTE & V);
    static void FillYUVFrame(void * buffer, BYTE R, BYTE G, BYTE B, int w, int h);
    static void FillCIFYUVFrame(void * buffer, BYTE R, BYTE G, BYTE B);
    static void FillQCIFYUVFrame(void * buffer, BYTE R, BYTE G, BYTE B);
    static void FillCIFYUVRect(void * frame, BYTE R, BYTE G, BYTE B, int xPos, int yPos, int rectWidth, int rectHeight);
    static void FillYUVRect(void * frame, int frameWidth, int frameHeight, BYTE R, BYTE G, BYTE B, int xPos, int yPos, int rectWidth, int rectHeight);
    static void CopyRectIntoCIF(const void * _src, void * _dst, int xpos, int ypos, int width, int height);
    static void CopyRectIntoQCIF(const void * _src, void * _dst, int xpos, int ypos, int width, int height);
    static void ConvertQCIFToCIF(const void * _src, void * _dst);
};

class MCUSimpleVideoMixer : public MCUVideoMixer
{
  public:
    MCUSimpleVideoMixer(BOOL forceScreenSplit = FALSE);
    virtual MCUVideoMixer * Clone() const
    { return new MCUSimpleVideoMixer(*this); }

    virtual BOOL ReadFrame(ConferenceMember &, void * buffer, int width, int height, PINDEX & amount);
    virtual BOOL WriteFrame(ConferenceMemberId id, const void * buffer, int width, int height, PINDEX amount);
    virtual void SetAudioLevel(ConferenceMemberId id, unsigned audioLevel);
    virtual void SetSubFrameLevel(VideoMixPosition & vmp, unsigned audioLevel);

    virtual BOOL WriteSubFrame(VideoMixPosition & vmp, const void * buffer, int width, int height, PINDEX amount);

    virtual BOOL AddVideoSource(ConferenceMemberId id, ConferenceMember & mbr);
    virtual void RemoveVideoSource(ConferenceMemberId id, ConferenceMember & mbr);
 
  protected:
    virtual void CalcVideoSplitSize(unsigned int imageCount, int & subImageWidth, int & subImageHeight, int & cols, int & rows);
    virtual void ReallocatePositions();
    virtual BOOL ReadMixedFrame(void * buffer, int width, int height, PINDEX & amount);
    BOOL ReadSrcFrame(VideoFrameStoreList & srcFrameStores, void * buffer, int width, int height, PINDEX & amount);

    PMutex mutex;
    BOOL forceScreenSplit;

    VideoFrameStoreList frameStores;  // list of framestores for data

    PBYTEArray imageStore;        // temporary conversion store
    PColourConverter * converter; // CIF to QCIF converter
};

#if ENABLE_TEST_ROOMS
class TestVideoMixer : public MCUSimpleVideoMixer
{
  public:
    TestVideoMixer(unsigned frames);
    BOOL AddVideoSource(ConferenceMemberId id, ConferenceMember & mbr);
    BOOL WriteFrame(ConferenceMemberId id, const void * buffer, int width, int height, PINDEX amount);
    BOOL ReadFrame(ConferenceMember &, void * buffer, int width, int height, PINDEX & amount);

  protected:
    unsigned frames;
    BOOL allocated;
};
#endif // ENABLE_TEST_ROOMS

#if ENABLE_ECHO_MIXER
class EchoVideoMixer : public MCUSimpleVideoMixer
{
  public:
    EchoVideoMixer();
    BOOL AddVideoSource(ConferenceMemberId id, ConferenceMember & mbr);
    BOOL WriteFrame(ConferenceMemberId id, const void * buffer, int width, int height, PINDEX amount);
    BOOL ReadFrame(ConferenceMember &, void * buffer, int width, int height, PINDEX & amount);
};
#endif

////////////////////////////////////////////////////

#endif  // OPENMCU_VIDEO

////////////////////////////////////////////////////

class Conference;

/**
  * this class describes a connection between a conference member and a conference
  * each conference member has one instance of class for every other member of the conference
  */

class ConferenceConnection : public PObject {
  PCLASSINFO(ConferenceConnection, PObject);
  public:
    ConferenceConnection(ConferenceMemberId _id);
    ~ConferenceConnection();

    ConferenceMemberId GetID() const
    { return id; }

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4311)
#endif
    virtual PString GetName() const
#ifdef P_64BIT
    { return psprintf("%lli", id); }
#else
    { return PString(PString::Unsigned, (unsigned)id); }
#endif
#ifdef _WIN32
#pragma warning(pop)
#endif

    virtual void OnUserInputIndication(const PString &)
    { }

    void WriteAudio(ConferenceMemberId source, const void * buffer, PINDEX amount);
    void Write(const BYTE * ptr, PINDEX amount);
    void ReadAudio(BYTE * ptr, PINDEX amount);
    void ReadAndMixAudio(BYTE * ptr, PINDEX amount, PINDEX channels);

  protected:
    Conference * conference;
    ConferenceMemberId id;

    void Mix(BYTE * dst, const BYTE * src, PINDEX count, PINDEX channels);

    BYTE * buffer;
    PINDEX bufferLen;     ///Number of bytes unread in the buffer.
    PINDEX bufferStart;   ///Current position in the buffer.
    PINDEX bufferSize;    ///Total number of bytes in buffer. Never gets changed.
    PMutex audioBufferMutex;
};

////////////////////////////////////////////////////

/**
  * this class describes a member of a conference
  */

class ConferenceManager;

class ConferenceMember : public PObject
{
  PCLASSINFO(ConferenceMember, PObject);
  public:
    typedef std::map<ConferenceMemberId, ConferenceConnection *> ConnectionListType;
    typedef std::map<ConferenceMemberId, ConferenceMember *> MemberListType;

    /**
      * create a new conference member. The single parameter is an "id" (usually a pointer) 
      * that can used to identify this member unambiguously
      */
    ConferenceMember(Conference * conference, ConferenceMemberId id, BOOL isMCU = FALSE);

    /**
      * destroy the conference member
      */
    ~ConferenceMember();

    /**
      * used to pre-emptively close a members connection
      */
    virtual void Close()
    { }

    /**
      * used to add a conference member to a conference. This is not done in the constructor
      * as some conference members have non-trivial startup requirements
      */
    virtual BOOL AddToConference(Conference * conference);

    /**
      * used to remove a conference member from a conference. This is not done in the destructor
      * as some conference members have non-trivial shutdown requirements
      */
    virtual void RemoveFromConference();

    /**
      * If this returns TRUE, the conference member will be visible in all publically displayed
      * conference lists. It will always be visible in the console displays
      */
    virtual BOOL IsVisible() const
    { return TRUE; }

    /**
      * return the conference member ID
      */
    ConferenceMemberId GetID() const
    { return id; }

    PTime GetStartTime() const
    { return startTime; }
     
#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4311)
#endif

    /**
      * return the title that should be used for the conference member
      */
    virtual PString GetTitle() const
#ifdef P_64BIT
    { return psprintf("%lli", id); }
#else
    { return PString(PString::Unsigned, (unsigned)id); }
#endif

#ifdef _WIN32
#pragma warning(pop)
#endif

    /**
      * return the conference this member belongs to
      */
    Conference * GetConference()
    { return conference; }

    /**
      * add a new connection for the specified member to this member to the internal list of connections
      */
    virtual void AddConnection(ConferenceMember * newMember);

    /**
      * remove any connections belong to the specified ID from the internal list of connections
      */
    virtual void RemoveConnection(ConferenceMemberId id);

    /**
     * This is called when the conference member want to send a user input indication to the the conference.
     * By default, this routines calls OnReceivedUserInputIndication for all of the other conference members
     */
    virtual void SendUserInputIndication(const PString & str);

    /**
     * this virtual function is called when the conference sends a user input indication to the endpoint
     * the conference
     */
    virtual void OnReceivedUserInputIndication(const PString & str)
    { }

    /**
      *  Called when the conference member want to send audio data to the cofnerence
      */
    virtual void WriteAudio(const void * buffer, PINDEX amount);

    /**
      *  Called when the conference member wants to read a block of audio from the conference
      *  By default, this calls ReadMemberAudio on the conference
      */
    virtual void ReadAudio(void * buffer, PINDEX amount);

    /**
      * Called when another conference member wants to send audio to the endpoint
      * By default, the audio is added to the queue for the specified member
      * so it can be retreived by a later call to OnIncomingAudio
      */
    virtual void OnExternalSendAudio(ConferenceMemberId id, const void * buffer, PINDEX amount);

    /**
      * called when another conference member wants to write a video frame to this endpoint
      * this will only be called when the conference is not "use same video for all members"
      */
    virtual void OnExternalSetAudioLevel(ConferenceMemberId id, unsigned audioLevel);

#if OPENMCU_VIDEO
    /**
      *  Called when the conference member wants to send video data to the conference
      */
    virtual void WriteVideo(const void * buffer, int width, int height, PINDEX amount);

    /**
      *  Called when a conference member wants to read a block of video from the conference
      *  By default, this calls ReadMemberVideo on the conference
      */
    virtual void ReadVideo(void * buffer, int width, int height, PINDEX & amount);

    /**
      * Called when another conference member wants to read video from the endpoint
      * UnlockExternalVideo must be called after video has been used
      */
    virtual void * OnExternalReadVideo(ConferenceMemberId /*id*/, int width, int height, PINDEX & /*amount*/);

    virtual void UnlockExternalVideo();

    /**
      * called when another conference member wants to write a video frame to this endpoint
      * this will only be called when the conference is not "use same video for all members"
      */
    virtual void OnExternalSendVideo(ConferenceMemberId id, const void * buffer, int width, int height, PINDEX amount);

    /**
      * called to when a new video source added
      */
    virtual BOOL AddVideoSource(ConferenceMemberId id);

    /**
      * called to when a new video source removed
      */
    virtual void RemoveVideoSource(ConferenceMemberId id);

    virtual BOOL OnIncomingVideo(const void * buffer, int width, int height, PINDEX amount);
    virtual BOOL OnOutgoingVideo(void * buffer, int width, int height, PINDEX & amount);

    double GetVideoTxFrameRate() const
    { 
      if (totalVideoFramesSent == 0) 
        return 0.0; 
      else 
        return totalVideoFramesSent * 1000.0 / ((PTime() - firstFrameSendTime).GetMilliSeconds()); }

    double GetVideoRxFrameRate() const
    { 
      if (totalVideoFramesReceived == 0)
        return 0.0; 
      else 
        return totalVideoFramesReceived * 1000.0 / ((PTime() - firstFrameReceiveTime).GetMilliSeconds()); 
    }
#endif

    /*
     *  Used to create a conference connection for this member
     */
    virtual ConferenceConnection * CreateConnection() = 0;

    void WaitForClose()
    { lock.WaitForClose(); }

    /*
     * used to output monitor information for the member
     */
    virtual PString GetMonitorInfo(const PString & hdr);

    ConnectionListType & GetConnectionList()
    { return connectionList; }

    unsigned GetAudioLevel() const
    { return audioLevel; }

    virtual int GetTerminalNumber() const             { return terminalNumber; }
    virtual void SetTerminalNumber(int n)             { terminalNumber = n; }

    void SetJoined(BOOL isJoinedNow)
    { memberIsJoined = isJoinedNow; }

    BOOL IsJoined() const
    { return memberIsJoined; }

    BOOL IsMCU() const
    { return isMCU; }

  protected:
    Conference * conference;
    ConferenceMemberId id;
    BOOL memberIsJoined;
    MCULock lock;
    ConnectionListType connectionList;
    MemberListType memberList;
    PTime startTime;
    unsigned audioLevel;
    int terminalNumber;
    BOOL isMCU;

#if OPENMCU_VIDEO
    //PMutex videoMutex;
    MCUVideoMixer * videoMixer;

    VideoFrameStoreList memberFrameStores;
    PMutex memberFrameStoreMutex;
    PColourConverter * fsConverter; 

    PTime firstFrameSendTime;
    PINDEX totalVideoFramesSent;

    PTime firstFrameReceiveTime;
    PINDEX totalVideoFramesReceived;
#endif
};


////////////////////////////////////////////////////

template <class KeyType>
class MCUNumberMapType : public std::map<int, KeyType>
{
  public:
    typedef std::map<int, KeyType> Ancestor;
    int GetNumber(const KeyType & id)
    {
      PWaitAndSignal m(mutex);
      int mcuNumber = 1;
      if (Ancestor::size() != 0) {
        mcuNumber = 1 + Ancestor::begin()->first;
        while (Ancestor::find(mcuNumber) != Ancestor::end())
          ++mcuNumber;
      }
      Ancestor::insert(std::pair<int, KeyType>(mcuNumber, id));
      return mcuNumber;
    }

    void RemoveNumber(int mcuNumber)
    { PWaitAndSignal m(mutex); Ancestor::erase(mcuNumber); }

  protected:
    PMutex mutex;
};

////////////////////////////////////////////////////

class ConferenceMonitorInfo;

/**
  * this class describes a conference or "room"
  */

class Conference : public PObject
{
  PCLASSINFO(Conference, PObject);
  public:
    typedef std::map<void *, ConferenceMember *> MemberList;

    Conference(ConferenceManager & manager,     
      const OpalGloballyUniqueID & _guid,
                   const PString & _number,
                   const PString & _name,
                               int _mcuNumber
#if OPENMCU_VIDEO
                  ,MCUVideoMixer * _videoMixer = NULL
#endif
                   );

    ~Conference();

    PMutex & GetMutex()
    { return memberListMutex; }

    ConferenceManager & GetManager()
    { return manager; }

    /**
      * add the specified member to the conference
      */
    BOOL AddMember(ConferenceMember * member);

    /**
     * remove the specifed member from the conference.
     * Note that this function does not actually delete the conference member
     * as sometimes a conference member needs to remove itself from a conference
     * 
     * @return if TRUE, the conference is now empty
     */
    BOOL RemoveMember(ConferenceMember * member);

    MemberList & GetMemberList() 
    { return memberList; }

    int GetMemberCount() const
    { PWaitAndSignal m(memberListMutex); return (int)memberList.size(); }

    int GetVisibleMemberCount() const;

    virtual PString GetName() const
    { return name; }

    virtual PString GetNumber() const
    { return number; }

    OpalGloballyUniqueID GetID() const
    { return guid; }

    virtual BOOL IsVisible() const
    { return TRUE; }

    PTime GetStartTime() const
    { return startTime; }

    PINDEX GetMaxMemberCount() const
    { return maxMemberCount; }

    int GetMCUNumber() const
    { return mcuNumber; }

    virtual BOOL BeforeMemberJoining(ConferenceMember *);

    virtual void OnMemberJoining(ConferenceMember *);

    virtual void OnMemberLeaving(ConferenceMember *);

    virtual void ReadMemberAudio(ConferenceMember * member, void * buffer, PINDEX amount);

    virtual void WriteMemberAudioLevel(ConferenceMember * member, unsigned audioLevel);

#if OPENMCU_VIDEO
    virtual void ReadMemberVideo(ConferenceMember * member, void * buffer, int width, int height, PINDEX & amount);

    virtual BOOL WriteMemberVideo(ConferenceMember * member, const void * buffer, int width, int height, PINDEX amount);

    virtual BOOL UseSameVideoForAllMembers()
    { return videoMixer != NULL; }

    virtual MCUVideoMixer * GetVideoMixer() const
    { return videoMixer; }
#endif

    void AddMonitorEvent(ConferenceMonitorInfo * info);

  protected:
    ConferenceManager & manager;
    PMutex memberListMutex;
    MemberList memberList;
    PINDEX maxMemberCount;

    OpalGloballyUniqueID guid;
    PString number;
    PString name;
    int mcuNumber;
    PTime startTime;
    MCUNumberMapType<ConferenceMemberId> terminalNumberMap;
    BOOL mcuMonitorRunning;

#if OPENMCU_VIDEO
    MCUVideoMixer * videoMixer;
#endif
};

////////////////////////////////////////////////////

class ConferenceMonitorInfo : public PObject
{
  PCLASSINFO(ConferenceMonitorInfo, PObject);
  public:
    ConferenceMonitorInfo(const OpalGloballyUniqueID & _guid, const PTime & endTime)
      : guid(_guid), timeToPerform(endTime) { }

    OpalGloballyUniqueID guid;
    PTime timeToPerform;

    virtual BOOL Perform(Conference &) = 0;
};

class ConferenceTimeLimitInfo : public ConferenceMonitorInfo
{
  public:
    ConferenceTimeLimitInfo(const OpalGloballyUniqueID & guid, const PTime & endTime)
      : ConferenceMonitorInfo(guid, endTime)
    { }

    BOOL Perform(Conference & conference);
};

class ConferenceRepeatingInfo : public ConferenceMonitorInfo
{
  public:
    ConferenceRepeatingInfo(const OpalGloballyUniqueID & guid, const PTimeInterval & _repeatTime)
      : ConferenceMonitorInfo(guid, PTime() + _repeatTime), repeatTime(_repeatTime)
    { }

    BOOL Perform(Conference & conference);

  protected:
    PTimeInterval repeatTime;
};

class ConferenceMCUCheckInfo : public ConferenceRepeatingInfo
{
  public:
    ConferenceMCUCheckInfo(const OpalGloballyUniqueID & guid, const PTimeInterval & _repeatTime)
      : ConferenceRepeatingInfo(guid, _repeatTime)
    { }

    BOOL Perform(Conference & conference);
};


class ConferenceMonitor : public PThread
{
  PCLASSINFO(ConferenceMonitor, PThread);
  public:
    ConferenceMonitor(ConferenceManager & _manager)
      : PThread(10000, NoAutoDeleteThread), manager(_manager)
    { Resume(); }

    void Main();
    void AddMonitorEvent(ConferenceMonitorInfo * info);
    void RemoveForConference(const OpalGloballyUniqueID & guid);

    typedef std::vector<ConferenceMonitorInfo *> MonitorInfoList;
    BOOL running;

  protected:
    ConferenceManager & manager;
    PMutex mutex;
    MonitorInfoList monitorList;
};

////////////////////////////////////////////////////


typedef std::map<OpalGloballyUniqueID, Conference *> ConferenceListType;

class ConferenceManager : public PObject
{
  PCLASSINFO(ConferenceManager, PObject);
  public:
    ConferenceManager();
    ~ConferenceManager();

    /**
     * Make a new conference with the specified conference ID, number and name
     */
    Conference * MakeAndLockConference(
      const OpalGloballyUniqueID & conferenceID, 
      const PString & number, 
      const PString & name
    );

    /**
     * Make a new conference with the specified number and name, and use a new conference ID
     */
    Conference * MakeAndLockConference(
      const PString & number, 
      const PString & name
    );
    Conference * MakeAndLockConference(
      const PString & number
    )
    { return MakeAndLockConference(number, PString::Empty()); }

    void UnlockConference()
    { conferenceListMutex.Signal(); }

    /**
      * return true if a conference with the specified ID exists
      */
    BOOL HasConference(
      const OpalGloballyUniqueID & conferenceID,
      PString & room
    );
    BOOL HasConference(
      const OpalGloballyUniqueID & conferenceID
    )
    { PString r; return HasConference(conferenceID, r); }

    /**
      * return true if a conference with the specified number exists
      */
    BOOL HasConference(
      const PString & number,
      OpalGloballyUniqueID & conferenceID
    );
    BOOL HasConference(
      const PString & number
    )
    { OpalGloballyUniqueID i; return HasConference(number, i); }

    /**
      * Remove and delete the specified conference
      */
    void RemoveConference(const OpalGloballyUniqueID & confId);

    /**
      * Remove the specified member from the specified conference.
      * The member will will deleted, and if the conference is empty after the removal, 
      * it is deleted too
      */
    void RemoveMember(const OpalGloballyUniqueID & confId, ConferenceMember * toRemove);

    PMutex & GetConferenceListMutex()
    { return conferenceListMutex; }

    ConferenceListType & GetConferenceList()
    { return conferenceList; }

    virtual void OnCreateConference(Conference *);

    virtual void OnDestroyConference(Conference *)
    { }

    virtual BOOL BeforeMemberJoining(Conference *, ConferenceMember *)
    { return TRUE; }

    virtual void OnMemberJoining(Conference *, ConferenceMember *)
    { }

    virtual void OnMemberLeaving(Conference *, ConferenceMember *)
    { }

    PINDEX GetMaxConferenceCount() const
    { return maxConferenceCount; }

    void AddMonitorEvent(ConferenceMonitorInfo * info);

  protected:
    virtual Conference * CreateConference(const OpalGloballyUniqueID & _guid,
                                                       const PString & _number,
                                                       const PString & _name,
                                                                   int mcuNumber);

    PMutex conferenceListMutex;       
    ConferenceListType conferenceList;
    PINDEX maxConferenceCount;
    MCUNumberMapType<OpalGloballyUniqueID> mcuNumberMap;
    ConferenceMonitor * monitor;
};

#endif  // _OpenMCU_CONFERENCE_H

