/***************************************************************************
                          jdxbase.h  -  description
                             -------------------
    begin                : Sun Jun 6 2004
    copyright            : (C) 2001 by Thies H. Jochimsen
    email                : jochimse@cns.mpg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef JDXBASE_H
#define JDXBASE_H


#include <tjutils/tjutils.h>
#include <tjutils/tjlabel.h>
#include <tjutils/tjlist.h>
#include <tjutils/tjarray.h>


// some defines for Bruker-type strings
#define _BRUKER_MODE_STRING_CAP_START_ 1000
#define _BRUKER_MODE_STRING_MIN_SIZE_ 256
#define _BRUKER_MODE_STRING_CAP_FACTOR_ 3


/**
  * @addtogroup jcampdx
  * @{
  */

/**
  *
  * This enum performs some fine tuning with the JcampDx classes according
  * its value:
  * - bruker: Strings are treated as arrays of characters (size after '=' and contents on next line).
  *           Used to be compatible with Brukers JCAMP-DX implementation.
  * - notBroken: Strings are written directly into the file (after '=').
  */
enum compatMode {bruker,notBroken};


/**
  * This enum determines whether the parameter can be seen/edited in the graphical user interface
  * - edit: Parameter is visible and can be edited
  * - noedit: Parameter is visible but cannot be edited
  * - hidden: Parameter is invisible
  */
enum parameterMode {edit,noedit,hidden};


/**
  * This enum determines whether the parameter will be included/compressed/excluded when reading/writing a file
  * - include: Parameter is included in the file
  * - compressed: Parameter is included in the file in compressed ASCII-format (e.g. Base64 encoding)
  *   if its representation exceeds a certain size
  * - exclude: Parameter is excluded from the file
  */
enum fileMode {include,compressed,exclude};



///////////////////////////////////////////////////////////////////////


class JcampDxBlock; // Forward Declaration
class JDXfunction; // Forward Declaration
class JDXtriple; // Forward Declaration
class JDXaction;  // Forward Declaration
class JDXenum;  // Forward Declaration
class JDXfileName; // Forward Declaration
class JDXformula; // Forward Declaration
class JDXrecoIndices; // Forward Declaration


///////////////////////////////////////////////////////////////////////

/**
  *
  *  Helper class for debugging the JcampDx component
  */
class JcampDx {
 public:
  static const char* get_compName();
};

///////////////////////////////////////////////////////////////////////

/**
  * Enum for plot/display scales/axes in the GUI
  * - displayScale: Scale for 2D/3D display
  * - xPlotScale:  x-axis (bottom)
  * - yPlotScaleLeft:  First y-axis (left)
  * - yPlotScaleRight:  Second y-axis (left)
  */
enum scaleType {displayScale=0,xPlotScale,yPlotScaleLeft,yPlotScaleRight,n_ScaleTypes};


/**
  * Data structure to store info about scales of JDXarrays in GUI
  */
struct ArrayScale {

  ArrayScale() : minval(0), maxval(0) {}

  ArrayScale(const STD_string& scalelabel, const STD_string& scaleunit, float scalemin, float scalemax);

  STD_string get_label_with_unit() const;

/**
  * Axis label
  */
  STD_string label;

/**
  * Axis physical unit
  */
  STD_string unit;

/**
  * Lower and upper bound of scale
  */
  float minval,maxval;
};

///////////////////////////////////////////////////////////////////////

/**
  * Data structure to store info about 2D/3D plots of JDXarrays
  */
struct PixmapProps {

  PixmapProps() : minsize(128), maxsize(1024), autoscale(true), color(false), overlay_minval(0), overlay_maxval(0), overlay_firescale(false), overlay_rectsize(0.8f) {}


  void get_overlay_range(float& min, float& max) const;

/**
  * The min and max width/height in the GUI
  */
  unsigned int minsize, maxsize;

/**
  * Do auto-windowing/scaling of values
  */
  bool autoscale;

/**
  * Display values color encoded
  */
  bool color;


/**
  * Extra map to be overlaid on plot in color code
  */
  farray overlay_map;

/**
  * Range of displayed values of overlay map. If both are zero, autoscaling is applied.
  */
  float  overlay_minval, overlay_maxval;

/**
  * Use only yellow to red from color wheel
  */
  bool overlay_firescale;

/**
  * The relative size (between 0 and 1) of rectangles representing overlayed voxels
  */
  float overlay_rectsize;
};


///////////////////////////////////////////////////////////////////////

/**
  * Data structure to store info about GUI display of JDXarrays
  */
struct GuiProps {
  GuiProps() : fixedsize(true) {}

/**
  * The scales in 1D or 2D display
  */
  ArrayScale scale[n_ScaleTypes];

/**
  * Whether the size is fixed in 1D mode. If so, it is possible to detach the widget
  * to display it in larger size.
  */
  bool fixedsize;

/**
  * Properties of the map in 2D mode
  */
  PixmapProps pixmap;
};

///////////////////////////////////////////////////////////////////////


/**
  *
  * Enum for PARX code generation:
  * - parx_def: Parameter definitions
  * - parx_passval: Parameter assignment
  * - parx_passval_head: Function head for parameter assignment
  * - parx_parclass_def: Definition of parameter class
  * - parx_parclass_init: Initialization of parameter class
  * - parx_parclass_passval: Assignment of parameter class
  */
enum parxCodeType {parx_def, parx_passval, parx_passval_head, parx_parclass_def, parx_parclass_init, parx_parclass_passval};


/**
  * Data structure to store info about the equivalent PARX(Bruker) parameter
  * to which this parameter will be mapped
  */
struct ParxEquiv {
  ParxEquiv() : factor(1.0), offset(0.0) {}

/**
  * Parx name
  */
  STD_string name;

/**
  * Parx type
  */
  STD_string type;

/**
  * Scaling factor in assignment
  */
  double factor;

/**
  * Offset in assignment
  */
  double offset;

};

///////////////////////////////////////////////////////////////////////

/**
  *
  *  This is the (virtual) base class
  *  that carries the JcampDx specific interface, i.e. all common
  *  functions that are specific to JcampDx. Most of the elements
  *  show a simple default behaviour that will be overwritten by
  *  a reasonable behaviour in the derived classes.
  */
class JcampDxClass : public ListItem<JcampDxClass>, public virtual Labeled {

 public:

  virtual ~JcampDxClass();

/**
  *
  *  This function returns the parameter in JcampDx-ASCII-format
  */
  virtual STD_string print() const;

/**
  *  Passes the parameter in ASCII format to the ostream 'os'
  */
  virtual STD_ostream& print2stream(STD_ostream& os) const {return os << printvalstring();}

/**
  *  Stream operator
  */
  friend STD_ostream& operator << (STD_ostream& s,const JcampDxClass& value) { return value.print2stream(s);}


/**
  *  Parses the JCAMP-DX parameter. Except for blocks, this is done by
  *  stripping the label (e.g. "##$MyParameter=" ) from the string and passing the
  *  result to the function 'parsevalstring' of the underlying data type.
  *  For blocks, a re-implemented 'parse' function is called.
  *  Returns 'true' if the parameter was parsed successfully.
  */
  virtual bool parse(STD_string& parstring);

/**
  *
  * Loads the parameter(s) from a JCAMP-DX file. In case of calling this function
  * of a single paramter, the file is searched for this parameter and the value
  * is assigned. In case of a parameter block, all parameter values are taken
  * from the file.
  * The return value is the number of parameters which are successfully parsed.
  * If an error occurs, -1 is returned.
  */
  virtual int load(const STD_string &filename);

/**
  *
  * Writes the parameter(s) to a JCAMP-DX file. In case of calling this function
  * of a single paramter, the file will consist of only this parameter. In case
  * of a parameter block, all parameter values are written to the file.
  * If an error occurs, -1 is returned.
  */
  virtual int write(const STD_string &filename) const;


/**
  *
  *  Parses and assigns the value in the given string
  */
  virtual bool parsevalstring (const STD_string&) = 0;

/**
  *
  *  Returns the value of the parameter as a string
  */
  virtual STD_string printvalstring() const = 0;


/**
  *
  *  Sets the compatability mode. See the documentation to 'compatMode'
  */
  virtual JcampDxClass& set_compatmode(compatMode compat_mode) {compatmode=compat_mode; return *this;}

/**
  *
  *  Returns the compatability mode. See the documentation to 'compatMode'
  */
  virtual compatMode get_compatmode() const {return compatmode;}


/**
  *  Returns a string describing the type of the parameter
  */
  virtual const char* get_typeInfo() const = 0;


/**
  * Returns the minimum allowed value of the parameter.
  * Only used when editing the parameter in a GUI
  */
  virtual double get_minval() const {return 0.0;}


/**
  * Returns the maximum allowed value of the parameter
  * Only used when editing the parameter in a GUI
  */
  virtual double get_maxval() const {return 0.0;}

/**
  * Returns whether the paramerer has a valid interval of allowed values.
  * Only used when editing the parameter in a GUI
  */
  bool has_minmax() const {return get_maxval()>get_minval();}


/**
  *  Returns a deep copy of the parameter
  */
  virtual JcampDxClass* create_copy() const = 0;



/**
  *
  *  Determines whether the parameter is user defined, see [1,2] for details.
  */
  JcampDxClass& set_userDefParameter(bool userDef) {userDefParameter=userDef; return *this;}


/**
  *
  *  Returns whether the parameter is user defined, see [1,2] for details.
  */
  bool isUserDefParameter() const {return userDefParameter;}


/**
  *
  *  Returns the description of the parameter, used in the text of tooltips in a GUI 
  */
  const STD_string& get_description() const {return description;}

/**
  *
  *  Sets the description of the parameter, used in the text of tooltips in a GUI
  */
  JcampDxClass&  set_description(const STD_string& descr) {description=descr; return *this;}


/**
  * Returns possible alternative values of the parameter, i.e. the item strings of an enum
  */
  virtual svector get_alternatives() const {return svector();}


/**
  *
  *  Returns the physical unit of the parameter
  */
  const STD_string& get_unit() const {return unit;}

/**
  *
  *  Sets the physical unit of the parameter
  */
  JcampDxClass& set_unit(const STD_string& un) {unit=un; return *this;}

  
/**
  *
  *  Returns the parameterMode
  */
  virtual parameterMode get_parmode() const {return parmode;}

/**
  *
  *  Sets the parameterMode
  */
  virtual JcampDxClass& set_parmode(parameterMode parameter_mode) {parmode=parameter_mode; return *this;}

/**
  *
  *  Returns the fileMode
  */
  virtual fileMode get_filemode() const {return filemode;}

/**
  *
  *  Sets the fileMode
  */
  virtual JcampDxClass& set_filemode(fileMode file_mode) {filemode=file_mode; return *this;}



/**
  *
  *  Returns the properties of axis display in the GUI, only useful for JDXarrays
  */
  virtual GuiProps get_gui_props() const {return GuiProps();}

/**
  *
  *  Sets the properties of axis display in the GUI, only useful for JDXarrays
  */
  virtual JcampDxClass& set_gui_props(const GuiProps&) {return *this;}



/**
  *
  * Returns C code that can be used together with the PARX(Bruker) compiler.
  */
  virtual STD_string get_parx_code(parxCodeType type, const ParxEquiv& equiv=ParxEquiv()) const;

/**
  *
  *  Returns the name and scaling of an equivalent parameter in PARX. Code for automatic
  *  mapping to the PARX parameter space can be generated using the function
  *  get_parx_code().
  */
  virtual ParxEquiv get_parx_equiv() const {ParxEquiv pe; pe.type=get_typeInfo(); return pe;}




  // virtual functions to emulate a dynamic cast
  virtual JcampDxBlock* cast(JcampDxBlock*) {return 0;}
  virtual int*          cast(int*)          {return 0;}
  virtual long*         cast(long*)         {return 0;}
  virtual float*        cast(float*)        {return 0;}
  virtual double*       cast(double*)       {return 0;}
  virtual bool*         cast(bool*)         {return 0;}
  virtual STD_complex*  cast(STD_complex*)  {return 0;}
  virtual JDXenum*      cast(JDXenum*)      {return 0;}
  virtual STD_string*   cast(STD_string*)   {return 0;}
  virtual JDXfileName*  cast(JDXfileName*)  {return 0;}
  virtual JDXformula*   cast(JDXformula*)   {return 0;}
  virtual JDXaction*    cast(JDXaction*)    {return 0;}
  virtual iarray*       cast(iarray*)       {return 0;}
  virtual darray*       cast(darray*)       {return 0;}
  virtual farray*       cast(farray*)       {return 0;}
  virtual carray*       cast(carray*)       {return 0;}
  virtual sarray*       cast(sarray*)       {return 0;}
  virtual JDXtriple*    cast(JDXtriple*)    {return 0;}
  virtual JDXfunction*  cast(JDXfunction*)  {return 0;}


  // with these functions it is possible to assign a unique id to each paramerer
  int set_parameter_id(int newid) {id=newid; return 0;}
  int get_parameter_id() const {return id;}

  JcampDxClass& set_cmdline_option(const STD_string& opt) {cmdline_option=opt; return *this;}
  STD_string get_cmdline_option() const {return cmdline_option;}


 protected:

  JcampDxClass();
  JcampDxClass(const JcampDxClass& jdc);
  JcampDxClass& operator = (const JcampDxClass& jdc);

  STD_string get_parx_def_string(const STD_string type, unsigned int dim) const;

 private:
  friend class JDXwidget;
  friend class JcampDxBlock;

  virtual STD_string get_jdx_prefix() const;
  virtual STD_string get_jdx_postfix() const {return "\n";}

  compatMode compatmode;
  bool userDefParameter;
  parameterMode parmode;
  fileMode filemode;

  STD_string description;
  STD_string unit;

  int id;

  STD_string cmdline_option;

};



///////////////////////////////////////////////////////////////
//
//  Some interface classes which help editing parameters in a GUI

class JDXeditWidget {

 public:

  virtual void updateWidget() = 0;

  virtual ~JDXeditWidget() {} // to avoid compiler warnings
};



class JDXeditCaller {

 public:

  virtual void parameter_relations(JDXeditWidget* editwidget) = 0;

  virtual ~JDXeditCaller() {} // to avoid compiler warnings
};



/** @}
  */











#endif
