#     Copyright 2012, Kay Hayen, mailto:kayhayen@gmx.de
#
#     Part of "Nuitka", an optimizing Python compiler that is compatible and
#     integrates with CPython, but also works on its own.
#
#     If you submit patches or make the software available to licensors of
#     this software in either form, you automatically them grant them a
#     license for your part of the code under "Apache License 2.0" unless you
#     choose to remove this notice.
#
#     Kay Hayen uses the right to license his code under only GPL version 3,
#     to discourage a fork of Nuitka before it is "finished". He will later
#     make a new "Nuitka" release fully under "Apache License 2.0".
#
#     This program is free software: you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation, version 3 of the License.
#
#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#     Please leave the whole of this copyright notice intact.
#
""" Nodes for unary and binary operations.

No short-circuit involved, boolean NOT is an unary operation.

"""

from .NodeBases import CPythonChildrenHaving, CPythonNodeBase

from .NodeMakingHelpers import getComputationResult

from nuitka import PythonOperators

class CPythonExpressionOperationBase( CPythonChildrenHaving, CPythonNodeBase ):
    named_children = ( "operands", )

    def __init__( self, operator, simulator, operands, source_ref ):
        CPythonNodeBase.__init__( self, source_ref = source_ref )

        CPythonChildrenHaving.__init__(
            self,
            values = {
                "operands" : operands
            }
        )

        self.operator = operator

        self.simulator = simulator

    def getOperator( self ):
        return self.operator

    def getDetail( self ):
        return self.operator

    def getDetails( self ):
        return { "operator" : self.operator }

    def getSimulator( self ):
        return self.simulator

    getOperands = CPythonChildrenHaving.childGetter( "operands" )

class CPythonExpressionOperationBinary( CPythonExpressionOperationBase ):
    kind = "EXPRESSION_OPERATION_BINARY"

    def __init__( self, operator, left, right, source_ref ):
        assert left.isExpression() and right.isExpression, ( left, right )

        CPythonExpressionOperationBase.__init__(
            self,
            operator   = operator,
            simulator  = PythonOperators.binary_operator_functions[ operator ],
            operands   = ( left, right ),
            source_ref = source_ref
        )

    def computeNode( self ):
        operator = self.getOperator()
        operands = self.getOperands()

        a, b = operands

        if operator == "Div":
            if a.isConstant() and b.isConstant():
                return getComputationResult(
                    node        = self,
                    computation = lambda : self.getSimulator()( a.getConstant(), b.getConstant() ),
                    description = "Range builtin"
                )

            else:
                return self, None, None
        else:
            raise AssertionError( "Operator not implemented", operator )


class CPythonExpressionOperationUnary( CPythonExpressionOperationBase ):
    kind = "EXPRESSION_OPERATION_UNARY"

    def __init__( self, operator, operand, source_ref ):
        assert operand.isExpression(), operand

        CPythonExpressionOperationBase.__init__(
            self,
            operator   = operator,
            simulator  = PythonOperators.unary_operator_functions[ operator ],
            operands   = ( operand, ),
            source_ref = source_ref
        )

    def getOperand( self ):
        operands = self.getOperands()

        assert len( operands ) == 1
        return operands[ 0 ]


class CPythonExpressionOperationNOT( CPythonExpressionOperationUnary ):
    kind = "EXPRESSION_OPERATION_NOT"

    def __init__( self, operand, source_ref ):
        CPythonExpressionOperationUnary.__init__(
            self,
            operator   = "Not",
            operand    = operand,
            source_ref = source_ref
        )
