/****************************************************************************************/
/*											*/
/* This program is free software; you can redistribute it and/or modify it under the	*/
/* terms of the GNU General Public License as published by the Free Software		*/
/* Foundation; either version 2 of the License, or (at your option) any later version.	*/
/*											*/
/* This program is distributed in the hope that it will be useful, but WITHOUT ANY	*/
/* WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A	*/
/* PARTICULAR PURPOSE. See the GNU General Public License for more details.		*/
/*											*/
/* You should have received a copy of the GNU General Public License along with this	*/
/* program; (See "COPYING"). If not, If not, see <http://www.gnu.org/licenses/>.        */
/*											*/
/*--------------------------------------------------------------------------------------*/
/*											*/
/*  Copyright   Joerg Anders, TU Chemnitz, Fakultaet fuer Informatik, GERMANY           */
/*		ja@informatik.tu-chemnitz.de						*/
/*											*/
/*											*/
/****************************************************************************************/

//#define INPUT_DEBUG
#include <cairo.h>
#include <cairo-ft.h>
#include <iostream>
#include "localization.h"
#include "mainwindow.h"
#include "chordorrest.h"
#include "note.h"
#include "resource.h"
#include "midiexport.h"
#include "measure.h"
#include "tuplet.h"

#include "icons/nted.h"

#include "icons/stroken_grace_icon.h"
#include "icons/grace_eighth_icon.h"
#include "icons/grace_sixth_icon.h"
#include "icons/n32ndnote_icon.h"
#include "icons/n64thnote_icon.h"
#include "icons/eightnote_icon.h"
#include "icons/fullnote_icon.h"
#include "icons/halfnote_icon.h"
#include "icons/quarternote_icon.h"
#include "icons/sixteenthnote_icon.h"
#include "icons/insert_mode_icon.h"
#include "icons/arrow_up_icon.h"
#include "icons/arrow_down_icon.h"
#include "icons/treble_clef_icon.h"
#include "icons/alto_clef_icon.h"
#include "icons/bass_clef_icon.h"


#include "icons/rest_icon.h"

#include "icons/shift_mode_icon.h"

#include "icons/dot_icon.h"
#include "icons/ddot_icon.h"
#include "icons/cross_icon.h"
#include "icons/flat_icon.h"
#include "icons/natural_icon.h"

#include "icons/tied_icon.h"

#include "icons/delete_staff_icon.h"

#include "icons/color_icon.h"

#include "icons/lyrics_icon.h"

#include "icons/key_icon.h"

#include "icons/normal_note_icon.h"
#include "icons/drum1_icon.h"
#include "icons/drum2_icon.h"
#include "icons/drum3_icon.h"
#include "icons/drum4_icon.h"
#include "icons/drum5_icon.h"
#include "icons/drum6_icon.h"

#include "icons/dcross_icon.h"
#include "icons/dflat_icon.h"
#include "icons/stacc_icon.h"
#include "icons/staccatissimo_icon.h"
#include "icons/str_pizz_icon.h"
#include "icons/tenuto_icon.h"
#include "icons/sforzato_icon.h"
#include "icons/sforzando_icon.h"
#include "icons/bow_up_icon.h"
#include "icons/bow_down_icon.h"

#include "icons/brace_icon.h"
#include "icons/bracket_icon.h"

#include "icons/ped_on_icon.h"
#include "icons/ped_off_icon.h"
#include "icons/trill_icon.h"
#include "icons/prall_icon.h"
#include "icons/open_icon.h"
#include "icons/mordent_icon.h"
#include "icons/fermata_icon.h"
#include "icons/arpeggio_icon.h"

#include "icons/lines.h"
#include "icons/signs.h"

#define CONFIG_DIR ".nted"
#define CONFIG_FILE "ntedrc"

GdkPixbuf *NedResource::m_the_nted_icon;

GdkPixbuf *NedResource::m_32ndnote_icon;
GdkPixbuf *NedResource::m_64thnote_icon;
GdkPixbuf *NedResource::m_eightnote_icon;
GdkPixbuf *NedResource::m_fullnote_icon;
GdkPixbuf *NedResource::m_halfnote_icon;
GdkPixbuf *NedResource::m_quarternote_icon;
GdkPixbuf *NedResource::m_sixteenthnote_icon;
GdkPixbuf *NedResource::m_stroken_grace_icon;
GdkPixbuf *NedResource::m_grace_eighth_icon;
GdkPixbuf *NedResource::m_grace_sixth_icon;

GdkPixbuf *NedResource::m_rest_icon;

GdkPixbuf *NedResource::m_shift_mode_icon;
GdkPixbuf *NedResource::m_insert_mode_icon;

GdkPixbuf *NedResource::m_dot_icon;
GdkPixbuf *NedResource::m_ddot_icon;
GdkPixbuf *NedResource::m_cross_icon;
GdkPixbuf *NedResource::m_flat_icon;
GdkPixbuf *NedResource::m_natural_icon;

GdkPixbuf *NedResource::m_tied_icon;

GdkPixbuf *NedResource::m_arrow_up_icon;
GdkPixbuf *NedResource::m_arrow_down_icon;

GdkPixbuf *NedResource::m_treble_clef_icon;
GdkPixbuf *NedResource::m_bass_clef_icon;
GdkPixbuf *NedResource::m_alto_clef_icon;

GdkPixbuf *NedResource::m_delete_staff_icon;

GdkPixbuf *NedResource::m_color_icon;

GdkPixbuf *NedResource::m_lyrics_icon;

GdkPixbuf *NedResource::m_key_icon;

GdkPixbuf *NedResource::m_normal_note_icon;
GdkPixbuf *NedResource::m_drum1_icon;
GdkPixbuf *NedResource::m_drum2_icon;
GdkPixbuf *NedResource::m_drum3_icon;
GdkPixbuf *NedResource::m_drum4_icon;
GdkPixbuf *NedResource::m_drum5_icon;
GdkPixbuf *NedResource::m_drum6_icon;

GdkPixbuf *NedResource::m_dcross_icon;
GdkPixbuf *NedResource::m_dflat_icon;
GdkPixbuf *NedResource::m_stacc_icon;
GdkPixbuf *NedResource::m_staccatissimo_icon;
GdkPixbuf *NedResource::m_str_pizz_icon;
GdkPixbuf *NedResource::m_tenuto_icon;
GdkPixbuf *NedResource::m_sforzato_icon;
GdkPixbuf *NedResource::m_sforzando_icon;
GdkPixbuf *NedResource::m_bow_up_icon;
GdkPixbuf *NedResource::m_bow_down_icon;

GdkPixbuf *NedResource::m_brace_icon;
GdkPixbuf *NedResource::m_bracket_icon;

GdkPixbuf *NedResource::m_ped_on_icon;
GdkPixbuf *NedResource::m_ped_off_icon;
GdkPixbuf *NedResource::m_trill_icon;
GdkPixbuf *NedResource::m_prall_icon;
GdkPixbuf *NedResource::m_open_icon;
GdkPixbuf *NedResource::m_mordent_icon;
GdkPixbuf *NedResource::m_fermata_icon;
GdkPixbuf *NedResource::m_arpeggio_icon;

GdkPixbuf *NedResource::m_lines;
GdkPixbuf *NedResource::m_signs;

int NedResource::m_lines_buf_width;
int NedResource::m_lines_buf_height;

int NedResource::m_signs_buf_width;
int NedResource::m_signs_buf_height;

struct timeval NedResource::m_expected_time;

snd_seq_t *NedResource::m_sequ = NULL;
char NedResource::m_word_buffer[128];

GtkIconFactory *NedResource::m_icon_factory;

NedBbox NedResource::m_acc_bbox;

double NedResource::m_flag_width;

const char *NedResource::m_error_message;
int NedResource::m_input_line;

GList *NedResource::m_midi_events = NULL;
GList *NedResource::m_play_ptr;
int NedResource::m_handler_id;
NedMainWindow *NedResource::m_main_window;
unsigned long long NedResource::m_system_start_time;
bool NedResource::m_playing = FALSE;
GList *NedResource::midi_devices = NULL;
snd_seq_addr_t NedResource::m_alsaaddr;
snd_seq_addr_t NedResource::m_alsa_dest_addr;

GList *NedResource::m_addr_rel_list = NULL;
int NedResource::m_marker_counter = 1;
bool NedResource::m_channel_list[16];
bool NedResource::m_midi_echo;

int NedResource::m_tuplet_tab[14]  = {0, 0, 3, 2, 6, 4, 4, 4, 12, 8, 8, 8, 8, 8};

int NedResource::m_last_immediate_pitch = -1;
int NedResource::m_last_immediate_channel = -1;
bool NedResource::m_avoid_immadiate_play = FALSE;
char *NedResource::m_recent_files[MAX_RECENT_FILES];
const char *NedResource::m_print_cmd;
int NedResource::m_width;
int NedResource::m_height;
GList *NedResource::m_temp_file_names = NULL;

unsigned int NedResource::m_length_field_value = 0;

double NedResource::m_tempo_inverse;
unsigned long long NedResource::m_accel_midi_start;
unsigned long long NedResource::m_accel_midi_end;
double NedResource::m_accel_start_tempoinverse;
double NedResource::m_accel_end_tempoinverse;

const char *NedResource::m_last_folder = DATADIR "/nted/examples";
char *NedResource::m_last_xml_dir = NULL;
char *NedResource::m_last_midi_dir = NULL;

const char *NedResource::GM_Instruments[] = {
 N_("Piano 1"),          N_("Piano 2"), N_("Piano 3"),          N_("Honky-tonk"),
 N_("E.Piano 1"),        N_("E.Piano 2"), N_("Harpsichord"),      N_("Clavinet"),
 N_("Celesta"),          N_("Glockenspiel"), N_("Music Box"),        N_("Vibraphone"),
 N_("Marimba"),          N_("Xylophone"), N_("Tubular Bells"),    N_("Dulcimer"),
 N_("Organ 1"),          N_("Organ 2"), N_("Organ 3"),          N_("Church Organ"),
 N_("Reed Organ"),       N_("Accordion"), N_("Harmonica"),        N_("Bandoneon"),
 N_("Nylon Guitar"),     N_("Steel Guitar"), N_("Jazz Guitar"),      N_("Clean Guitar"),
 N_("Guitar Mutes"),     N_("Overdrive Guitar"), N_("Guitar Harmonics"), N_("Guitar Harmonics"),
 N_("Acoustic Bass"),    N_("Fingered Bass"), N_("Picked Bass"),      N_("Fretless Bass"),
 N_("Slap Bass 1"),      N_("Slap Bass 2"), N_("Synth Bass 1"),     N_("Synth Bass 2"), 
 N_("Violin"),           N_("Viola"), N_("Cello"),            N_("Contrabass"),
 N_("Tremolo Strings"),  N_("Pizzicato"), N_("Harp"),             N_("Timpani"),
 N_("Strings"),          N_("Slow Strings"), N_("Synth Strings 1"),  N_("Synth Strings 2"),
 N_("Choir Aahs"),       N_("Voice Oohs"), N_("Synth Vox"),        N_("Orchestra Hit"),
 N_("Trumpet"),          N_("Trombone"), N_("Tuba"),             N_("Mute Trumpet"), 
 N_("French Horns"),     N_("Brass"), N_("Synth Brass 1"),    N_("Synth Brass 2"),
 N_("Soprano Sax"),      N_("Alto Sax"), N_("Tenor Sax"),        N_("Baritone Sax"),
 N_("Oboe"),             N_("English Horn"), N_("Bassoon"),          N_("Clarinet"),
 N_("Piccolo"),          N_("Flute"), N_("Recorder"),         N_("Pan Flute"),
 N_("Bottle Chiff"),     N_("Shakuhachi"), N_("Whistle"),          N_("Ocarina"),
 N_("Square Wave"),      N_("Saw Wave"), N_("Synth Calliope"),   N_("Chiffer Lead"),
 N_("Charang"),          N_("Solo Vox"), N_("5th Saw Wave"),     N_("Bass & Lead"),
 N_("Fantasia"),         N_("Warm Pad"), N_("Poly Synth"),       N_("Space Voice"),
 N_("Bowed Glass"),      N_("Metal Pad"), N_("Halo Pad"),         N_("Sweep Pad"),
 N_("Ice Rain"),         N_("Soundtrack"), N_("Crystal"),          N_("Atmosphere"), 
 N_("Brightness"),       N_("Goblin"), N_("Echo Drops"),       N_("Star Theme"),
 N_("Sitar"),            N_("Banjo"), N_("Shamisen"),         N_("Koto"),
 N_("Kalimba"),          N_("Bagpipe"), N_("Fiddle"),           N_("Shenai"),
 N_("Tinker Bell"),      N_("Agogo"), N_("Steel Drum"),       N_("Wood Block"),
 N_("Taiko Drum"),       N_("Melodic Tom"), N_("Synth Drum"),       N_("Reverse Cymbal"),
 N_("Fret Noise"),       N_("Breath Noise"), N_("Seashore"),         N_("Bird"),
 N_("Telephone"),        N_("Helicopter"), N_("Applause"),         N_("Gun Shot")
}; 

#define MAX_ZOOM 76.0
#define MIN_ZOOM 9.0

double NedResource::m_braceOutline[16][6] = {
{-42.322, 58.999, -48.185, 97.0849, -49.834, 163.292},
{5.001, 78.333, 10.499, 145.924, 8.332, 204.748},
{-1.69434, 45.9834, -16.75, 68.25, -33, 110},
{25.002, 38.665, 37.3057, 62.0166, 39, 108},
{2.16699, 58.8242, -9.333, 128, -14.334, 206.333},
{1.649, 66.207, 7.512, 104.293, 49.834, 163.292},
{-18.002, -32.335, -29.5178, -50.8487, -34.002, -87.001},
{-6.15526, -49.6221, 1.66701, -77.167, 7.33401, -128},
{5.45799, -48.962, 12.9874, -76.8682, 10.333, -126.167},
{-1.95312, -36.2705, -9.621, -56.3516, -21.999, -90.501},
{-7.03421, -19.4062, -16.6231, -27.6777, -27.251, -45.374},
{10.6289, -18.3457, 20.1052, -27.0782, 27.253, -47.039},
{12.246, -34.1964, 20.0461, -54.2305, 21.999, -90.501},
{2.65425, -49.2989, -4.875, -77.2051, -10.333, -126.167},
{-5.66701, -50.833, -13.489, -78.3779, -7.334, -128},
{4.48433, -36.1524, 16, -54.666, 34.002, -87.001}
};


double NedResource::m_bracketEndOutline[5][6] = {
{142, 6, 184, -8, 336, 106},
{103.961, 77.9707, 132.98, 153.72, 218, 252},
{-68, -104, -98.0664, -170.846, -184, -270},
{-104, -120, -204, -158, -256, -178},
{-35.9062, -13.8105, -92, -22, -115, -27}
};

struct paper_info_struct NedResource::m_paper_info[] = {
{"A4", 595, 842},
{"A3", 842, 1190},
{"B4", 729, 1032},
{NULL, 0, 0}};


cairo_font_face_t *NedResource::m_font_face;
cairo_scaled_font_t *NedResource::m_scaled_fonts[ZOOMLEVELS];
char *NedResource::m_font_file_name;
cairo_text_extents_t NedResource::fontextentions[ZOOMLEVELS][MAXELEMENTS];
double NedResource::m_zoom_factors[ZOOMLEVELS];
cairo_font_options_t *NedResource::m_fontoptions;

#ifndef HAS_SET_SCALED_FONT
cairo_matrix_t NedResource::m_matrixes[ZOOMLEVELS];
#endif

struct fermata_info {
	unsigned long long midi_time;
	double tempo_inverse;
	bool a_tempo;
};

GList *NedResource::m_fermata_list = NULL;


void NedResource::initialize() {
	int i;
	int dummyw, dummyh;
	GtkStockItem stock_item_up;
	GtkStockItem stock_item_down;
	GtkStockItem stock_item_delete_staff;
	m_midi_echo = TRUE;
	m_height = HEIGHT;
	m_width = WIDTH;
	const char *fontdir;

	for (i = 0; i < MAX_RECENT_FILES; m_recent_files[i++] = NULL);

	
	fontdir = getenv("NTED_FONT_DIR");
	if (fontdir == NULL) {
		fontdir = DATADIR;
	}
	if ((m_font_file_name = (char *) malloc(strlen(fontdir) + strlen(FONTFILE) + 10)) == NULL) {
		Abort("malloc error");
	}
	sprintf(m_font_file_name, "%s/%s", fontdir, FONTFILE);
	init_fonts(m_font_file_name);
	m_icon_factory = gtk_icon_factory_new();
	loadPixmap(nted, &m_the_nted_icon, "the-nted-icon", &dummyw, &dummyh);

	loadPixmap(n32ndnote_icon, &m_32ndnote_icon, "32ndnote-icon", &dummyw, &dummyh);
	loadPixmap(n64thnote_icon, &m_64thnote_icon, "64thnote-icon", &dummyw, &dummyh);
	loadPixmap(eightnote_icon, &m_eightnote_icon, "eighthnote-icon", &dummyw, &dummyh);
	loadPixmap(fullnote_icon, &m_fullnote_icon, "fullnote_icon", &dummyw, &dummyh);
	loadPixmap(halfnote_icon, &m_halfnote_icon, "halfnote-icon",  &dummyw, &dummyh);
	loadPixmap(quarternote_icon, &m_quarternote_icon, "quarternote-icon", &dummyw, &dummyh);
	loadPixmap(sixteenthnote_icon, &m_sixteenthnote_icon, "sixteenthnote-icon", &dummyw, &dummyh);

	loadPixmap(stroken_grace_icon, &m_stroken_grace_icon, "stroken-grace-icon", &dummyw, &dummyh);
	loadPixmap(grace_eighth_icon, &m_grace_eighth_icon, "grace-eighth-icon", &dummyw, &dummyh);
	loadPixmap(grace_sixth_icon, &m_grace_sixth_icon, "grace-sixth-icon", &dummyw, &dummyh);

	loadPixmap(rest_icon, &m_rest_icon, "rest-icon", &dummyw, &dummyh);

	loadPixmap(shift_mode_icon, &m_shift_mode_icon, "shift-mode-icon", &dummyw, &dummyh);
	loadPixmap(insert_mode_icon, &m_insert_mode_icon, "insert-mode-icon", &dummyw, &dummyh);

	loadPixmap(dot_icon, &m_dot_icon, "dot-icon", &dummyw, &dummyh);
	loadPixmap(ddot_icon, &m_ddot_icon, "ddot-icon", &dummyw, &dummyh);
	loadPixmap(cross_icon, &m_cross_icon, "cross-icon", &dummyw, &dummyh);
	loadPixmap(flat_icon, &m_flat_icon, "flat-icon", &dummyw, &dummyh);
	loadPixmap(natural_icon, &m_natural_icon, "natural-icon", &dummyw, &dummyh);

	loadPixmap(tied_icon, &m_tied_icon, "tied-icon", &dummyw, &dummyh);

	loadPixmap(delete_staff_icon, &m_delete_staff_icon, "delete-staff", &dummyw, &dummyh);

	loadPixmap(color_icon, &m_color_icon, "color-notes", &dummyw, &dummyh);

	loadPixmap(lyrics_icon, &m_lyrics_icon, "lyrics-icon", &dummyw, &dummyh);

	loadPixmap(key_icon, &m_key_icon, "key-icon", &dummyw, &dummyh);

	loadPixmap(normal_note_icon, &m_normal_note_icon, "normal-note-icon", &dummyw, &dummyh);
	loadPixmap(drum1_icon, &m_drum1_icon, "drum1-icon", &dummyw, &dummyh);
	loadPixmap(drum2_icon, &m_drum2_icon, "drum2-icon", &dummyw, &dummyh);
	loadPixmap(drum3_icon, &m_drum3_icon, "drum3-icon", &dummyw, &dummyh);
	loadPixmap(drum4_icon, &m_drum4_icon, "drum4-icon", &dummyw, &dummyh);
	loadPixmap(drum5_icon, &m_drum5_icon, "drum5-icon", &dummyw, &dummyh);
	loadPixmap(drum6_icon, &m_drum6_icon, "drum6-icon", &dummyw, &dummyh);

	loadPixmap(dcross_icon, &m_dcross_icon, "dcross-icon", &dummyw, &dummyh);
	loadPixmap(dflat_icon, &m_dflat_icon, "dflat-icon", &dummyw, &dummyh);
	loadPixmap(stacc_icon, &m_stacc_icon, "stacc-icon", &dummyw, &dummyh);
	loadPixmap(staccatissimo_icon, &m_staccatissimo_icon, "staccatissimo-icon", &dummyw, &dummyh);
	loadPixmap(str_pizz_icon, &m_str_pizz_icon, "str-pizz-icon", &dummyw, &dummyh);
	loadPixmap(tenuto_icon, &m_tenuto_icon, "tenuto-icon", &dummyw, &dummyh);
	loadPixmap(sforzato_icon, &m_sforzato_icon, "sforzato-icon", &dummyw, &dummyh);
	loadPixmap(sforzando_icon, &m_sforzando_icon, "sforzando-icon", &dummyw, &dummyh);
	loadPixmap(bow_up_icon, &m_bow_up_icon, "bow-up-icon", &dummyw, &dummyh);
	loadPixmap(bow_down_icon, &m_bow_down_icon, "bow-down-icon", &dummyw, &dummyh);

	loadPixmap(brace_icon, &m_brace_icon, "brace-icon", &dummyw, &dummyh);
	loadPixmap(bracket_icon, &m_bracket_icon, "bracket-icon", &dummyw, &dummyh);

	loadPixmap(ped_on_icon, &m_ped_on_icon, "ped-on-icon", &dummyw, &dummyh);
	loadPixmap(ped_off_icon, &m_ped_on_icon, "ped-off-icon", &dummyw, &dummyh);
	loadPixmap(trill_icon, &m_trill_icon, "trill-icon", &dummyw, &dummyh);
	loadPixmap(prall_icon, &m_prall_icon, "prall-icon", &dummyw, &dummyh);
	loadPixmap(open_icon, &m_open_icon, "open-icon", &dummyw, &dummyh);
	loadPixmap(mordent_icon, &m_mordent_icon, "mordent-icon", &dummyw, &dummyh);
	loadPixmap(fermata_icon, &m_fermata_icon, "fermata-icon", &dummyw, &dummyh);
	loadPixmap(arpeggio_icon, &m_arpeggio_icon, "arpeggio-icon", &dummyw, &dummyh);

	loadPixmap(lines, &m_lines, "lines", &m_lines_buf_width, &m_lines_buf_height);
	loadPixmap(signs, &m_signs, "signs", &m_signs_buf_width, &m_signs_buf_height);

	loadPixmap(arrow_up_icon, &m_arrow_up_icon, "my-arrow-up-icon", &dummyw, &dummyh);
	loadPixmap(arrow_down_icon, &m_arrow_down_icon, "my-arrow-down-icon", &dummyw, &dummyh);

	loadPixmap(treble_clef_icon, &m_treble_clef_icon, "treble-clef-icon", &dummyw, &dummyh);
	loadPixmap(bass_clef_icon, &m_bass_clef_icon, "bass-clef-icon", &dummyw, &dummyh);
	loadPixmap(alto_clef_icon, &m_alto_clef_icon, "alto-clef-icon", &dummyw, &dummyh);

	stock_item_up.stock_id = (char *) "my-arrow-up-icon";
	stock_item_up.label = NULL;
	stock_item_up.modifier = (GdkModifierType) 0;
	stock_item_up.keyval = -1;
	stock_item_up.translation_domain = NULL;
	gtk_stock_add (&stock_item_up, 1);

	stock_item_down.stock_id = (char *) "my-arrow-down-icon";
	stock_item_down.label = NULL;
	stock_item_down.modifier = (GdkModifierType) 0;
	stock_item_down.keyval = -1;
	stock_item_down.translation_domain = NULL;
	gtk_stock_add (&stock_item_down, 1);

	stock_item_delete_staff.stock_id = (char *) "delete-staff";
	stock_item_delete_staff.label = (char *) "Delete Staff";
	stock_item_delete_staff.modifier = (GdkModifierType) 0;
	stock_item_delete_staff.keyval = -1;
	stock_item_delete_staff.translation_domain = NULL;
	gtk_stock_add (&stock_item_delete_staff, 1);

	gtk_icon_factory_add_default(m_icon_factory);
	open_seq();
	m_print_cmd = "lpr %s";
	read_config();
}

void NedResource::init_fonts(char *font_file_name) {
	int i;
	if (!loadfontface(font_file_name)) {
		char Str[128];
		sprintf(Str, "error in loadfontfile %s\n", font_file_name);
		Abort(Str);
	}
	for (i = 0; i < ZOOMLEVELS; i++) {
		m_zoom_factors[i] = MIN_ZOOM + (MAX_ZOOM - MIN_ZOOM) / 
				ZOOMLEVELS * i;
	}
	m_fontoptions= cairo_font_options_create();
	cairo_font_options_set_antialias(m_fontoptions, CAIRO_ANTIALIAS_SUBPIXEL);
	getScaledFont(START_ZOOM_LEVEL); // important, computes some "constants"
}

int NedResource::loadfontface(char *font_file_name) {
	FT_Library thelibrary;
	FT_Face theface;
	FT_Open_Args openargs;
	
	
	if (FT_Init_FreeType(&thelibrary) != 0) {
		DbgMsg(DBG_CRITICAL, "Error in FT_Init_FreeType\n");
		return 0;
	}
	
	openargs.flags = FT_OPEN_PATHNAME;
	openargs.pathname = font_file_name;
	if (FT_Open_Face( thelibrary, &openargs, 0, &theface) != 0) { 
		DbgMsg(DBG_CRITICAL, "Error in FT_Open_Face\n");
		return 0;
	}
	
	m_font_face = cairo_ft_font_face_create_for_ft_face(theface, 0);
	return 1;
}

#ifndef HAS_SET_SCALED_FONT
cairo_matrix_t *NedResource::getFontMatrix(int zoom_level) {
	if (zoom_level < 0 || zoom_level >= ZOOMLEVELS) {
		Abort("NedResource::getFontMatrix");
	}
	return &(m_matrixes[zoom_level]);
}
#endif

cairo_scaled_font_t *NedResource::createScaledFont(int zoom_level) {
	cairo_matrix_t the_font_matrix2;
#ifdef HAS_SET_SCALED_FONT
	cairo_matrix_t the_font_matrix;
#endif
	cairo_glyph_t glyph;
	cairo_scaled_font_t *scaled_font;
	int i;
	if (zoom_level < 0 || zoom_level >= ZOOMLEVELS) {
		Abort("NedResource::createScaledFont");
	}

	
#ifdef HAS_SET_SCALED_FONT
	cairo_matrix_init_scale (&the_font_matrix, FONT_FACTOR * m_zoom_factors[zoom_level],
		FONT_FACTOR * m_zoom_factors[zoom_level]);
#else
	cairo_matrix_init_scale (&(m_matrixes[zoom_level]), FONT_FACTOR * m_zoom_factors[zoom_level],
		FONT_FACTOR * m_zoom_factors[zoom_level]);
#endif
	cairo_matrix_init_identity (&the_font_matrix2);
	
	
	if (cairo_font_options_status(m_fontoptions) != CAIRO_STATUS_SUCCESS) {
		Abort("NedResource::getScaled: FontFehler in cairo_font_options_create");
	}
	
#ifdef HAS_SET_SCALED_FONT
	scaled_font = cairo_scaled_font_create(m_font_face, &the_font_matrix,&the_font_matrix2, m_fontoptions);
#else
	scaled_font = cairo_scaled_font_create(m_font_face, &(m_matrixes[zoom_level]),&the_font_matrix2, m_fontoptions);
#endif
	
	if (cairo_scaled_font_status(scaled_font) != CAIRO_STATUS_SUCCESS) {
		cairo_font_options_destroy(m_fontoptions);
		Abort("NedResource::getScaled: Fehler in cairo_scaled_font_create");
	}
	for (i = 0; i < MAXELEMENTS; i++) {
		glyph.index = BASE + i;
		glyph.y = 0;
		glyph.x = 0;
		cairo_scaled_font_glyph_extents (scaled_font, &glyph, 1, &(fontextentions[zoom_level][i]));
	}
	m_acc_bbox.x = fontextentions[zoom_level][0].x_bearing / m_zoom_factors[zoom_level];
	m_acc_bbox.y = fontextentions[zoom_level][0].y_bearing / m_zoom_factors[zoom_level];
	m_acc_bbox.width = fontextentions[zoom_level][0].width / m_zoom_factors[zoom_level];
	m_acc_bbox.height = fontextentions[zoom_level][0].height / m_zoom_factors[zoom_level];
	double xx = fontextentions[zoom_level][16].x_bearing / m_zoom_factors[zoom_level];
	double yy = fontextentions[zoom_level][16].y_bearing / m_zoom_factors[zoom_level];
	double w = fontextentions[zoom_level][16].width / m_zoom_factors[zoom_level];
	double h = fontextentions[zoom_level][16].height / m_zoom_factors[zoom_level];
	if (w > m_acc_bbox.width) m_acc_bbox.width = w;
	if (h > m_acc_bbox.height) m_acc_bbox.height = h;
	if (xx < m_acc_bbox.x) m_acc_bbox.x = xx;
	if (yy < m_acc_bbox.y) m_acc_bbox.y = yy;
	w = fontextentions[zoom_level][16].width / m_zoom_factors[zoom_level];
	h = fontextentions[zoom_level][16].height / m_zoom_factors[zoom_level];
	w = fontextentions[zoom_level][17].width / m_zoom_factors[zoom_level];
	h = fontextentions[zoom_level][17].height / m_zoom_factors[zoom_level];
	if (w > m_acc_bbox.width) m_acc_bbox.width = w;
	if (h > m_acc_bbox.height) m_acc_bbox.height = h;
	if (xx < m_acc_bbox.x) m_acc_bbox.x = xx;
	if (yy < m_acc_bbox.y) m_acc_bbox.y = yy;
	m_flag_width = (fontextentions[zoom_level][5].x_bearing + fontextentions[zoom_level][5].width) / m_zoom_factors[zoom_level];
	return scaled_font;
}

int NedResource::getNumInstruments() {
	return (sizeof(GM_Instruments) / sizeof(char *));
}

double NedResource::getZoomFactor(int zoom_level) {
	if (zoom_level < 0 || zoom_level >= ZOOMLEVELS) {
		Abort("NedResource::getZoomLevel");
	}
	return m_zoom_factors[zoom_level];
}

void NedResource::open_seq() {
	snd_seq_client_info_t *cinfo;

	if (snd_seq_open(&m_sequ, "default", SND_SEQ_OPEN_OUTPUT, 0) < 0) {
		Info(_("Cannot open ALSA MIDI sequencer --> replay won't work :-((\nTry (as root)\nmodprobe snd-seq-midi\nand restart NtEd\nPerhaps this helps(?)"));
		m_sequ = NULL;
		return;
	}
	snd_seq_set_client_name(m_sequ, "Nted");
	if ((m_alsaaddr.port = snd_seq_create_simple_port(m_sequ, "Nted output",
		SND_SEQ_PORT_CAP_READ|SND_SEQ_PORT_CAP_SUBS_READ,
			SND_SEQ_PORT_TYPE_APPLICATION)) != 0) {
		Abort("Error creating sequencer port");
	}
	snd_seq_client_info_alloca(&cinfo);
	snd_seq_get_client_info(m_sequ, cinfo);
	m_alsaaddr.client = snd_seq_client_info_get_client(cinfo);
	m_alsa_dest_addr.client = 0;
}

void NedResource::close_sequ() {
	if (m_sequ != NULL) {
		snd_seq_close(m_sequ);
	}
	m_sequ = NULL;
}

void NedResource::MidiProgramChange(int channel, int pgm) {
	snd_seq_event_t ev;

	snd_seq_ev_clear(&ev);
	ev.type = SND_SEQ_EVENT_PGMCHANGE;
	ev.data.control.channel = channel;
	ev.data.control.value = pgm;
	snd_seq_ev_set_subs(&ev);  
	snd_seq_ev_set_direct(&ev);
	snd_seq_event_output_direct(m_sequ, &ev);
}

void NedResource::MidiCtrl(unsigned int ctrl_param, int channel, int val) {
	snd_seq_event_t ev;

	snd_seq_ev_clear(&ev);
	
	ev.type = SND_SEQ_EVENT_CONTROLLER;
	ev.data.control.channel = channel;
	ev.data.control.param = ctrl_param;
	ev.data.control.value = val;
	snd_seq_ev_set_subs(&ev);  
	snd_seq_ev_set_direct(&ev);
	snd_seq_event_output_direct(m_sequ, &ev);
}

void NedResource::NoteOn(int channel, int pitch, int vol) {
	snd_seq_event_t ev;

	snd_seq_ev_clear(&ev);
	ev.type = SND_SEQ_EVENT_NOTEON;
	ev.data.note.channel = channel;
	ev.data.note.velocity = vol;
	ev.data.note.note = pitch;
	snd_seq_ev_set_subs(&ev);  
	snd_seq_ev_set_direct(&ev);
	snd_seq_event_output_direct(m_sequ, &ev);
}

void NedResource::NoteOff(int channel, int pitch) {
	snd_seq_event_t ev;
	snd_seq_ev_clear(&ev);

	ev.type = SND_SEQ_EVENT_NOTEOFF;
	ev.data.note.channel = channel;
	ev.data.note.velocity = 0;
	ev.data.note.note = pitch;
	snd_seq_ev_set_subs(&ev);  
	snd_seq_ev_set_direct(&ev);
	snd_seq_event_output_direct(m_sequ, &ev);
}

void NedResource::SeqReset() {
	snd_seq_event_t ev;

	if (m_sequ == NULL) return;
	snd_seq_ev_clear(&ev);

	for (int i = 0; i < 16; i++) {
		MidiCtrl(MIDI_CTL_SUSTAIN, i, 1);
	}

	ev.type = SND_SEQ_EVENT_RESET;
	ev.data.note.channel = 0;
	ev.data.note.velocity = 0;
	ev.data.note.note = 0;
	snd_seq_ev_set_subs(&ev);  
	snd_seq_ev_set_direct(&ev);
	snd_seq_event_output_direct(m_sequ, &ev);
}


void NedResource::cleanup() {
	GList *lptr;
	write_config();
	snd_seq_port_subscribe_t *subs;
	if (m_sequ != NULL && m_alsa_dest_addr.client != 0) {
		snd_seq_port_subscribe_alloca(&subs);
		snd_seq_port_subscribe_set_sender(subs, &m_alsaaddr);
		snd_seq_port_subscribe_set_dest(subs, &m_alsa_dest_addr);
		snd_seq_unsubscribe_port(m_sequ, subs);
		m_alsa_dest_addr.client = 0;
	}
	close_sequ();
	for (lptr = g_list_first(m_temp_file_names); lptr; lptr = g_list_next(lptr)) {
		unlink((char *) (lptr->data));
	}
}

void NedResource::write_config() {
	char fname[4096], *homedir;
	FILE *fp;
	int i;

	if ((homedir = getenv("HOME")) == NULL) {
		Warning("cannot write config: HOME not found");
		return;
	}
	strcpy(fname, homedir);
	strcat(fname, "/");
	strcat(fname, CONFIG_DIR);

	if (access(fname, R_OK | W_OK | X_OK) != 0) {
		if (access(fname, F_OK) == 0) {
			Warning("cannot write config:  cannot access dir(1)");
			return;
		}
		if (mkdir(fname, 0755) < 0) {
			Warning("cannot write config: mkdir");
			return;
		}
		if (access(fname, R_OK | W_OK | X_OK) != 0) {
			Warning("cannot write config:  cannot access dir(2)");
			return;
		}
	}
	strcat(fname, "/");
	strcat(fname, CONFIG_FILE);
	if ((fp = fopen(fname, "w")) == NULL) {
		Warning("cannot write config:  open");
		return;
	}
	if (m_alsa_dest_addr.client != 0) {
		fprintf(fp, "ALSACLIENT = %d\n", m_alsa_dest_addr.client);
		fprintf(fp, "ALSAPORT = %d\n", m_alsa_dest_addr.port);
	}
	if (m_last_folder != NULL) {
		fprintf(fp, "LASTFOLDER = %s\n", m_last_folder);
	}
	if (m_last_xml_dir  != NULL) {
		fprintf(fp, "LASTXMLDIR = %s\n", m_last_xml_dir );
	}
	if (m_last_midi_dir != NULL) {
		fprintf(fp, "LASTMIDIDIR = %s\n", m_last_midi_dir );
	}
	fprintf(fp, "PRINTCMD = %s\n", m_print_cmd);
	gtk_window_get_size (GTK_WINDOW(m_main_window->getWindow()), &m_width, &m_height);
	fprintf(fp, "WIDTH = %d\n", m_width?m_width:WIDTH);
	fprintf(fp, "HEIGHT = %d\n", m_height?m_height:HEIGHT);
	for (i = 0; i < MAX_RECENT_FILES; i++) {
		if (m_recent_files[i] != NULL) {
			fprintf(fp, "RECENTFILE %d = %s\n", i, m_recent_files[i]);
		}
	}
	fclose(fp);
}

void NedResource::addToRecentFiles(char *fname) {
	int i;

	for (i = 0; i < MAX_RECENT_FILES; i++) {
		if (m_recent_files[i] != NULL) {
			if (!strcmp(m_recent_files[i], fname)) {
				return;
			}
		}
	}
	if (m_recent_files[MAX_RECENT_FILES-1] != NULL) {
		free(m_recent_files[MAX_RECENT_FILES-1]);
	}

	for (i = MAX_RECENT_FILES-1; i > 0; i--) {
		m_recent_files[i] = m_recent_files[i-1];
	}
	m_recent_files[0] = strdup(fname);
}


void NedResource::read_config() {
	char fname[4096], *homedir;
	char buffer[4096];
	char Str[128];
	int client = 0, port;
	int idx;
	bool client_set = FALSE, port_set = FALSE;
	FILE *fp;
	snd_seq_port_subscribe_t *subs;

	if ((homedir = getenv("HOME")) == NULL) {
		Warning("cannot read config: HOME not found");
		return;
	}
	strcpy(fname, homedir);
	strcat(fname, "/");
	strcat(fname, CONFIG_DIR);
	strcat(fname, "/");
	strcat(fname, CONFIG_FILE);
	if ((fp = fopen(fname, "r")) == NULL) {
		DbgMsg(DBG_CRITICAL, "cannot read config. Ignore this if this is the first call!\n");
		return;
	}
	m_input_line = 1;
	while(readWord(fp, buffer)) {
		if (!strcmp(buffer, "ALSACLIENT")) {
			if (!readWord(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: '=' expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readInt(fp, &client)) {
				sprintf(Str, "cannot read config: line %d: client number expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			client_set = TRUE;
		}
		else if (!strcmp(buffer, "ALSAPORT")) {
			if (!readWord(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: '=' expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readInt(fp, &port)) {
				sprintf(Str, "cannot read config: line %d: port number expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			port_set = TRUE;
		}
		else if (!strcmp(buffer, "PRINTCMD")) {
			if (!readWord(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: '=' expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readTillEnd(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: print command expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			m_print_cmd = strdup(buffer);
		}
		else if (!strcmp(buffer, "WIDTH")) {
			if (!readWord(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: '=' expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readInt(fp, &m_width)) {
				sprintf(Str, "cannot read config: line %d: width size expected", m_input_line);
				Warning(Str);
				fclose(fp);
				m_width = WIDTH;
				return;
			}
		}
		else if (!strcmp(buffer, "HEIGHT")) {
			if (!readWord(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: '=' expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readInt(fp, &m_height)) {
				sprintf(Str, "cannot read config: line %d: height size expected", m_input_line);
				Warning(Str);
				fclose(fp);
                m_height = HEIGHT;
				return;
			}
		}
		else if (!strcmp(buffer, "LASTFOLDER")) {
			if (!readWord(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: '=' expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readTillEnd(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: last folder expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			m_last_folder = strdup(buffer);
		}
		else if (!strcmp(buffer, "LASTXMLDIR")) {
			if (!readWord(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: '=' expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readTillEnd(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: last xml file expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			m_last_xml_dir = strdup(buffer);
		}
		else if (!strcmp(buffer, "LASTMIDIDIR")) {
			if (!readWord(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: '=' expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readTillEnd(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: last midi file expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			m_last_midi_dir = strdup(buffer);
		}
		else if (!strcmp(buffer, "RECENTFILE")) {
			if (!readInt(fp, &idx) || idx < 0 || idx >= MAX_RECENT_FILES) {
				sprintf(Str, "cannot read config: line %d: index of recent file expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readWord(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: '=' expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			if (!readString(fp, buffer)) {
				sprintf(Str, "cannot read config: line %d: file name expected", m_input_line);
				Warning(Str);
				fclose(fp);
				return;
			}
			m_recent_files[idx] = strdup(buffer);
		}
	}
	fclose(fp);

	if (m_sequ != NULL && client_set && port_set) {
		m_alsa_dest_addr.client = client;
		m_alsa_dest_addr.port = port;
		snd_seq_port_subscribe_alloca(&subs);
		snd_seq_port_subscribe_set_sender(subs, &m_alsaaddr);
		snd_seq_port_subscribe_set_dest(subs, &m_alsa_dest_addr);
		snd_seq_subscribe_port(m_sequ, subs);
	}
}
	

cairo_scaled_font_t *NedResource::getScaledFont(int zoom_level) {
	if (zoom_level < 0 || zoom_level >= ZOOMLEVELS) {
		Abort("NedResource::getScaledFont");
	}
	if (m_scaled_fonts[zoom_level] == NULL) {
		m_scaled_fonts[zoom_level] = createScaledFont(zoom_level);
	}
	return m_scaled_fonts[zoom_level];
}

bool NedResource::equalContent(GList *l1, GList *l2, NedChordOrRest *except, int *diff_pos) {
	GList *lptr2;
	int lendiff;

	lendiff = g_list_length(l2) - g_list_length(l1);
	if (lendiff != 0) {
		if (except == NULL) {
			DbgMsg(DBG_TESTING, "FAIL A\n"); 
			return FALSE;
		}
		if (lendiff != 1) {
			DbgMsg(DBG_TESTING, "FAIL B\n"); 
			return FALSE;
		}
	}
	
	*diff_pos = -1;
	for (lptr2 = g_list_first(l2); lptr2; lptr2 = g_list_next(lptr2)) {
		if (g_list_find(l1, lptr2->data) == NULL) {
			if (except != NULL || *diff_pos >= 0) {
				DbgMsg(DBG_TESTING, "FAIL C\n"); 
				return FALSE;
			}
			if (lptr2->data != except) {
				DbgMsg(DBG_TESTING, "FAIL D\n"); 
				return FALSE;
			}
			*diff_pos =  g_list_position(l2, lptr2);
		}
	}
	DbgMsg(DBG_TESTING, "IDENT\n"); 
	return TRUE;
}
            
void NedResource::Abort(const char *s) {
	GtkWidget *error_dialog;
	error_dialog = gtk_message_dialog_new (NULL, 
		(GtkDialogFlags) 0,
		GTK_MESSAGE_ERROR,
		GTK_BUTTONS_OK,
		"Internal error detected: %s. Excuse the program must be terminated :-((( Please contact ja@informatik.tu-chemnitz.de", s);
	gtk_dialog_run (GTK_DIALOG (error_dialog));
	gtk_widget_destroy (error_dialog);	
	exit(10);
}

#define CURRENT_DEBUG_LEVEL DBG_CRITICAL
void NedResource::DbgMsg(int level, const char *format, ...) {
	va_list ap;
	if (level < CURRENT_DEBUG_LEVEL) return;
	va_start (ap, format);
	vprintf(format, ap);
	va_end(ap);
}

void NedResource::Warning(const char *s) {
	GtkWidget *error_dialog;
	error_dialog = gtk_message_dialog_new (NULL, 
		(GtkDialogFlags) 0,
		GTK_MESSAGE_WARNING,
		GTK_BUTTONS_OK,
		"Internal problem detected: %s. Please contact ja@informatik.tu-chemnitz.de", s);
	gtk_dialog_run (GTK_DIALOG (error_dialog));
	gtk_widget_destroy (error_dialog);	
	fprintf(stderr, "Warning: %s\n", s);
}

void NedResource::Info(const char *s) {
	GtkWidget *error_dialog;
	error_dialog = gtk_message_dialog_new (NULL, 
		(GtkDialogFlags) 0,
		GTK_MESSAGE_INFO,
		GTK_BUTTONS_OK,
		"%s", s);
	gtk_dialog_run (GTK_DIALOG (error_dialog));
	gtk_widget_destroy (error_dialog);	
}

void NedResource::loadPixmap(const guint8 *data, GdkPixbuf **pixbuf, const char *stock_id, int *width, int *height) {
	GtkIconSet* icon_set;
        GError *err = NULL;
        *pixbuf = gdk_pixbuf_new_from_inline(-1, data, false, &err);
        if (err != NULL) {
                DbgMsg(DBG_CRITICAL, "NedResource::loadPixmap: Fehler %d beim Laden einer Pixmap: %s\n", err->code, err->message);
                exit(1);
        }
        *width = gdk_pixbuf_get_width(*pixbuf);
        *height = gdk_pixbuf_get_height(*pixbuf);
	if (stock_id != NULL) {
		icon_set = gtk_icon_set_new_from_pixbuf(*pixbuf);
		gtk_icon_factory_add(m_icon_factory, stock_id, icon_set);
	}

}


void NedResource::unreadWord(char *word) {
	strcpy(m_word_buffer, word);
}


bool NedResource::readWordWithNum(FILE *fp, char *word) {
	int pos = 0;

	if (m_word_buffer[0] != '\0') {
		strcpy(word, m_word_buffer);
		m_word_buffer[0] = '\0';
		return true;
	}

	while (((word[0] = getc(fp)) != EOF) && (word[0] == ' ' || word[0] == '\t' || word[0] == '\n')) {
		if (word[0] == '\n') {
			NedResource::m_input_line++;
		}
	}
	if (feof(fp)) return FALSE;
	if (!((word[0] >= 'A' && word[0] <= 'Z') || (word[0] >= 'a' && word[0] <= 'z'))) {
		word[1] = '\0';
#ifdef INPUT_DEBUG
		printf("WORD: %s\n", word);
#endif
		return TRUE;
	}
	pos++;
	while ((word[pos] = getc(fp)) != EOF &&
		 ((word[pos] >= 'A' && word[pos] <= 'Z') ||
		 (word[pos] >= 'a' && word[pos] <= 'z') || word[pos] == '_') || (word[pos] >= '1' && word[pos] <= '9')) {
		pos++;
	}
	if (!feof(fp)) {
		ungetc(word[pos], fp);
	}
	word[pos] = '\0';
#ifdef INPUT_DEBUG
	printf("WORD: %s\n", word);
#endif
	return TRUE;
}

bool NedResource::readWordOfLength(FILE *fp, char *word) {
	int pos;
	char c;
	int len;
	bool not_eof;

	if (m_word_buffer[0] != '\0') {
		strcpy(word, m_word_buffer);
		m_word_buffer[0] = '\0';
		return true;
	}

	if (!readInt(fp, &len) || len < 0 || len > 255) {
		return false;
	}

	pos = -1;
	while (pos < len && (not_eof = ((c = getc(fp)) != EOF))) {
		if (c == '\n') {
			NedResource::m_input_line++;
		}
		if (pos >= 0) {
			word[pos] = c;
		}
		pos++;
	}
	word[pos] = '\0';
#ifdef INPUT_DEBUG
	printf("WORD: %s\n", word);
#endif
	return TRUE;
}

bool NedResource::readWord(FILE *fp, char *word) {
	int pos = 0;

	if (m_word_buffer[0] != '\0') {
		strcpy(word, m_word_buffer);
		m_word_buffer[0] = '\0';
		return true;
	}

	while (((word[0] = getc(fp)) != EOF) && (word[0] == ' ' || word[0] == '\t' || word[0] == '\n')) {
		if (word[0] == '\n') {
			NedResource::m_input_line++;
		}
	}
	if (feof(fp)) return FALSE;
	if (!((word[0] >= 'A' && word[0] <= 'Z') || (word[0] >= 'a' && word[0] <= 'z'))) {
		word[1] = '\0';
#ifdef INPUT_DEBUG
		printf("WORD: %s\n", word);
#endif
		return TRUE;
	}
	pos++;
	while ((word[pos] = getc(fp)) != EOF &&
		 ((word[pos] >= 'A' && word[pos] <= 'Z') ||
		 (word[pos] >= 'a' && word[pos] <= 'z') || word[pos] == '_')) {
		pos++;
	}
	if (!feof(fp)) {
		ungetc(word[pos], fp);
	}
	word[pos] = '\0';
#ifdef INPUT_DEBUG
	printf("WORD: %s\n", word);
#endif
	return TRUE;
}

bool NedResource::readTillEnd(FILE *fp, char *word) {
	int pos = 0;

	while (((word[0] = getc(fp)) != EOF) && (word[0] == ' ' || word[0] == '\t' || word[0] == '\n')) {
		if (word[0] == '\n') {
			NedResource::m_input_line++;
		}
	}
	if (feof(fp)) return FALSE;
	pos++;
	while ((word[pos] = getc(fp)) != EOF &&
		 word[pos] != '\n') {
		pos++;
	}
	word[pos] = '\0';
#ifdef INPUT_DEBUG
	printf("WORD: %s\n", word);
#endif
	return TRUE;
}

bool NedResource::readString(FILE *fp, char *word) {
	int pos = 0;

	while (((word[0] = getc(fp)) != EOF) && (word[0] == ' ' || word[0] == '\t' || word[0] == '\n')) {
		if (word[0] == '\n') {
			NedResource::m_input_line++;
		}
	}
	if (feof(fp)) return FALSE;
	pos++;
	while ((word[pos] = getc(fp)) != EOF &&
		 word[pos] != ' ' && word[pos] != '\t' && word[pos] != '\n') {
		pos++;
	}
	word[pos] = '\0';
#ifdef INPUT_DEBUG
	printf("WORD: %s\n", word);
#endif
	return TRUE;
}

bool NedResource::readInt(FILE *fp, int *val) {
	char buffer[128];
	int pos = 0;

	while (((buffer[0] = getc(fp)) != EOF) && (buffer[0] == ' ' || buffer[0] == '\t' || buffer[0] == '\n')) {
		if (buffer[0] == '\n') {
			NedResource::m_input_line++;
		}
	}
	if (feof(fp)) return FALSE;
	if (buffer[0] != '-' && !(buffer[0] >= '0' && buffer[0] <= '9')) {
		return FALSE;
	}
		
	pos++;
	while ((buffer[pos] = getc(fp)) != EOF &&
		 (buffer[pos] >= '0' && buffer[pos] <= '9')) {
		pos++;
	}
	if (!feof(fp)) {
		ungetc(buffer[pos], fp);
	}
	buffer[pos] = '\0';
	if (sscanf(buffer, "%d", val) != 1) {
		return FALSE;
	}
#ifdef INPUT_DEBUG
	printf("INT: %d\n", *val);
#endif
	return TRUE;
}

bool NedResource::readUnsignedInt(FILE *fp, unsigned int *val) {
	char buffer[128];
	int pos = 0;

	while (((buffer[0] = getc(fp)) != EOF) && (buffer[0] == ' ' || buffer[0] == '\t' || buffer[0] == '\n')) {
		if (buffer[0] == '\n') {
			NedResource::m_input_line++;
		}
	}
	if (feof(fp)) return FALSE;
	if (buffer[0] != '-' && !(buffer[0] >= '0' && buffer[0] <= '9')) {
		return FALSE;
	}
		
	pos++;
	while ((buffer[pos] = getc(fp)) != EOF &&
		 (buffer[pos] >= '0' && buffer[pos] <= '9')) {
		pos++;
	}
	if (!feof(fp)) {
		ungetc(buffer[pos], fp);
	}
	buffer[pos] = '\0';
	if (sscanf(buffer, "%u", val) != 1) {
		return FALSE;
	}
#ifdef INPUT_DEBUG
	printf("UNSIGNED INT: %u\n", *val);
#endif
	return TRUE;
}

bool NedResource::readLong(FILE *fp, unsigned long long *val) {
	char buffer[1024];
	int pos = 0;

	while (((buffer[0] = getc(fp)) != EOF) && (buffer[0] == ' ' || buffer[0] == '\t' || buffer[0] == '\n')) {
		if (buffer[0] == '\n') {
			NedResource::m_input_line++;
		}
	}
	if (feof(fp)) return FALSE;
	if (buffer[0] != '-' && !(buffer[0] >= '0' && buffer[0] <= '9')) {
		return FALSE;
	}
		
	pos++;
	while ((buffer[pos] = getc(fp)) != EOF &&
		 (buffer[pos] >= '0' && buffer[pos] <= '9')) {
		pos++;
	}
	if (!feof(fp)) {
		ungetc(buffer[pos], fp);
	}
	buffer[pos] = '\0';
	if (sscanf(buffer, "%llu", val) != 1) {
		return FALSE;
	}
#ifdef INPUT_DEBUG
	printf("LONG: %llu\n", *val);
#endif
	return TRUE;
}

bool NedResource::readHex(FILE *fp, int *val) {
	char buffer[128];
	int pos = 0;
	bool not_eof;

	while ((not_eof = (buffer[0] = getc(fp)) != EOF) && (buffer[0] == ' ' || buffer[0] == '\t' || buffer[0] == '\n')) {
		if (buffer[0] == '\n') {
			NedResource::m_input_line++;
		}
	}
	if (!not_eof) return FALSE;
	if (!((buffer[0] >= 'A' && buffer[0] <= 'F') || (buffer[0] >= 'a' && buffer[0] <= 'f') ||
		(buffer[0] >= '0' && buffer[0] <= '9'))) {
		DbgMsg(DBG_TESTING, "raus bei 1 buffer[0] = %c (0x%x)\n", buffer[0],buffer[0]); 
		return FALSE;
	}
		
	pos++;
	while ((buffer[pos] = getc(fp)) != EOF &&
		 ((buffer[pos] >= 'A' && buffer[pos] <= 'F') ||
		 (buffer[pos] >= '0' && buffer[pos] <= '9') ||
		 (buffer[pos] >= 'a' && buffer[pos] <= 'f'))) {
		pos++;
	}
	if (!feof(fp)) {
		ungetc(buffer[pos], fp);
	}
	buffer[pos] = '\0';
	if (sscanf(buffer, "%x", val) != 1) {
		return FALSE;
	}
#ifdef INPUT_DEBUG
	printf("HEX: %d(0x%x)\n", *val, *val);
#endif
	return TRUE;
}

bool NedResource::readFloat(FILE *fp, double *val) {
	int pos = 0;
	char word[128];
	bool dot_seen = false;

	if (m_word_buffer[0] != '\0') {
		strcpy(word, m_word_buffer);
		m_word_buffer[0] = '\0';
	}
	else {
		while (((word[0] = getc(fp)) != EOF) && (word[0] == ' ' || word[0] == '\t' || word[0] == '\n')) {
			if (word[0] == '\n') {
				NedResource::m_input_line++;
			}
		}
		if (feof(fp)) return FALSE;
		if (word[0] != '-' && word[0] != '+' && word[0] != '.' && word[0] < '0' && word[0] > '9') return false;
		if (word[0] == '.') dot_seen = true;
		pos++;
		while ((word[pos] = getc(fp)) != EOF && ((word[pos] >= '0' && word[pos] <= '9') || word[pos] == '.')) {
		 	if (word[pos] == '.') {
				if (dot_seen) return false;
				dot_seen = true;
			}
			pos++;
		}
		word[pos] = '\0';
	}
	if (sscanf(word, "%lf", val) != 1) return false;
	return true;
}

bool NedResource::removeWhiteSpaces(char **s) {
	gchar *cptr;
	char buffer[4096];
	int len;

	if (g_utf8_strlen(*s, -1) < 1) return false;
	cptr = g_utf8_offset_to_pointer(*s, 0);
	while (cptr) {
		if (!g_unichar_isspace(*cptr)) break;
		cptr = g_utf8_find_next_char(cptr, NULL);
	}
	if (cptr == NULL) return false;
	strcpy(buffer, cptr);
	len = g_utf8_strlen(buffer, -1);
	if (len < 1) return false;
	cptr = g_utf8_offset_to_pointer(buffer, len - 1);
	while(cptr != NULL) {
		if (!g_unichar_isspace(*cptr)) {
			cptr = g_utf8_find_next_char(cptr, NULL);
			*cptr = '\0';
			if (*s != NULL) {
				free(*s);
			}
			*s = strdup(buffer);
			return true;
		}
		cptr = g_utf8_find_prev_char(buffer, cptr);
	}
	return false;
}




void NedResource::increaseSystemStartTime(unsigned long long system_duration) {
	m_system_start_time += system_duration;
}

bool NedResource::permtest(snd_seq_port_info_t *pinfo, unsigned int perm) {
	return (((snd_seq_port_info_get_capability(pinfo) & perm) == perm) &&
		!(snd_seq_port_info_get_capability(pinfo) & SND_SEQ_PORT_CAP_NO_EXPORT));
}

GList *NedResource::listMidiDevices(int *index_of_subscribed_device) {
	GList *lptr;
	snd_seq_port_info_t *pinfo;
	snd_seq_client_info_t *cinfo;
	MidiPortStruct *port_struct;
	int i = 0;

	*index_of_subscribed_device = 0;

	if (midi_devices != NULL) {
		for (lptr = g_list_first(midi_devices); lptr; lptr = g_list_next(lptr)) {
			g_free(((MidiPortStruct *) (lptr->data))->name);
			g_free(lptr->data);
		}
		g_list_free(midi_devices);
	}
	midi_devices = NULL;
	snd_seq_port_info_alloca(&pinfo);
	snd_seq_client_info_alloca(&cinfo);
	snd_seq_client_info_set_client(cinfo, -1);
	while (snd_seq_query_next_client(m_sequ, cinfo) >= 0) {
		snd_seq_port_info_set_client(pinfo, snd_seq_client_info_get_client(cinfo));
		snd_seq_port_info_set_port(pinfo, -1);
		while(snd_seq_query_next_port(m_sequ, pinfo) >= 0) {
			if (permtest(pinfo, SND_SEQ_PORT_CAP_WRITE|SND_SEQ_PORT_CAP_SUBS_WRITE)) {
				if ((port_struct = (MidiPortStruct  *) g_malloc(sizeof(MidiPortStruct))) == NULL) {
					Abort("g_malloc_error");
				}
				port_struct->client = snd_seq_client_info_get_client(cinfo);
				port_struct->port = snd_seq_port_info_get_port(pinfo);
				port_struct->name = strdup(snd_seq_client_info_get_name(cinfo));
				if (m_alsa_dest_addr.client != 0) {
					if (m_alsa_dest_addr.client == port_struct->client && m_alsa_dest_addr.port == port_struct->port) {
						*index_of_subscribed_device = i;
					}
				}
				midi_devices = g_list_append(midi_devices, port_struct);
				i++;
			}
		}
	}
	return midi_devices;
}

void NedResource::subscribe(int idx) {
	GList *lptr;
	snd_seq_port_subscribe_t *subs;


	if ((lptr = g_list_nth(midi_devices, idx)) == NULL) {
		NedResource::Abort("NedResource::subscribe");
	}

	snd_seq_port_subscribe_alloca(&subs);
	if (m_alsa_dest_addr.client != 0) {
		snd_seq_port_subscribe_set_sender(subs, &m_alsaaddr);
		snd_seq_port_subscribe_set_dest(subs, &m_alsa_dest_addr);
		snd_seq_unsubscribe_port(m_sequ, subs);
		m_alsa_dest_addr.client = 0;
	}
	m_alsa_dest_addr.client = ((MidiPortStruct *) lptr->data)->client;
	m_alsa_dest_addr.port = ((MidiPortStruct *) lptr->data)->port;
	snd_seq_port_subscribe_set_sender(subs, &m_alsaaddr);
	snd_seq_port_subscribe_set_dest(subs, &m_alsa_dest_addr);
	snd_seq_subscribe_port(m_sequ, subs);
}
					

void NedResource::prepareMidiEventList(NedMainWindow *main_window, double tempo_inverse) {
	GList *lptr1, *lptr2;

	if (m_midi_events != NULL) {
		for (lptr1 = g_list_first(m_midi_events); lptr1; lptr1 = g_list_next(lptr1)) {
			for (lptr2 = (GList *) g_list_first((GList *) (lptr1->data)); lptr2; lptr2 = g_list_next(lptr2)) {
				g_free(lptr2->data);
			}
			g_list_free((GList *) g_list_first((GList *) (lptr1->data)));
		}
		g_list_free(m_midi_events);
	}
	for (lptr1 = g_list_first(m_fermata_list); lptr1; lptr1 = g_list_next(lptr1)) {
		free(lptr1->data);
	}
	g_list_free(m_fermata_list);
	m_fermata_list = NULL;
	m_midi_events = NULL;
	m_main_window = main_window;
	m_system_start_time = 0;
	m_tempo_inverse = tempo_inverse;
}

void NedResource::addMidiEvent(int channel, unsigned int offset, unsigned int type, int data1, int data2,
			NedNote *note) {
	MidiEventStruct *ev;
	double current_tempo_inverse;
	if ((ev = (MidiEventStruct *) g_malloc(sizeof(MidiEventStruct))) == NULL) {
		Abort("error in g_malloc\n");
	}
	ev->channel = channel;
	ev->midi_time = m_system_start_time + offset;
	if (m_accel_midi_start < ev->midi_time && m_accel_midi_end >= ev->midi_time) {
			current_tempo_inverse = m_accel_start_tempoinverse + 
			(m_accel_end_tempoinverse - m_accel_start_tempoinverse) / 
			(double) (m_accel_midi_end - m_accel_midi_start) *
			(double) (ev->midi_time - m_accel_midi_start);
			addTempoChange(offset, current_tempo_inverse);
	}
	ev->type = type;
	ev->data1 = data1;
	ev->data2 = data2;
	ev->note = note;
	ev->repetition = false;

	placeMidiEvent(ev);
}

void NedResource::addMidiCtrlEvent(int channel, unsigned int offset, unsigned int ctrl_param, int val) {
	MidiEventStruct *ev;
	if ((ev = (MidiEventStruct *) g_malloc(sizeof(MidiEventStruct))) == NULL) {
		Abort("error in g_malloc\n");
	}
	ev->channel = channel;
	ev->midi_time = m_system_start_time + offset;
	ev->type = SND_SEQ_EVENT_CONTROLLER;
	ev->data1 = ctrl_param;
	ev->data2 = val;

	placeMidiEvent(ev);
}

void NedResource::addMeasureEntry(NedMeasure *measure) {
	MidiEventStruct *ev;

	if ((ev = (MidiEventStruct *) g_malloc(sizeof(MidiEventStruct))) == NULL) {
		Abort("error in g_malloc\n");
	}
	ev->channel = 0; /* dummy */
	ev->midi_time = m_system_start_time + measure->midi_start;
	ev->type = PSEUDO_EVENT_MEASURE;
	ev->measure = measure;

	placeMidiEvent(ev);
}

void NedResource::addTempoChange(unsigned int offset, double tempoinverse) {
	MidiEventStruct *ev;

	if ((ev = (MidiEventStruct *) g_malloc(sizeof(MidiEventStruct))) == NULL) {
		Abort("error in g_malloc\n");
	}
	ev->channel = 0; /* dummy */
	ev->midi_time = m_system_start_time + offset;
	ev->type = PSEUDO_EVENT_TEMPO_CHANGE;
	ev->tempoinverse = tempoinverse;
	m_tempo_inverse = tempoinverse;

	placeMidiEvent(ev);
}

void NedResource::addTempoChangeAbs(unsigned long long midi_time, double tempoinverse) {
	MidiEventStruct *ev;

	if ((ev = (MidiEventStruct *) g_malloc(sizeof(MidiEventStruct))) == NULL) {
		Abort("error in g_malloc\n");
	}
	ev->channel = 0; /* dummy */
	ev->midi_time = midi_time;
	ev->type = PSEUDO_EVENT_TEMPO_CHANGE;
	ev->tempoinverse = tempoinverse;

	placeMidiEvent(ev);
}

void NedResource::setAccelRange(unsigned int offset, unsigned long long duration, double tempodiff) {
	double tempo = 60000.0 / m_tempo_inverse;
	m_accel_midi_start = m_system_start_time + offset;
	m_accel_midi_end = m_accel_midi_start + duration;
	m_accel_start_tempoinverse = m_tempo_inverse;
	tempo += tempodiff;
	m_accel_end_tempoinverse = 60000.0 / tempo;
}

void NedResource::addFermata(unsigned int offset) {
	struct fermata_info *fermata_info;

	if ((fermata_info = (struct fermata_info *) malloc(sizeof(struct fermata_info))) == NULL) {
		Abort("NedResource::addFermata");
	}
	fermata_info->midi_time = m_system_start_time + offset;
	fermata_info->tempo_inverse = m_tempo_inverse;
	fermata_info->a_tempo = true;
	m_fermata_list = g_list_append(m_fermata_list, fermata_info);
}

void NedResource::removeAllPseudosEvents() {
	GList *lptr1, *lptr2, *lptr3;


	lptr1 = g_list_first(m_midi_events);
	while (lptr1) {
		lptr2 = (GList *) (lptr1->data);
		lptr3 = g_list_first(lptr2);
		while (lptr3) {
			if ((((MidiEventStruct *) lptr3->data)->type & PSEUDO_FLAG) != 0) {
				lptr2 = g_list_delete_link(lptr2, lptr3);
				lptr3 = g_list_first(lptr2);
			}
			else {
				lptr3 = g_list_next(lptr3);
			}
		}
		if (lptr2 == NULL) {
			m_midi_events = g_list_delete_link(m_midi_events, lptr1);
			lptr1 = g_list_first(m_midi_events);
		}
		else {
			lptr1->data = lptr2;
			lptr1 = g_list_next(lptr1);
		}
	}
}

void NedResource::correctFermataTempo() {
	GList *lptr1, *lptr2, *lptr3;
	unsigned long long timenote, timeferm;
	double target_tempo_inverse, current_tempo_inverse;
	bool tempo_inserted;
#define FERMATA_MIDI_BEFORE_DIST NOTE_4
#define FERMATA_AFTER_MIDI_DIST NOTE_4
#define FERMATA_MIDI_TEMPO_RATIO 3.0

	if (m_fermata_list == NULL) return;

	for (lptr1 = g_list_first(m_midi_events);  lptr1; lptr1 = g_list_next(lptr1)) {
		lptr2 = (GList *) g_list_first((GList *) (lptr1->data));
		tempo_inserted = false;
		for (; !tempo_inserted && lptr2; lptr2 = g_list_next(lptr2)) {
			if (((MidiEventStruct *) lptr2->data)->type == SND_SEQ_EVENT_NOTEON) {
				timenote = ((MidiEventStruct *) lptr2->data)->midi_time;
				for (lptr3 = g_list_first(m_fermata_list); lptr3; lptr3 = g_list_next(lptr3)) {
					timeferm = ((struct fermata_info *) lptr3->data)->midi_time;
					if (timeferm - FERMATA_MIDI_BEFORE_DIST <= timenote && timeferm > timenote) {
						target_tempo_inverse = ((struct fermata_info *) lptr3->data)->tempo_inverse;
						current_tempo_inverse = target_tempo_inverse + (FERMATA_MIDI_TEMPO_RATIO * target_tempo_inverse - target_tempo_inverse) /
									(double) FERMATA_MIDI_BEFORE_DIST * (double) (timenote - (timeferm - FERMATA_MIDI_BEFORE_DIST));
						addTempoChangeAbs(timenote, current_tempo_inverse);
						tempo_inserted = true;
						((struct fermata_info *) lptr3->data)->a_tempo = false;
						break;
					}
					else if (timeferm <= timenote && timeferm + FERMATA_AFTER_MIDI_DIST > timenote) {
						target_tempo_inverse = ((struct fermata_info *) lptr3->data)->tempo_inverse;
						current_tempo_inverse = FERMATA_MIDI_TEMPO_RATIO * target_tempo_inverse + (target_tempo_inverse - FERMATA_MIDI_TEMPO_RATIO * target_tempo_inverse) /
									(double) FERMATA_AFTER_MIDI_DIST * (double) (timenote - timeferm);
						addTempoChangeAbs(timenote, current_tempo_inverse);
						tempo_inserted = true;
						((struct fermata_info *) lptr3->data)->a_tempo = false;
						break;
					}
					else if (!(((struct fermata_info *) lptr3->data)->a_tempo) && timenote >= timeferm + FERMATA_AFTER_MIDI_DIST) {
						addTempoChangeAbs(timenote, target_tempo_inverse);
						((struct fermata_info *) lptr3->data)->a_tempo = true;
					}
				}
			}
		}
	}
}


GList *NedResource::findListAtMeasure(int meas_number, NedMeasure **measure) {
	GList *lptr1, *lptr2;

	for (lptr1 = g_list_first(m_midi_events);  lptr1; lptr1 = g_list_next(lptr1)) {
		lptr2 = (GList *) g_list_first((GList *) (lptr1->data));
		for (; lptr2; lptr2 = g_list_next(lptr2)) {
			if (((MidiEventStruct *) lptr2->data)->type == PSEUDO_EVENT_MEASURE) {
				if (((MidiEventStruct *) lptr2->data)->measure->getMeasureNumber() == meas_number) {
					*measure = ((MidiEventStruct *) lptr2->data)->measure;
					return lptr1;
				}
			}
		}
	}
	return NULL;
}

void NedResource::copyAllBetweenMeasures(int start_measure_number, int alternative1_measure_number, int end_measure_number) {
	NedMeasure *start_measure, *end_measure;
	GList *start_ptr, *end_ptr, *alter1_ptr, *lptr;
	GList *repeat_part;
	GList *lptr2;
	unsigned long long extra_offs, extra_offs2;
	unsigned long long midi_start_time, midi_end_time, midi_alter1_time;
	if (end_measure_number <= start_measure_number) {
		Abort("NedResource::copyAllBetweenMeasures(1)");
	}
	if (alternative1_measure_number >= 0) {
		if (end_measure_number <= alternative1_measure_number) {
			Abort("NedResource::copyAllBetweenMeasures(2)");
		}
		if (alternative1_measure_number <= start_measure_number) {
			Abort("NedResource::copyAllBetweenMeasures(3)");
		}
		if ((alter1_ptr = findListAtMeasure(alternative1_measure_number, &start_measure)) == NULL) {
			Abort("NedResource::copyAllBetweenMeasures(4)");
		}
	}
	if ((start_ptr = findListAtMeasure(start_measure_number, &start_measure)) == NULL) {
		Abort("NedResource::copyAllBetweenMeasures(5)");
	}
	if ((end_ptr = findListAtMeasure(end_measure_number, &end_measure)) == NULL) {
		Abort("NedResource::copyAllBetweenMeasures(6)");
	}
	lptr2 = (GList *) g_list_first((GList *) (start_ptr->data));
	midi_start_time = ((MidiEventStruct *) lptr2->data)->midi_time;
	lptr2 = (GList *) g_list_first((GList *) (end_ptr->data));
	midi_end_time = ((MidiEventStruct *) lptr2->data)->midi_time;
	if (alternative1_measure_number >= 0) {
		lptr2 = (GList *) g_list_first((GList *) (alter1_ptr->data));
		midi_alter1_time = ((MidiEventStruct *) lptr2->data)->midi_time;
	}
#ifdef AAA
	if (midi_end_time <= midi_start_time) {
		if (end_measure->midi_start == 0) {
			if ((end_ptr2 = findListAtMeasure(end_measure_number - 1, &end_measure2)) == NULL) {
				Abort("NedResource::copyAllBetweenMeasures(7)");
			}
			lptr2 = (GList *) g_list_first((GList *) (end_ptr2->data));
			midi_end_time = ((MidiEventStruct *) lptr2->data)->midi_time + 
					end_measure2->midi_end - end_measure2->midi_start;
			extra_offs = midi_end_time - midi_start_time;
			/*
			DbgMsg(DBG_TESTING, "Sonderfall: end_measure->midi_end = %llu, start_measure->midi_start = %llu, extra_offs = %llu\n",
				end_measure2->midi_end / NOTE_4, start_measure->midi_start / NOTE_4,
				 extra_offs / NOTE_4); 
				 */
		}
		else {
			DbgMsg(DBG_CRITICAL, "start_measure_number = %d(%d) --> %llu, end_measure_number = %d(%d) --> %llu\n",
				start_measure_number, start_measure->getMeasureNumber(),
				start_measure->midi_start, end_measure_number,
				end_measure->getMeasureNumber(), end_measure->midi_start); 
			Abort("NedResource::copyAllBetweenMeasures(8)");
		}
	}
	else {
#endif
		if (alternative1_measure_number >= 0) {
			extra_offs2 = midi_end_time - midi_alter1_time;
		}
		extra_offs = midi_end_time - midi_start_time;
#ifdef AAA
	}
#endif
	//DbgMsg(DBG_TESTING, "end_measure->midi_start = %llu, start_measure->midi_start = %llu, extra_offs = %llu\n",
	//end_measure->midi_start / NOTE_4, start_measure->midi_start / NOTE_4, extra_offs / NOTE_4); 
	if (alternative1_measure_number >= 0) {
		addTimeOffsetAfter(end_ptr, extra_offs - extra_offs2);
	}
	else {
		addTimeOffsetAfter(end_ptr, extra_offs);
	}
	if (alternative1_measure_number >= 0) {
		repeat_part = cloneAllMidiEventsBetween(start_ptr, alter1_ptr, extra_offs);
	}
	else {
		repeat_part = cloneAllMidiEventsBetween(start_ptr, end_ptr, extra_offs);
	}
	for (lptr = g_list_first(repeat_part); lptr; lptr = g_list_next(lptr)) {
		placeMidiEvent((MidiEventStruct *) lptr->data);
	}
	g_list_free(repeat_part);
	/*
	DbgMsg(DBG_TESTING, "nach clone:\n");
	for (lptr1 = g_list_first(m_midi_events); lptr1; lptr1 = g_list_next(lptr1)) {
		DbgMsg(DBG_TESTING, "###\n");
		lptr2 = (GList *) g_list_first((GList *) (lptr1->data));
		for(; lptr2; lptr2 = g_list_next(lptr2)) {
			printMidiEvent((MidiEventStruct *) lptr2->data);
		}
	}
	*/
}

void NedResource::addTimeOffsetAfter(GList *end_ptr, unsigned long long extra_offs) {
	GList *lptr2, *lptr3;
	GList *pendingOns = NULL;

	if (g_list_position(m_midi_events, end_ptr) < 0) {
		Abort("addTimeOffsetAfter(1)");
	}
	lptr2 = (GList *) (end_ptr->data);
	lptr3 = g_list_first(lptr2);
	while (lptr3) {
		if (((MidiEventStruct *) lptr3->data)->type != SND_SEQ_EVENT_NOTEON && ((MidiEventStruct *) lptr3->data)->type != PSEUDO_EVENT_MEASURE) {
			lptr3 = g_list_next(lptr3);
		}
		else {
			((MidiEventStruct *) lptr3->data)->midi_time += extra_offs;
			pendingOns = g_list_append(pendingOns, lptr3->data);
			lptr2 = g_list_delete_link(lptr2, lptr3);
			lptr3 = g_list_first(lptr2);
		}
	}
	end_ptr->data = lptr2;
	/*
	lptr3 = end_ptr;
	end_ptr = g_list_next(end_ptr);
	if (lptr2 == NULL) {
		if (g_list_position(m_midi_events, lptr3) < 0) {
			Abort("addTimeOffsetAfter(3)");
		}
		m_midi_events = g_list_delete_link(m_midi_events, lptr3);
	}
	*/
	for (end_ptr = g_list_next(end_ptr); end_ptr; end_ptr = g_list_next(end_ptr)) {
		lptr2 = (GList *) g_list_first((GList *) (end_ptr->data));
		for (;lptr2; lptr2 = g_list_next(lptr2)) {
			((MidiEventStruct *) lptr2->data)->midi_time += extra_offs;
		}
	}
	for (lptr2 = g_list_first(pendingOns); lptr2; lptr2 = g_list_next(lptr2)) {
		placeMidiEvent((MidiEventStruct *) lptr2->data);
	}
	g_list_free(pendingOns);
}

GList * NedResource::cloneAllMidiEventsBetween(GList *start_ptr, GList *end_ptr, unsigned long long extra_offs) {
	GList *lptr2;
	int start_pos, end_pos;
	GList *repeat_part = NULL;
	MidiEventStruct *ev;


	if ((start_pos = g_list_position (m_midi_events, start_ptr)) < 0) {
		Abort("cloneAllMidiEventsBetween(1)");
	}

	if ((end_pos = g_list_position (m_midi_events, end_ptr)) < 0) {
		Abort("cloneAllMidiEventsBetween(2)");
	}

	if (end_pos <= start_pos) {
		DbgMsg(DBG_TESTING, "start_pos = %d, end_pos = %d\n", start_pos, end_pos); 
		Abort("cloneAllMidiEventsBetween(3)");
	}

	lptr2 = (GList *) g_list_first((GList *) (start_ptr->data));
	for (;lptr2; lptr2 = g_list_next(lptr2)) {
		if (((MidiEventStruct *) lptr2->data)->type == SND_SEQ_EVENT_NOTEOFF) continue;
		if (((MidiEventStruct *) lptr2->data)->type == PSEUDO_EVENT_MEASURE) continue;
		if ((ev = (MidiEventStruct *) g_malloc(sizeof(MidiEventStruct))) == NULL) {
			Abort("cloneAllMidiEventsBetween: error in g_malloc\n");
		}
		*ev = *(((MidiEventStruct *) lptr2->data));
		ev->midi_time += extra_offs;
		ev->repetition = true;
		repeat_part = g_list_append(repeat_part, ev);
	}
	for (start_ptr = g_list_next(start_ptr); start_ptr && (start_ptr != end_ptr); start_ptr = g_list_next(start_ptr)) {
		lptr2 = (GList *) g_list_first((GList *) (start_ptr->data));
		for (;lptr2; lptr2 = g_list_next(lptr2)) {
			if (((MidiEventStruct *) lptr2->data)->type == PSEUDO_EVENT_MEASURE) continue;
			if ((ev = (MidiEventStruct *) g_malloc(sizeof(MidiEventStruct))) == NULL) {
				Abort("cloneAllMidiEventsBetween: error in g_malloc\n");
			}
			*ev = *(((MidiEventStruct *) lptr2->data));
			ev->midi_time += extra_offs;
			ev->repetition = true;
			repeat_part = g_list_append(repeat_part, ev);
		}
	}
	if (start_ptr && (start_ptr == end_ptr)) {
		lptr2 = (GList *) g_list_first((GList *) (start_ptr->data));
		for (;lptr2; lptr2 = g_list_next(lptr2)) {
			if (((MidiEventStruct *) lptr2->data)->type == SND_SEQ_EVENT_NOTEON) continue;
			if (((MidiEventStruct *) lptr2->data)->type == PSEUDO_EVENT_MEASURE) continue;
			if ((ev = (MidiEventStruct *) g_malloc(sizeof(MidiEventStruct))) == NULL) {
				Abort("cloneAllMidiEventsBetween: error in g_malloc\n");
			}
			*ev = *(((MidiEventStruct *) lptr2->data));
			ev->midi_time += extra_offs;
			ev->repetition = true;
			repeat_part = g_list_append(repeat_part, ev);
		}
	}
	return repeat_part;
}

void NedResource::printMidiEvent(MidiEventStruct *ev) {
	bool ok = true;
	if (ev->type == PSEUDO_EVENT_MEASURE) return;
	DbgMsg(DBG_TESTING, "type = ");
	switch (ev->type) {
		case SND_SEQ_EVENT_NOTEON: DbgMsg(DBG_TESTING, "SND_SEQ_EVENT_NOTEON"); break;
		case SND_SEQ_EVENT_NOTEOFF: DbgMsg(DBG_TESTING, "SND_SEQ_EVENT_NOTEOFF"); break;
		case PSEUDO_EVENT_MEASURE: DbgMsg(DBG_TESTING, "PSEUDO_EVENT_MEASURE"); ok = false; break;
		default: DbgMsg(DBG_TESTING, "unknown %d(0x%x)", ev->type, ev->type);ok = false;  break;
	}
	DbgMsg(DBG_TESTING, ", time = %llu", ev->midi_time / NOTE_4);
	if (!ok) {
		if (ev->type == PSEUDO_EVENT_MEASURE) {
			DbgMsg(DBG_TESTING, ", measnum = %d\n", ev->measure->getMeasureNumber());
		}
		DbgMsg(DBG_TESTING, "\n"); 
		return;
	}
	DbgMsg(DBG_TESTING, ", chan = %d, pitch = %d, vol = %d\n", ev->channel, ev->data1, ev->data2); 
}

void NedResource::placeMidiEvent(MidiEventStruct *ev) {
	GList *event_list;
	GList *lptr1, *lptr2;

	if (m_midi_events == NULL) {
		event_list = g_list_append(NULL, ev);
		m_midi_events = g_list_append(m_midi_events, event_list);
		return;
	}

	for (lptr1 = g_list_first(m_midi_events); lptr1; lptr1 = g_list_next(lptr1)) {
		lptr2 = (GList *) g_list_first((GList *) (lptr1->data));
		if (lptr2 == NULL) continue;
		if (((MidiEventStruct *) lptr2->data)->midi_time == ev->midi_time) {
			lptr1->data = g_list_append((GList *) (lptr1->data), ev);
			return;
		}
		if (((MidiEventStruct *) lptr2->data)->midi_time > ev->midi_time) {
			event_list = g_list_append(NULL, ev);
			m_midi_events = g_list_insert_before(m_midi_events, lptr1, event_list);
			return;
		}
	}
	event_list = g_list_append(NULL, ev);
	m_midi_events = g_list_append(m_midi_events, event_list);
}

void NedResource::collectTempoEvents(GList **tempo_chords) {
	GList *lptr1, *lptr2;

	for (lptr1 = g_list_first(m_midi_events); lptr1; lptr1 = g_list_next(lptr1)) {
		lptr2 = (GList *) g_list_first((GList *) (lptr1->data));
		for (; lptr2; lptr2 = g_list_next(lptr2)) {
			if (((MidiEventStruct *) lptr2->data)->type != PSEUDO_EVENT_TEMPO_CHANGE) continue;
			*tempo_chords = g_list_append(*tempo_chords, lptr2->data);
		}
	}
}

void NedResource::startReplay(NedNote *startNote, double tempo_inverse) {
	GList *lptr1, *lptr2, *lptr22;
	unsigned long long time_diff;
	unsigned int sleeptime;
	NedNote *notes[1024];
	int num_notes = 0;
	bool found;
	bool first = true;
	struct timeval now;

	removeAllPseudosEvents();

	if (m_midi_events == NULL) {
		m_main_window->stopReplay();
		return;
	}

	m_accel_midi_start = m_accel_midi_end = 0;
	m_tempo_inverse = tempo_inverse;


	m_play_ptr = g_list_first(m_midi_events);
	if (startNote != NULL) {
		found = FALSE;
		for (m_play_ptr = g_list_first(m_midi_events); !found && m_play_ptr; m_play_ptr = g_list_next(m_play_ptr)) {
			lptr2 = (GList *) g_list_first((GList *) (m_play_ptr->data));
			for (; lptr2; lptr2 = g_list_next(lptr2)) {
				if (((MidiEventStruct *) lptr2->data)->type == PSEUDO_EVENT_TEMPO_CHANGE) {
					m_tempo_inverse = ((MidiEventStruct *) lptr2->data)->tempoinverse;
				}
				else if (((MidiEventStruct *) lptr2->data)->type == SND_SEQ_EVENT_NOTEON && ((MidiEventStruct *) lptr2->data)->note == startNote) {
					found = true;
					break;
				}
			}
			if (found) break;
		}
		if (!found) {
			m_play_ptr = g_list_first(m_midi_events);
		}
	}
	m_playing = TRUE;
	m_main_window->repaint(); // repaint without aux lines;

	do {
	   lptr2 = (GList *) g_list_first((GList *) (m_play_ptr->data));
	   lptr1 = g_list_next(m_play_ptr);
	   if (lptr1 != NULL) {
		lptr22 = (GList *) (lptr1->data);
		time_diff = ((MidiEventStruct *) lptr22->data)->midi_time - ((MidiEventStruct *) lptr2->data)->midi_time;
		sleeptime = (unsigned int) ((double) time_diff / (double) NOTE_4 * m_tempo_inverse);
	   }
	   for (; lptr2; lptr2 = g_list_next(lptr2)) {
		switch (((MidiEventStruct *) lptr2->data)->type) {
			case SND_SEQ_EVENT_NOTEON: if (((MidiEventStruct *) lptr2->data)->note->getTieBackward() == NULL) {
							NoteOn(((MidiEventStruct *) lptr2->data)->channel, ((MidiEventStruct *) lptr2->data)->data1,
								((MidiEventStruct *) lptr2->data)->data2);
						   }
							if (first) {
								first = false;
								m_main_window->setVisible(((MidiEventStruct *) lptr2->data)->note);
							}
							((MidiEventStruct *) lptr2->data)->note->m_active = TRUE;
							notes[num_notes++] = ((MidiEventStruct *) lptr2->data)->note;
							break;
			case SND_SEQ_EVENT_NOTEOFF:     if (((MidiEventStruct *) lptr2->data)->note->getTieForward() == NULL) {
								NoteOff(((MidiEventStruct *) lptr2->data)->channel, ((MidiEventStruct *) lptr2->data)->data1);
							}
							((MidiEventStruct *) lptr2->data)->note->m_active = FALSE;
							notes[num_notes++] = ((MidiEventStruct *) lptr2->data)->note;
							break;
			case SND_SEQ_EVENT_CONTROLLER:  MidiCtrl(((MidiEventStruct *) lptr2->data)->data1, ((MidiEventStruct *) lptr2->data)->channel,
								((MidiEventStruct *) lptr2->data)->data2);
							break;
		}
	   }
	   m_play_ptr = lptr1;
	}
	while (num_notes < 1 && m_play_ptr != NULL);
	if (m_play_ptr == NULL) {
		m_main_window->stopReplay();
		return;
	}
	//m_main_window->setVisible(notes[0]);
	m_main_window->repaintDuringReplay(notes, num_notes);

	if (m_play_ptr) {
		gettimeofday(&now, NULL);
		addtime(&now, sleeptime, &m_expected_time);
		m_handler_id = g_timeout_add (sleeptime, playNext, NULL);
		//m_playing = TRUE; is already done
	}
	else {
		m_playing = FALSE;
		m_main_window->stopReplay();
	}
}

unsigned int NedResource::subtime(struct timeval *future, struct timeval *now) {
	struct timeval t;
	if (timercmp(future, now, >)) {
		timersub(future, now, &t);
		return t.tv_usec;
	}
	return 0;
}

void NedResource::addtime(struct timeval *now, unsigned int msecs, struct timeval *res) {
	struct timeval t;

	t.tv_sec = 0; 
	t.tv_usec = msecs * 1000;
	timeradd(now, &t, res);
}

gboolean NedResource::playNext(void *data) {
	GList *lptr1, *lptr2, *lptr22;
	unsigned int sleeptime;
	unsigned long long time_diff;
	bool first = true;
	NedNote *notes[1024];
	int num_notes = 0;
	struct timeval now;
	g_source_remove (m_handler_id);
	if (m_play_ptr == NULL) return FALSE;
	if (!m_playing) {
		if (m_midi_events != NULL) {
			for (lptr1 = g_list_first(m_midi_events); lptr1; lptr1 = g_list_next(lptr1)) {
				for (lptr2 = (GList *) g_list_first((GList *) (lptr1->data)); lptr2; lptr2 = g_list_next(lptr2)) {
					if (((MidiEventStruct *) lptr2->data)->type == SND_SEQ_EVENT_NOTEON) {
						NoteOff(((MidiEventStruct *) lptr2->data)->channel, ((MidiEventStruct *) lptr2->data)->data1);
						((MidiEventStruct *) lptr2->data)->note->m_active = FALSE;
					}
					
				}
			}
		}
		SeqReset();
		m_main_window->repaint();
		return FALSE;
	}


	lptr2 = (GList *) g_list_first((GList *) (m_play_ptr->data));
	lptr1 = g_list_next(m_play_ptr);
	if (lptr1 != NULL) {
		lptr22 = (GList *) (lptr1->data);
		time_diff = ((MidiEventStruct *) lptr22->data)->midi_time - ((MidiEventStruct *) lptr2->data)->midi_time;
	}
	gettimeofday(&now, NULL);
	usleep(subtime(&m_expected_time, &now));
	for (; lptr2; lptr2 = g_list_next(lptr2)) {
		switch (((MidiEventStruct *) lptr2->data)->type) {
			case SND_SEQ_EVENT_NOTEOFF: if (((MidiEventStruct *) lptr2->data)->note->getTieForward() == NULL) {
							NoteOff(((MidiEventStruct *) lptr2->data)->channel, ((MidiEventStruct *) lptr2->data)->data1);
						    }
							((MidiEventStruct *) lptr2->data)->note->m_active = FALSE;
							notes[num_notes++] = ((MidiEventStruct *) lptr2->data)->note;
							break;
		}
	}
	lptr2 = (GList *) g_list_first((GList *) (m_play_ptr->data));
	for (; lptr2; lptr2 = g_list_next(lptr2)) {
		switch (((MidiEventStruct *) lptr2->data)->type) {
			case SND_SEQ_EVENT_NOTEON: if (((MidiEventStruct *) lptr2->data)->note->getTieBackward() == NULL) {
								NoteOn(((MidiEventStruct *) lptr2->data)->channel, ((MidiEventStruct *) lptr2->data)->data1,
									((MidiEventStruct *) lptr2->data)->data2); 
						   }
						   if (first) {
						   	first = false;
							m_main_window->setVisible(((MidiEventStruct *) lptr2->data)->note);
						   }

							((MidiEventStruct *) lptr2->data)->note->m_active = TRUE;
							notes[num_notes++] = ((MidiEventStruct *) lptr2->data)->note;
							break;
			case SND_SEQ_EVENT_CONTROLLER:  MidiCtrl(((MidiEventStruct *) lptr2->data)->data1, ((MidiEventStruct *) lptr2->data)->channel,
								((MidiEventStruct *) lptr2->data)->data2);
							break;
			case PSEUDO_EVENT_TEMPO_CHANGE: m_tempo_inverse = ((MidiEventStruct *) lptr2->data)->tempoinverse;
							break;
		}
	}
	sleeptime = (unsigned int) ((double) time_diff / (double) NOTE_4 * m_tempo_inverse);
	gettimeofday(&now, NULL);
	addtime(&now, sleeptime, &m_expected_time);
	/*
	if (num_notes > 0) {
		m_main_window->setVisible(notes[0]);
	}
	*/

	m_main_window->repaintDuringReplay(notes, num_notes);
	m_play_ptr = lptr1;

	if (m_play_ptr) {
		m_handler_id = g_timeout_add (sleeptime > 10 ? sleeptime - 10 : sleeptime, playNext, NULL);
	}
	else {
		m_playing = FALSE;
		m_main_window->stopReplay();
	}
	return FALSE;
}

void NedResource::stopReplay() {
	m_playing = FALSE;
	SeqReset();
}

void NedResource::playImmediately(int channel, int pgm, int pitch, int vol) {
	snd_seq_event_t ev;

	if (m_sequ == NULL) return;
	if (m_avoid_immadiate_play) return;
	if (m_last_immediate_pitch >= 0) return;
	m_last_immediate_pitch = pitch;
	m_last_immediate_channel = channel;

	snd_seq_ev_clear(&ev);
	ev.type = SND_SEQ_EVENT_PGMCHANGE;
	ev.data.control.channel = channel;
	ev.data.control.value = pgm;
	snd_seq_ev_set_subs(&ev);  
	snd_seq_ev_set_direct(&ev);
	snd_seq_event_output_direct(m_sequ, &ev);
	snd_seq_ev_clear(&ev);
	ev.type = SND_SEQ_EVENT_NOTEON;
	ev.data.note.channel = channel;
	ev.data.note.velocity = vol;
	ev.data.note.note = pitch;
	snd_seq_ev_set_subs(&ev);  
	snd_seq_ev_set_direct(&ev);
	snd_seq_event_output_direct(m_sequ, &ev);
	g_timeout_add (200, stopImmediate, NULL);
}

gboolean NedResource::stopImmediate(void *data) {
	snd_seq_event_t ev;

	if (m_last_immediate_pitch < 0) return FALSE;
	snd_seq_ev_clear(&ev);
	ev.type = SND_SEQ_EVENT_NOTEOFF;
	ev.data.note.channel = m_last_immediate_channel;
	ev.data.note.velocity = 0;
	ev.data.note.note = m_last_immediate_pitch;
	snd_seq_ev_set_subs(&ev);  
	snd_seq_ev_set_direct(&ev);
	snd_seq_event_output_direct(m_sequ, &ev);
	m_last_immediate_pitch = -1;
	return FALSE;
}



void NedResource::startMidiExport(FILE *fp, NedMainWindow *main_window) {
	NedMidiExport::exportMidi(fp, main_window, NULL, m_midi_events);
}

void NedResource::appendTempFileName(char *fname) {
	m_temp_file_names = g_list_append(m_temp_file_names, strdup(fname));
}


struct AddrStruct {
	int marker;
	void *addr;
};

void NedResource::prepareAddrStruct() {
	GList *lptr;

	if (m_addr_rel_list != NULL) {
		for (lptr = g_list_first(m_addr_rel_list); lptr; lptr = g_list_next(lptr)) {
			g_free(lptr->data);
		}
		g_list_free(m_addr_rel_list);
		m_addr_rel_list = NULL;
	}
	m_marker_counter = 1;
}

void NedResource::addAddr(int marker, void *addr) {
	AddrStruct *adrstruct;

	if ((adrstruct = (AddrStruct *) g_malloc(sizeof(AddrStruct))) == NULL) {
		Abort("error in g_malloc");
	}
	adrstruct->marker = marker;
	adrstruct->addr = addr;
	m_addr_rel_list = g_list_append(m_addr_rel_list, adrstruct);
}

void *NedResource::getAdressOfMarker(int marker) {
	GList *lptr;

	for (lptr = g_list_first(m_addr_rel_list); lptr; lptr = g_list_next(lptr)) {
		if (((AddrStruct *) lptr->data)->marker == marker) {
			return ((AddrStruct *) lptr->data)->addr;
		}
	}
	DbgMsg(DBG_CRITICAL, "marker = %d\n", marker);
	Abort("NedResource::getAdressOfMarker");
	return NULL;
}

int NedResource::getMarkerOfAddress(void *addr) {
	GList *lptr;

	for (lptr = g_list_first(m_addr_rel_list); lptr; lptr = g_list_next(lptr)) {
		if (((AddrStruct *) lptr->data)->addr == addr) {
			return ((AddrStruct *) lptr->data)->marker;
		}
	}
	Abort("NedResource::getMarkerOfAddress");
	return 0;
}

int NedResource::addAddr(void *addr) {
	AddrStruct *adrstruct;

	if ((adrstruct = (AddrStruct *) g_malloc(sizeof(AddrStruct))) == NULL) {
		Abort("error in g_malloc");
	}
	adrstruct->marker = m_marker_counter++;
	adrstruct->addr = addr;
	m_addr_rel_list = g_list_append(m_addr_rel_list, adrstruct);
	return adrstruct->marker;
}
		
struct paper_info_struct *NedResource::getPaperInfo(const char *papername) {
	paper_info_struct *ptr;

	for (ptr = m_paper_info; ptr->name != NULL; ptr++) {
		if (!strcmp(ptr->name, papername)) return ptr;
	}
	return NULL;
}

GList *NedResource::clone_chords_and_rests(GList *group, bool remove_completely_hidden_voices) {
	GList *copylist = NULL;
	GList *lptr;
	NedVoice *voi;
	NedChordOrRest *chord_or_rest;
	struct addr_ref_str *addrlist = NULL, *addr_ptr;
	struct voice_list_struct {
		NedVoice *voice;
		bool hidden;
		struct voice_list_struct *next;
	} *voice_list = NULL, *vl_ptr;

	if (remove_completely_hidden_voices) {
		for (lptr = g_list_first(group); lptr; lptr = g_list_next(lptr)) {
			voi = ((NedChordOrRest *) lptr->data)->getVoice();
			for (vl_ptr = voice_list; vl_ptr != NULL; vl_ptr = vl_ptr->next) {
				if (vl_ptr->voice == voi) {
					break;
				}
			}
			if (vl_ptr == NULL) {
				vl_ptr = (struct voice_list_struct *) alloca(sizeof(struct voice_list_struct));
				vl_ptr->voice = voi;
				vl_ptr->hidden = true;
				vl_ptr->next = voice_list;
				voice_list = vl_ptr;
			}
			if (!((NedChordOrRest *) lptr->data)->isHidden()) {
				vl_ptr->hidden = false;
			}
		}
	}
					
	for (lptr = g_list_first(group); lptr; lptr = g_list_next(lptr)) {
		if (remove_completely_hidden_voices) {
			voi = ((NedChordOrRest *) lptr->data)->getVoice();
			for (vl_ptr = voice_list; vl_ptr != NULL; vl_ptr = vl_ptr->next) {
				if (vl_ptr->voice == voi) {
					break;
				}
			}
			if (vl_ptr == NULL) {
				Abort("NedResource::clone_chords_and_rests");
			}
			if (vl_ptr->hidden) continue;
		}
		chord_or_rest = ((NedChordOrRest *) lptr->data)->clone(&addrlist, false);
		copylist = g_list_append(copylist, chord_or_rest);
	}
	for (lptr = g_list_first(copylist); lptr; lptr = g_list_next(lptr)) {
		((NedChordOrRest *) lptr->data)->adjust_pointers(addrlist);
	}
	while (addrlist != NULL) {
		addr_ptr = addrlist->next;
		free(addrlist);
		addrlist = addr_ptr;
	}
	return copylist;
}

void NedResource::split_element(GList **elements, GList *sp_element, unsigned long long delta1, unsigned long long delta2) {
	NedChordOrRest *chord_or_rest1, *chord_or_rest2, *chord_or_rest_before;
	int pos, dotcount;
	unsigned int len2;

	if ((pos = g_list_position(*elements, sp_element)) < 0) {
		Abort("NedResource::split_element");
	}
	chord_or_rest1 = (NedChordOrRest *) sp_element->data;
	chord_or_rest1->changeDuration(delta1, 0);
	delta1 -= chord_or_rest1->getDuration();
	chord_or_rest_before = chord_or_rest1;
	setLengthField(delta1);
	while ((len2 = getPartLength(&dotcount)) > 0) {
		pos++;
		chord_or_rest2 = chord_or_rest1->cloneWithDifferentLength(len2, dotcount);
		chord_or_rest_before->tieCompleteTo(chord_or_rest2);
		chord_or_rest_before = chord_or_rest2;
		*elements = g_list_insert(*elements, chord_or_rest2, pos);
		delta1 -= len2;
	}
	setLengthField(delta2);
	while ((len2 = getPartLength(&dotcount)) > 0) {
		pos++;
		chord_or_rest2 = chord_or_rest1->cloneWithDifferentLength(len2, dotcount);
		chord_or_rest_before->tieCompleteTo(chord_or_rest2);
		chord_or_rest_before = chord_or_rest2;
		*elements = g_list_insert(*elements, chord_or_rest2, pos);
		delta2 -= len2;
	}
	chord_or_rest_before->correctTiesForward();
}

unsigned int NedResource::getPartLength(int *dotcount) {
	unsigned int ret = 0, m;
	*dotcount = 0;

		
	if (m_length_field_value >= NOTE_2) {
		ret = NOTE_2;
	}
	else {
		for (m = NOTE_2; m >= NOTE_64; m >>= 1) {
			if (m_length_field_value >= m * 3 / 2) {
				*dotcount = 1;
				ret = m;
				break;
			}
			if (m_length_field_value >= m) {
				ret = m;
				break;
			}
		}
	}
	if (*dotcount == 1) {
		m_length_field_value -= ret * 3 / 2;
	}
	else {
		m_length_field_value -= ret;
	}
	return ret;
}
unsigned int NedResource::getPartLength(int *dotcount, bool is_rest, bool build_whole_rests, unsigned int measure_duration) {
	unsigned int ret = 0, m;
	*dotcount = 0;

		
	if (m_length_field_value >= NOTE_2) {
		if (is_rest && build_whole_rests && m_length_field_value >= WHOLE_NOTE) {
			m_length_field_value -= measure_duration;
			return WHOLE_NOTE;
		}
		else if (!is_rest && m_length_field_value >= WHOLE_NOTE) {
			ret = WHOLE_NOTE;
		}
		else {
			ret = NOTE_2;
		}
	}
	else {
		for (m = NOTE_2; m >= NOTE_64; m >>= 1) {
			if (m_length_field_value >= m * 3 / 2) {
				*dotcount = 1;
				ret = m;
				break;
			}
			if (m_length_field_value >= m) {
				ret = m;
				break;
			}
		}
	}
	if (*dotcount == 1) {
		m_length_field_value -= ret * 3 / 2;
	}
	else {
		m_length_field_value -= ret;
	}
	return ret;
}

bool NedResource::fittingDuration(unsigned int duration) {
	unsigned int m, d;

	for (m = WHOLE_NOTE; m >= NOTE_64; m >>= 1) {
		d = duration / m;
		if (m * d == duration) return true;
	}
	return false;
}

bool NedResource::fittingPosition(unsigned long long position) {
	unsigned long long m;

	m = position / NOTE_64;
	return m * NOTE_64 == position;
}


bool NedResource::test_for_incomplete_tuplets(GList *elements) {
	GList *lptr1, *lptr2;
	NedTuplet *tuplet;

	for (lptr1 = g_list_first(elements); lptr1; lptr1 = g_list_next(lptr1)) {
		if (((NedChordOrRest *) lptr1->data)->getTupletVal() != 0) {
			if ((tuplet = ((NedChordOrRest *) lptr1->data)->getTupletPtr()) == NULL) {
				Abort("NedResource::test_for_incomplete_tuplets");
			}
			for (lptr2 = tuplet->getElementList(); lptr2; lptr2 = g_list_next(lptr2)) {
				if (g_list_find(elements, lptr2->data) == NULL) {
					return TRUE;
				}
			}
		}
	}
	return FALSE;
}

bool NedResource::selection_has_uncomplete_tuplets(GList *elements) {
	NedTuplet *tuplet;
	GList *lptr1, *lptr2;

	for (lptr1 = g_list_first(elements); lptr1; lptr1 = g_list_next(lptr1)) {
		if ((tuplet = ((NedChordOrRest *) lptr1->data)->getTupletPtr()) == NULL) continue;
		for (lptr2 = tuplet->getElementList(); lptr2; lptr2 = g_list_next(lptr2)) {
			if (g_list_find(elements, lptr2->data) == NULL) return TRUE;
		}
	}
	return FALSE;
}

int NedResource::determineLastLine(int treble_line, int clef) {
	switch (clef) {
		case BASS_CLEF: treble_line += 12; break;
		case NEUTRAL_CLEF1:
		case NEUTRAL_CLEF2: 
		case ALTO_CLEF: treble_line += 6; break;
		case TENOR_CLEF: treble_line += 8; break;
		case SOPRAN_CLEF: treble_line += 2; break;
	}
	return treble_line;
}

const char *NedResource::getLilyPondClefName(int clef_number) {
	switch (clef_number) {
		case TREBLE_CLEF: return "treble";
		case BASS_CLEF: return "bass";
		case ALTO_CLEF: return "alto";
		case SOPRAN_CLEF: return "soprano";
		case TENOR_CLEF: return "tenor";
		case NEUTRAL_CLEF1: 
		case NEUTRAL_CLEF2: return "percussion";
	}
	return "treble";
}
const char *NedResource::getLilyPondKeySigName(int keysig_number) {
	switch (keysig_number) {
		case -6: return "\\key ges \\major";
		case -5: return "\\key des \\major";
		case -4: return "\\key as \\major";
		case -3: return "\\key es \\major";
		case -2: return "\\key bes \\major";
		case -1: return "\\key f \\major";
		case  1: return "\\key g \\major";
		case  2: return "\\key d \\major";
		case  3: return "\\key a \\major";
		case  4: return "\\key e \\major";
		case  5: return "\\key b \\major";
		case  6: return "\\key fis \\major";
	}
	return "";
}

void NedResource::makeLilyString(char *ori, char *buffer) {
	int len, i;
	glong pos;
	gchar* cpori;
	gunichar cori;

	strcpy(buffer+1, ori);
	buffer[0] = '"';
	cpori = buffer + 1;
	cori = g_utf8_get_char(cpori);
	while (cori != '\0') {
		if (cori == '"') {
			len = strlen(buffer);
			pos = g_utf8_pointer_to_offset(buffer, cpori);
			for (i = len + 1; i > pos; i--) {
				buffer[i] = buffer[i - 1];
			}
			buffer[pos] = '\\';
			cpori = g_utf8_next_char(cpori);
		}
		cpori = g_utf8_next_char(cpori);
		cori = g_utf8_get_char(cpori);
	}
	len = strlen(buffer);
	buffer[len++] = '"';
	buffer[len++] = '\0';
}
			



void NedResource::showLicense(GtkWindow *parent) {
	const gchar *the_text = "This program is free software; you can redistribute it and/or modify it under the\n"
	"terms of the GNU General Public License as published by the Free Software\n"
	"Foundation; either version 2 of the License, or (at your option) any later version.\n"
	"\n"
	"This program is distributed in the hope that it will be useful, but WITHOUT ANY\n"
	"WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A\n"
	"PARTICULAR PURPOSE. See the GNU General Public License for more details.\n"
	"\n"
	"You should have received a copy of the GNU General Public License along with this\n"
	"program; (See \"COPYING\"). If not, If not, see <http://www.gnu.org/licenses/>.";
	GtkWidget *license_text = gtk_text_view_new();
	GtkTextBuffer *textbuffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (license_text));
	gtk_text_buffer_set_text (textbuffer, the_text, -1);
	gtk_text_view_set_editable(GTK_TEXT_VIEW (license_text), FALSE);
	GtkWidget *license_dialog = gtk_dialog_new_with_buttons("LICENSE", parent, (GtkDialogFlags) (GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT),
		GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);
	g_signal_connect_swapped (license_dialog, "response", G_CALLBACK (gtk_widget_destroy), license_dialog);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(license_dialog)->vbox), license_text);
	gtk_widget_show_all(license_dialog);
}
