/* -*- Mode:C++; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/*
 * Copyright (c) 2007-2009 Strasbourg University
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Author: Sebastien Vincent <vincent@clarinet.u-strasbg.fr>
 */

#ifndef IPV6_L4_PROTOCOL_H
#define IPV6_L4_PROTOCOL_H

#include "ns3/object.h"
#include "ns3/ipv6-header.h"

namespace ns3
{

class Packet;
class Ipv6Address;
class Ipv6Interface;

/**
 * \class Ipv6L4Protocol
 * \brief IPv6 L4 protocol abstract class
 */
class Ipv6L4Protocol : public Object
{
public:
  /**
   * \enum RxStatus_e
   * \brief Status of receive.
   */
  enum RxStatus_e 
  {
    RX_OK, /**< Receive OK */
    RX_CSUM_FAILED, /**< Checksum of layer 4 protocol failed */
    RX_ENDPOINT_UNREACH /**< Destination unreachable */
  };

  /**
   * \brief Get the type identifier.
   * \return type identifier
   */
  static TypeId GetTypeId (void);

  /**
   * \brief Destructor.
   */
  virtual ~Ipv6L4Protocol ();

  /**
   * \brief Get the protocol number.
   * \return protocol number
   */
  virtual int GetProtocolNumber () const = 0;

  /**
   * \brief Receive method.
   *
   * Called from lower-level layers to send the packet up
   * in the stack.
   * \param p packet to forward up
   * \param src source address of packet received
   * \param dst address of packet received
   * \param incomingInterface the Ipv6Interface on which the packet arrived
   * \return status (OK, destination unreachable or checksum failed)
   */
  virtual enum RxStatus_e Receive (Ptr<Packet> p, Ipv6Address const &src, 
                                   Ipv6Address const &dst, 
                                   Ptr<Ipv6Interface> incomingInterface) = 0;

  /**
   * \brief ICMPv6 receive method.
   * \param icmpSource the source address of the ICMPv6 message
   * \param icmpTtl the ttl of the ICMPv6 message
   * \param icmpType the 'type' field of the ICMPv6 message
   * \param icmpCode the 'code' field of the ICMPv6 message
   * \param icmpInfo extra information dependent on the ICMPv6 message
   *        generated by Icmpv6L4Protocol
   * \param payloadSource the source address of the packet which triggered
   *        the ICMPv6 message
   * \param payloadDestination the destination address of the packet which
   *        triggered the ICMPv6 message.
   * \param payload the first 8 bytes of the UDP header of the packet
   *        which triggered the ICMPv6 message.
   */
  virtual void ReceiveIcmp (Ipv6Address icmpSource, uint8_t icmpTtl,
                            uint8_t icmpType, uint8_t icmpCode, uint32_t icmpInfo,
                            Ipv6Address payloadSource, Ipv6Address payloadDestination,
                            const uint8_t* payload);

};

} /* namespace ns3 */

#endif /* IPV6_L4_PROTOCOL_H */

