/* Written by and copyright Seth LaForge <sethml@ofb.net>. */

/* _indextable: NoSQL internal tool, called by nosql index.
   Copyright (C) 2000 Seth LaForge <sethml@ofb.net>

   Modifications by Carlo Strozzi <carlos@linux.it>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 dated June, 1991.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/**********************************************************************
 * Usage: $0 initial_tell indices ... <table |sort ... >table.x.*.rdb
 *   initial_tell: file position immediately after header.
 *   indices: indices of column numbers to index on, zero-based.
 *   stdin: the table, with header munged to reflect the output.
 *   stdout: initial two lines (header) verbatim, followed by table
 *   with selected columns and index information.
 *********************************************************************/

/*
static char Psz_RCS [] =
    "$Id: _indextable.c,v 1.2 2006/03/10 11:26:13 carlo Exp $";
*/

/*********************************************************************/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>


#define READSIZE (1024 * 64)

typedef struct ReadBuffer {
  char *buffer, *start;
  int size, fill;
} ReadBuffer;

static void initBuffer(ReadBuffer *b) {
  b->size = READSIZE * 2;
  b->start = b->buffer = malloc(b->size);
  b->fill = 0;
}

/* Add data to the input buffer.  Returns 0 on EOF. */
static int fillBuffer(ReadBuffer *b, int fd) {
  int n;
  if (b->size - (b->start - b->buffer) - b->fill < READSIZE) {
    if (b->start - b->buffer >= READSIZE) {
      memmove(b->buffer, b->start, b->fill);
      b->start = b->buffer;
    } else {
      n = b->start - b->buffer;
      b->buffer = realloc(b->buffer, b->size += READSIZE);
      b->start = b->buffer + n;
    }
  }
  assert(b->size - (b->start - b->buffer) - b->fill >= READSIZE);
  n = read(fd, b->start + b->fill, READSIZE);
  if (n == -1) { perror("read"); exit(1); }
  b->fill += n;
  return n;
}

/* Advance start. */
static void drainBuffer(ReadBuffer *b, int delta) {
  assert(delta <= b->fill);
  b->start += delta; b->fill -= delta;
}


int main(int argc, char **argv) {
  int tell, *indices, i, maxIndex = 0, *columns;
  ReadBuffer b;

  /* Process arguments. */
  if (argc < 3) {
    fprintf(stderr, "%s: too few arguments.\n", argv[0]);
    exit(255);
  }
  tell = atoi(argv[1]);
  if (!(indices = malloc((argc - 2 + 1) * sizeof(*indices))))
    { perror("malloc"); exit(1); }
  for (i = 0; i < argc - 2; ++i) {
    indices[i] = atoi(argv[i + 2]);
    if (indices[i] > maxIndex)
      maxIndex = indices[i];
  }
  indices[i] = -1;
  initBuffer(&b);
  if (!(columns = malloc(sizeof(*columns) * (maxIndex + 1))))
    { perror("malloc"); exit(1); }
  
  /* Pass the first line through unchanged - header. */
  for (i = 0; i < 1; ++i) {
    int off = 0;
    while (1) {
      if (off == b.fill)
	if (0 == fillBuffer(&b, 0)) {
	  fprintf(stderr, "Premature end of file.\n");
	  exit(255);
	}
      if (b.start[off] == '\n') 
	{ ++off; break; }
      ++off;
    }
    fwrite(b.start, 1, off, stdout);
    drainBuffer(&b, off);
  }

  /* Process rows of input. */
  while (1) {
    int col = 0, off = 0;

    /* Find columns. */
    columns[col++] = off;
    while (1) {
      if (off == b.fill)
	if (0 == fillBuffer(&b, 0)) {
	  if (off == 0) {
	    exit(0);  /* All done! */
	  } else {
	    fprintf(stderr, "Premature end of file.\n");
	    exit(255);
	  }
	}
      if (b.start[off] == '\n') {
	++off;
	if (col <= maxIndex + 1) columns[col] = off;
	++col;
	break;
      } else if (b.start[off] == '\t') {
	++off;
	if (col <= maxIndex + 1) columns[col] = off;
	++col;
      } else
	++off;
    }

    /* Output row. */
    if (col - 1 <= maxIndex) {
      fprintf(stderr, "Short row at %d.\n", tell);
    } else {
      /* Output index columns. */
      for (i = 0; ; ++i) {
	int ind = indices[i], cur, next, count;
	if (ind == -1) break;
	cur = columns[ind]; next = columns[ind + 1]; count = next - cur - 1;
	if (count != fwrite(b.start + cur, 1, count, stdout)) 
	  { perror("write"); exit(1); }
	if (EOF == putchar('\t'))
	  { perror("write"); exit(1); }
      }

      /* Output index value. */
      printf("%d\n", tell);
    }

    tell += off;
    drainBuffer(&b, off);
  }
}

/* End of program */
