/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright 2009 by Sun Microsystems, Inc.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * $RCSfile: LBFGSSolverImpl.java,v $
 * $Revision: 1.1 $
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

package com.sun.star.NLPSolver;

import com.sun.star.uno.XComponentContext;
import com.sun.star.lib.uno.helper.Factory;
import com.sun.star.lang.XSingleComponentFactory;
import com.sun.star.registry.XRegistryKey;
import com.sun.star.sheet.SolverConstraint;
import com.sun.star.sheet.SolverConstraintOperator;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import riso.numerical.LBFGS;


public final class LBFGSSolverImpl extends BaseNLPSolver
   implements com.sun.star.lang.XServiceInfo
{
    private static final String m_implementationName = LBFGSSolverImpl.class.getName();
    private static final String[] m_serviceNames = {
        "com.sun.star.sheet.Solver",
        "com.sun.star.beans.PropertySet"
    };

    public LBFGSSolverImpl( XComponentContext context )
    {
        super(context, "L-BFGS (unconstrained)");

        registerProperty(m_numberOfCorrections);
        registerProperty(m_epsilon);
        registerProperty(m_gradientStepSize);
    }

    public static XSingleComponentFactory __getComponentFactory( String sImplementationName ) {
        XSingleComponentFactory xFactory = null;

        if ( sImplementationName.equals( m_implementationName ) )
            xFactory = Factory.createComponentFactory(LBFGSSolverImpl.class, m_serviceNames);
        return xFactory;
    }

    public static boolean __writeRegistryServiceInfo( XRegistryKey xRegistryKey ) {
        return Factory.writeRegistryServiceInfo(m_implementationName,
                                                m_serviceNames,
                                                xRegistryKey);
    }

    // com.sun.star.lang.XServiceInfo:
    public String getImplementationName() {
         return m_implementationName;
    }

    public boolean supportsService( String sService ) {
        int len = m_serviceNames.length;

        for( int i=0; i < len; i++) {
            if (sService.equals(m_serviceNames[i]))
                return true;
        }
        return false;
    }

    public String[] getSupportedServiceNames() {
        return m_serviceNames;
    }

    // com.sun.star.sheet.XSolver:
    private int[] m_iflag = { -1 };
    private double[] m_gradient;

    private PropertyInfo<Integer> m_numberOfCorrections = new PropertyInfo<Integer>("NumberOfCorrections", 7, "Number of Correction (3-7)"); //parameter "m"
    private PropertyInfo<Double> m_epsilon = new PropertyInfo<Double>("Epsilon", 0.000001, "Epsilon"); //precision
    private PropertyInfo<Double> m_gradientStepSize = new PropertyInfo<Double>("GradientStepSize", 0.0001, "Gradient Stepsize"); //parameter "h"
    
    private ArrayList<ExtSolverConstraint> m_extConstraints = new ArrayList<ExtSolverConstraint>();

    @Override
    public void setConstraints(SolverConstraint[] constraints) {
        super.setConstraints(constraints);
        
        m_extConstraints.clear();
        for (int i = 0; i < m_constraintCount; i++)
            m_extConstraints.add(new ExtSolverConstraint(m_constraintCells[i], m_constraints[i].Operator, m_constraints[i].Right));
    }

    private double evaluateFunction() {
        return m_objectiveCell.getValue() * (m_maximize ? -1.0 : 1.0);
    }

    private double[] approximateGradient(double[] parameters) {
        double[] gradient = new double[m_variableCount];

        //Since XCell.setValue is probably the slowest part in this algorithm,
        //I call it only as often es necessary. Therefore I skip the last one
        //in the iteration since the modulo operation in the next loop will
        //address it anyway.
        for (int i = 0; i < m_variableCount - 1; i++) {
            m_variableCells[i].setValue(parameters[i]);
        }

        for (int i = 0; i < m_variableCount; i++) {
            //Always reset the last (i - 1) cell to the original value.
            int lastCell = (i - 1 + m_variableCount) % m_variableCount;
            m_variableCells[lastCell].setValue(parameters[lastCell]);

            m_variableCells[i].setValue(parameters[i] + m_gradientStepSize.getValue());
            double g1 = evaluateFunction();
            
            m_variableCells[i].setValue(parameters[i] - m_gradientStepSize.getValue());
            double g2 = evaluateFunction();

            gradient[i] = (g1 - g2) / (2*m_gradientStepSize.getValue());
        }

        return gradient;
    }

    public double getPenalty() {
        double result = 0.0;

        for (int i = 0; i < m_constraintCount; i++) {
            ExtSolverConstraint constraint = m_extConstraints.get(i);
            switch (constraint.Operator.getValue()) {
                case SolverConstraintOperator.EQUAL_value:
                    result += Math.pow(Math.abs(constraint.Data - constraint.getLeftValue()), 2);
                    break;
                case SolverConstraintOperator.GREATER_EQUAL_value:
                    result += Math.pow(Math.min(0.0, constraint.Data - constraint.getLeftValue()), 2);
                    break;
                case SolverConstraintOperator.LESS_EQUAL_value:
                    result += Math.pow(Math.min(0.0, constraint.getLeftValue() - constraint.Data), 2);
                    break;
            }
        }

        return result;
    }

    public void solve() {
        initializeSolve();

        m_gradient = approximateGradient(m_currentParameters);
        m_functionValue = evaluateFunction() + getPenalty();

        m_iflag[0] = 0;
        double[] diag = new double[m_variableCount];
        int[] iprint = {-1, 0};

        try {
            do {
                LBFGS.lbfgs(m_variableCount, m_numberOfCorrections.getValue(),
                        m_currentParameters, m_functionValue, m_gradient, false,
                        diag, iprint, m_epsilon.getValue(), 10e-16, m_iflag);

                if (m_iflag[0] == 1) {
                    for (int i = 0; i < m_variableCount; i++)
                        m_variableCells[i].setValue(m_currentParameters[i]);

                    if (m_objectiveCell.getError() != 0)
                        m_iflag[0] = -1;

                    m_functionValue = evaluateFunction() + getPenalty();

                    m_gradient = approximateGradient(m_currentParameters);
                }
            } while (m_iflag[0] != 0);
        } catch (Exception ex) {
            Logger.getLogger(LBFGSSolverImpl.class.getName()).log(Level.SEVERE, null, ex);
        }

        finalizeSolve();
    }

}
