# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003-2005 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import optparse
import urlparse
import urllib
import os
import socket
import sys
import logging
import shutil
import random
import traceback
import cStringIO
import imp
import locale
import multiprocessing

if os.name == 'nt': # locale setting for windows
    lang = os.getenv('LANG')
    if lang is None:
        os.environ['LANG'] = locale._build_localename(locale.getdefaultlocale())
import gettext
gettext.install('ninix')

import gtk
import glib
import pango

import ninix.pix
import ninix.home
import ninix.prefs
import ninix.sakura
import ninix.sstp
import ninix.communicate
import ninix.ngm
import ninix.lock
import ninix.install
import ninix.plugin
import ninix.nekodorif
import ninix.kinoko

USAGE = 'Usage: ninix [options]'
parser = optparse.OptionParser(USAGE)
parser.add_option('--sstp-port', type='int', dest='sstp_port',
                  help='additional port for listening SSTP requests')
parser.add_option('--debug', action='store_true', help='debug')
parser.add_option('--logfile', type='string', help='logfile name')

# XXX: check stderr - logger's default destination and redirect it if needed
# (See http://bugs.python.org/issue706263 for more details.)
try:
    tmp = os.dup(sys.__stderr__.fileno())
except:
    sys.stderr = file(os.devnull, 'w')
else:
    os.close(tmp)
logger = logging.getLogger()
logger.setLevel(logging.INFO) # XXX

def handleException(exception_type, value, tb):
    logger.error('Uncaught exception',
                 exc_info=(exception_type, value, tb))
    response_id = 1
    dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_ERROR, gtk.BUTTONS_NONE,
                               _('A ninix-aya error has been detected.'))
    dialog.set_title(_('Bug Detected'))
    dialog.set_position(gtk.WIN_POS_CENTER)
    dialog.set_gravity(gtk.gdk.GRAVITY_CENTER)
    button = dialog.add_button(_('Show Details'), response_id)
    dialog.add_button(gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE)
    textview = gtk.TextView()
    textview.set_editable(False)
    left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
    width = scrn_w / 2
    height = scrn_h / 4
    textview.set_size_request(width, height)
    textview.show()
    sw = gtk.ScrolledWindow()
    sw.show()
    sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.add(textview)
    frame = gtk.Frame()
    frame.set_shadow_type(gtk.SHADOW_IN)
    frame.add(sw)
    frame.set_border_width(7)
    dialog.vbox.add(frame)
    stringio = cStringIO.StringIO()
    traceback.print_exception(exception_type, value, tb, None, stringio)
    textbuffer = textview.get_buffer()
    textbuffer.set_text(stringio.getvalue())
    while 1:
        if dialog.run() == response_id:
            frame.show()
            button.set_sensitive(0)
        else: # close button
            break
    dialog.destroy()
    raise SystemExit

sys.excepthook = handleException


def main():
    if gtk.pygtk_version < (2,10,0):
        logging.critical('PyGtk 2.10.0 or later required')
        raise SystemExit
    # parse command line arguments
    (options, rest) = parser.parse_args()
    if rest:
        parser.error('Unknown option(s)')
    if options.logfile:
        logger.addHandler(logging.FileHandler(options.logfile))
    # TCP 7743：伺か（未使用）(IANA Registered Port for SSTP)
    # UDP 7743：伺か（未使用）(IANA Registered Port for SSTP)
    # TCP 9801：伺か          (IANA Registered Port for SSTP)
    # UDP 9801：伺か（未使用）(IANA Registered Port for SSTP)
    # TCP 9821：SSP
    # TCP 11000：伺か（廃止） (IANA Registered Port for IRISA)
    sstp_port = [9801]
    # parse command line options
    if options.sstp_port is not None:
        if options.sstp_port < 1024:
            logging.warning('Invalid --sstp-port number (ignored)')
        else:
            sstp_port.append(options.sstp_port)
    if options.debug is not None and options.debug:
        logger.setLevel(logging.DEBUG)
    home_dir = ninix.home.get_ninix_home()
    if not os.path.exists(home_dir):
        try:
            os.makedirs(home_dir)
        except:
            raise SystemExit, 'Cannot create Home directory (abort)\n'
    # aquire Inter Process Mutex (not Global Mutex)
    with open(os.path.join(ninix.home.get_ninix_home(), '.lock'), 'w') as f:
        try:
            ninix.lock.lockfile(f)
        except:
            raise SystemExit, 'ninix-aya is already running'
        # start
        logging.info('loading...')
        app = Application(sstp_port)
        logging.info('done.')
        app.run()
        try:
            ninix.lock.unlockfile(f)
        except:
            pass


class SSTPControler(object):

    def __init__(self, sstp_port):
        self.request_parent = lambda *a: None # dummy
        self.sstp_port = sstp_port
        self.sstp_servers = []
        self.__sstp_queue = []
        self.__sstp_flag = 0
        self.__current_sender = None

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            }
        handler = handlers.get(event, getattr(self, event, None))
        if handler is None:
            result = self.request_parent(
                event_type, event, *arglist, **argdict)
        else:
            result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def enqueue_request(self, event, script_odict, sender, handle,
                        address, show_sstp_marker, use_translator,
                        entry_db, request_handler):
        self.__sstp_queue.append(
            (event, script_odict, sender, handle, address, show_sstp_marker,
             use_translator, entry_db, request_handler))

    def check_request_queue(self, sender):
        count = 0
        for request in self.__sstp_queue:
            if request[2].split(' / ')[0] == sender.split(' / ')[0]:
                count += 1
        if self.__sstp_flag and \
           self.__current_sender.split(' / ')[0] == sender.split(' / ')[0]:
            count += 1
        return str(count), str(len(self.__sstp_queue))

    def set_sstp_flag(self, sender):
        self.__sstp_flag = 1
        self.__current_sender = sender

    def reset_sstp_flag(self):
        self.__sstp_flag = 0
        self.__current_sender = None        

    def handle_sstp_queue(self):
        if self.__sstp_flag or not self.__sstp_queue:
            return
        event, script_odict, sender, handle, address, \
            show_sstp_marker, use_translator, \
            entry_db, request_handler = self.__sstp_queue.pop(0)
        working = bool(event is not None)
        for if_ghost in script_odict.keys():
           if if_ghost and self.request_parent('GET', 'if_ghost', if_ghost, working):
              self.request_parent('NOTIFY', 'select_current_sakura', if_ghost)
              default_script = script_odict[if_ghost]
              break
        else:
           if not self.request_parent('GET', 'get_preference', 'allowembryo'):
              if event is None:
                 if request_handler:
                    request_handler.send_response(420) # Refuse
                 return
              else:
                 default_script = None
           else:
              if '' in script_odict: # XXX
                 default_script = script_odict['']
              else:
                 default_script = script_odict.values()[0]
        if event is not None:
           script = self.request_parent('GET', 'get_event_response', event)
        else:
           script = None
        if not script:
              script = default_script
        if script is None:
           if request_handler:
              request_handler.send_response(204) # No Content
           return
        self.set_sstp_flag(sender)
        self.request_parent(
           'NOTIFY', 'enqueue_script',
           script, sender, handle, address,
           show_sstp_marker, use_translator, entry_db,
           request_handler, temp_mode=True)

    def receive_sstp_request(self):
        try:
            for sstp_server in self.sstp_servers:
                sstp_server.handle_request()
        except socket.error as e:
            code, message = e.args
            logging.error('socket.error: {0} ({1:d})'.format(message, code))
        except ValueError: # may happen when ninix is terminated
            return

    def get_sstp_port(self):
        if not self.sstp_servers:
            return None
        return self.sstp_servers[0].server_address[1]

    def quit(self):
        for server in self.sstp_servers:
            server.close()

    def start_servers(self):
        for port in self.sstp_port:
            try:
                server = ninix.sstp.SSTPServer(('', port))
            except socket.error as e:
                code, message = e.args
                logging.warning(
                    'Port {0:d}: {1} (ignored)'.format(port, message))
                continue
            server.set_responsible(self.handle_request)
            self.sstp_servers.append(server)
            logging.info('Serving SSTP on port {0:d}'.format(port))


class PluginControler:

    def __init__(self):
        self.jobs = {}
        self.queue = {}
        self.data = {}
        self.request_parent = lambda *a: None # dummy
        
    def set_responsible(self, request_method):
        self.request_parent = request_method

    def save_data(self, plugin_dir):
        if plugin_dir not in self.data:
            return
        home_dir = ninix.home.get_ninix_home()
        target_dir = os.path.join(home_dir, 'plugin', plugin_dir)
        path = os.path.join(target_dir, 'SAVEDATA')
        with open(path, 'w') as f:
            f.write('#plugin: {0:1.1f}\n'.format(ninix.home.PLUGIN_STANDARD[1]))
            for name, value in self.data[plugin_dir].items():
                if value is None:
                    continue
                f.write('{0}:{1}\n'.format(name, value))

    def load_data(self, plugin_dir):
        home_dir = ninix.home.get_ninix_home()
        target_dir = os.path.join(home_dir, 'plugin', plugin_dir)
        path = os.path.join(target_dir, 'SAVEDATA')
        if not os.path.exists(path):
            return {}
        data = {}
        try:
            with open(path, 'r') as f:
                line = f.readline()
                line = line.rstrip('\r\n')
                if not line.startswith('#plugin:'):
                    return {}
                try:
                    standard = float(line[8:])
                except:
                    return {}
                if standard < ninix.home.PLUGIN_STANDARD[0] or \
                        standard > ninix.home.PLUGIN_STANDARD[1]:
                    return {}
                for line in f:
                    line = line.rstrip('\r\n')
                    key, value = line.split(':', 1)
                    data[key] = value
        except IOError as e:
            code, message = e.args
            logging.error('cannot read {0}'.format(path))
        return data

    def terminate_plugin(self):
        for plugin_dir in self.data.keys():
            self.save_data(plugin_dir)
        return ## FIXME

    def check_queue(self):
        for plugin_dir in self.jobs.keys():
            if not self.queue[plugin_dir].empty():
                data = self.queue[plugin_dir].get()
                self.queue[plugin_dir].task_done()
                if not isinstance(data, dict):
                    continue
                for name, value in data.items():
                    if not isinstance(name, basestring) or ':' in name or \
                            not (isinstance(value, basestring) or value is None):
                        break
                else:
                    if plugin_dir not in self.data:
                        self.data[plugin_dir] = {}
                    self.data[plugin_dir].update(data)

    def exec_plugin(self, plugin_dir, argv, caller):
        if plugin_dir in self.jobs and self.jobs[plugin_dir].is_alive():
            logging.warning('plugin {0} is already running'.format(plugin_dir))
            return
        module_name, ext = os.path.splitext(argv[0])
        home_dir = ninix.home.get_ninix_home()
        target_dir = os.path.join(home_dir, 'plugin', plugin_dir)
        module = self.__import_module(module_name, target_dir)
        if module is None:
            return
        port = self.request_parent('GET', 'get_sstp_port')
        queue = multiprocessing.JoinableQueue()
        if plugin_dir in self.data:
            data = self.data[plugin_dir]
        else:
            data = self.load_data(plugin_dir)
        p = module.Plugin(port, target_dir, argv[1:], home_dir, caller,
                          queue, data)
        if not isinstance(p, ninix.plugin.BasePlugin):
            return
        self.jobs[plugin_dir] = p
        self.queue[plugin_dir] = queue
        p.daemon = True
        p.start()
        if os.name == 'nt' and target_dir in sys.path:
            # XXX: an opposite of the BasePlugin.__init__ hack
            sys.path.remove(target_dir)

    def start_plugins(self, plugins):
        for plugin_name, plugin_dir, startup, menu_items in plugins:
            if startup is not None:
                self.exec_plugin(plugin_dir, startup,
                                 {'name': '', 'directory': ''})

    def __import_module(self, name, directory):
        fp = None
        try:
            return reload(sys.modules[name])
        except:
            pass
        try:
            fp, pathname, description = imp.find_module(name, [directory])
        except:
            return None
        try:
            return imp.load_module(name, fp, pathname, description)
        finally:
            if fp:
                fp.close()
        return None


class Application(object):

    def __init__(self, sstp_port=[9801, 11000]):
        self.loaded = False
        self.confirmed = False
        self.console = Console(self)
        # create preference dialog
        self.prefs = ninix.prefs.PreferenceDialog()
        self.prefs.set_responsible(self.handle_request)
        self.sstp_controler = SSTPControler(sstp_port)
        self.sstp_controler.set_responsible(self.handle_request)
        # create usage dialog
        self.usage_dialog = UsageDialog()
        self.communicate = ninix.communicate.Communicate()
        # create plugin manager
        self.plugin_controler = PluginControler()
        self.plugin_controler.set_responsible(self.handle_request)
        # create ghost manager
        self.__ngm = ninix.ngm.NGM()
        self.__ngm.set_responsible(self.handle_request)
        self.current_sakura = None
        # create installer
        self.installer = ninix.install.Installer()
        self.ghosts = ninix.home.search_ghosts()
        self.balloons = ninix.home.search_balloons()
        self.plugins = ninix.home.search_plugins()
        self.nekoninni = ninix.home.search_nekoninni()
        self.katochan = ninix.home.search_katochan()
        self.kinoko = ninix.home.search_kinoko()
        self.ghost_list = [
            self.__create_ghost_list_item(i) for i in \
                sorted(self.ghosts.keys())]

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            'close_all': self.close_all_ghosts,
            'edit_preferences': self.prefs.edit_preferences,
            'get_preference': self.prefs.get,
            'get_otherghostname': self.communicate.get_otherghostname,
            'rebuild_ghostdb': self.communicate.rebuild_ghostdb,
            'reset_sstp_flag': self.sstp_controler.reset_sstp_flag,
            'start_sakura': self.start_sakura_cb,
            'send_message': self.communicate.send_message,
            'get_sstp_port' : self.sstp_controler.get_sstp_port,
            }
        handler = handlers.get(event,
                               getattr(self, event,
                                       lambda *a: None)) ## FIXME
        result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def do_install(self, filename):
        try:
            filetype, target_dir = self.installer.install(
                filename, ninix.home.get_ninix_home())
        except:
            target_dir = None
        if target_dir is not None:
            if filetype == 'ghost':
                self.add_sakura(target_dir)
            elif filetype == 'supplement':
                pass ## FIXME
            elif filetype == 'balloon':
                self.balloons = ninix.home.search_balloons()
            elif filetype == 'plugin':
                self.plugins = ninix.home.search_plugins()
            elif filetype == 'nekoninni':
                self.nekoninni = ninix.home.search_nekoninni()
            elif filetype == 'katochan':
                self.katochan = ninix.home.search_katochan()
            elif filetype == 'kinoko':
                self.kinoko = ninix.home.search_kinoko()

    @property
    def current_sakura_instance(self):
        set_type, i, j = self.current_sakura
        num = self.get_index(i) ## FIXME
        return self.ghost_list[num]['instance']

    def get_index(self, i): ## FIXME
        keys = sorted(self.ghosts.keys())
        return keys.index(i)

    def create_ghost(self, data):
        ghost = ninix.sakura.Sakura(data)
        ghost.set_responsible(self.handle_request)
        return ghost

    def get_sakura_cantalk(self):
        return self.current_sakura_instance.cantalk

    def get_event_response(self, event, *arglist, **argdict): ## FIXME
        return self.current_sakura_instance.get_event_response(*event)

    def keep_silence(self, quiet):
        self.current_sakura_instance.keep_silence(quiet)

    def get_ghost_name(self): ## FIXME
        sakura = self.current_sakura_instance
        return '{0},{1}'.format(sakura.get_selfname(), sakura.get_keroname())

    def enqueue_event(self, event, *arglist, **argdict):
        self.current_sakura_instance.enqueue_event(event, *arglist, **argdict)

    def enqueue_script(self, script, sender, handle,
                       host, show_sstp_marker, use_translator,
                       db=None, request_handler=None, temp_mode=False):
        sakura = self.current_sakura_instance
        if temp_mode:
            sakura.enter_temp_mode()
        sakura.enqueue_script(script, sender, handle,
                              host, show_sstp_marker, use_translator,
                              db, request_handler)

    def get_working_ghost(self, cantalk=0):
        working_list = [item['instance'] for item in self.ghost_list \
                            if item is not None and \
                            item['instance'] is not None and \
                            item['instance'].is_running()]
        if cantalk:
            working_list = [sakura for sakura in working_list if sakura.cantalk]
        return working_list

    def get_ghost_list(self): ## FIXME
        return self.ghost_list

    def __create_ghost_list_item(self, i):
        assert i in self.ghosts
        if self.ghosts[i] is None:
            return None
        set_type = 'g'
        desc = self.ghosts[i][0]
        shiori_dir = self.ghosts[i][1]
        icon = desc.get('icon', None)
        if icon is not None:
            icon_path = os.path.join(shiori_dir, icon)
            if not os.path.exists(icon_path):
                icon_path = None
        else:
            icon_path = None
        name = desc.get('name', i)
        shell_list = []
        surface_set = self.ghosts[i][3]
        for j in sorted(surface_set.keys()):
            value = (set_type, i, j)
            shell_name = surface_set[j][0]
            shell_list.append((shell_name, value))
        if not shell_list: # XXX
            return None
        item = {}
        item['name'] = name
        item['icon'] = icon_path
        item['shell'] = shell_list
        item['instance'] = None
        return item

    def update_ghost_list(self, i, sakura):
        keys = sorted(self.ghosts.keys())
        num = keys.index(i)
        self.ghost_list[num] = self.__create_ghost_list_item(i)
        if self.ghost_list[num] is not None:
            self.ghost_list[num]['instance'] = sakura
        else:
            pass ## FIXME: ghost instance should be deleted

    def get_balloon_list(self):
        balloon_list = []
        balloons = self.balloons
        keys = sorted(balloons.keys())
        for key in keys:
            desc, balloon = balloons[key]
            subdir = balloon['balloon_dir'][0]
            name = desc.get('name', subdir)
            balloon_list.append((name, subdir))
        return balloon_list

    def get_plugin_list(self): ## FIXME
        plugin_list = []
        for i, plugin in enumerate(self.plugins):
            plugin_name = plugin[0]
            menu_items = plugin[3]
            if not menu_items:
                continue
            item = {}
            item['name'] = plugin_name
            item['icon'] = None
            item_list = []
            for j, menu_item in enumerate(menu_items):
                label = menu_item[0]
                value = (i, j)
                item_list.append((label, value))
            item['items'] = item_list
            plugin_list.append(item)
        return plugin_list

    def get_nekodorif_list(self): ## FIXME
        nekodorif_list = []
        nekoninni = self.nekoninni
        for nekoninni_name, nekoninni_dir in nekoninni:
            if not nekoninni_name:
                continue
            item = {}
            item['name'] = nekoninni_name
            item['dir'] = nekoninni_dir
            nekodorif_list.append(item)
        return nekodorif_list

    def get_kinoko_list(self): ## FIXME
        return self.kinoko

    def load(self):
        # load user preferences
        self.prefs.load()
        # choose default ghost/shell
        directory = self.prefs.get('sakura_dir')
        name = self.prefs.get('sakura_name') # XXX: backward compat
        surface = self.prefs.get('sakura_surface')
        default_sakura = self.find_ghost_by_dir(directory, surface) or \
                         self.find_ghost_by_name(name, surface) or \
                         self.choose_default_sakura()
        # load ghost
        self.current_sakura = default_sakura
        keys = sorted(self.ghosts.keys())
        for num, i in enumerate(keys):
            if self.ghost_list[num] is None: # XXX
                continue
            sakura = self.create_ghost(self.ghosts[i])
            self.ghost_list[num]['instance'] = sakura
        ##names = self.get_ghost_names()
        ##for i in range(len(names)):
        ##    logging.info(
        ##        'GHOST({0:d}): {1}'.format(
        ##            i, names[i].encode('utf-8', 'ignore')))
        self.start_sakura(self.current_sakura, init=1)

    def find_ghost_by_dir(self, directory, surface):
        if directory in self.ghosts:
            surface_set = self.ghosts[directory][3]
            if surface not in surface_set:
                surface = sorted(surface_set.keys())[0]
            return ('g', directory, surface)
        return None

    def find_ghost_by_name(self, name, surface):
        for i in self.ghosts:
            if self.ghosts[i] is None:
                continue
            desc = self.ghosts[i][0]
            try:
                if desc.get('name') == name:
                    surface_set = self.ghosts[i][3]
                    assert surface_set
                    if surface not in surface_set:
                        surface = sorted(surface_set.keys())[0]
                    return ('g', i, surface)
            except: # old preferences(EUC-JP)
                pass
        return None

    def choose_default_sakura(self):
        keys = sorted(self.ghosts.keys())
        for i in keys:
            if self.ghosts[i] is None:
                continue
            if self.ghosts[i][3]: # surface_set
                surface = sorted(self.ghosts[i][3].keys())
                return ('g', i, surface[0])
        raise RuntimeError, 'should not reach here'

    def find_balloon_by_name(self, name):
        balloons = self.balloons
        for desc, balloon in balloons.values(): # FIXME: sort
            try:
                if desc.get('name') == name:
                    return desc, balloon
                if balloon['balloon_dir'][0] == ninix.home.get_normalized_path(name): # XXX
                    return desc, balloon
            except: # old preferences(EUC-JP)
                pass
        return None

    def find_balloon_by_subdir(self, subdir):
        balloons = self.balloons
        for desc, balloon in balloons.values(): ## FIXME: sort
            try:
                if balloon['balloon_dir'][0] == subdir:
                    return desc, balloon
                if ninix.home.get_normalized_path(desc.get('name')) == subdir: # XXX
                    return desc, balloon
            except: # old preferences(EUC-JP)
                pass
        return None

    def run(self):
        self.timeout_id = glib.timeout_add(100, self.do_idle_tasks) # 100ms
        gtk.main()

    def get_ghost_names(self): ## FIXME
        return [item['instance'].get_selfname() for item in self.ghost_list if item is not None]

    def if_ghost(self, if_ghost, working=False):
        for sakura in (item['instance'] for item in self.ghost_list \
                           if item is not None):
            assert sakura is not None
            if working:
               if not sakura.is_running() or not sakura.cantalk:
                  continue
            ghost_name = '{0},{1}'.format(sakura.get_selfname(),
                                          sakura.get_keroname())
            if ghost_name == if_ghost:
               return 1
        else:
            return 0

    def update_sakura(self, name, sender):
        item = self.find_ghost_by_name(name, 0)
        if item is None:
            return
        sakura = self.select_ghost_from_list(item)
        if not sakura.is_running():
            self.start_sakura(item, init=1)
        sakura.enqueue_script('\![updatebymyself]\e', sender,
                              None, None, 0, 0, None)

    def select_current_sakura(self, ifghost=None):
        if ifghost is not None:
            for item in self.ghost_list:
                if item is None:
                    continue
                sakura = item['instance']
                assert sakura is not None
                names = '{0},{1}'.format(sakura.get_selfname(),
                                         sakura.get_keroname())
                name =  '{0}'.format(sakura.get_selfname())
                if ifghost in [name, names]:
                    if not sakura.is_running():
                        self.current_sakura = item['shell'][0][1] ## FIXME
                        self.start_sakura(self.current_sakura, init=1, temp=1) ## FIXME
                    else:
                        self.current_sakura = sakura.current
                    break
                else:
                    pass
            else:
                return
        else:
            working_list = self.get_working_ghost(cantalk=1)
            if working_list:
                self.current_sakura = random.choice(working_list).current
            else:
                return ## FIXME

    def close_ghost(self, sakura):
        if not self.get_working_ghost():
            set_type, i, j = sakura.current
            self.prefs.set_current_sakura(i, j)
            self.quit()
        elif self.current_sakura == sakura.current:
            self.select_current_sakura()

    def close_all_ghosts(self):
        for sakura in self.get_working_ghost():
            sakura.close()

    def quit(self):
        glib.source_remove(self.timeout_id)
        self.usage_dialog.close()
        self.sstp_controler.quit() ## FIXME
        self.plugin_controler.terminate_plugin() ## FIXME
        self.save_preferences()
        gtk.main_quit()

    def save_preferences(self):
        try:
            self.prefs.save()
        except IOError:
            logging.error('Cannot write preferences to file (ignored).')
        except:
            pass ## FIXME

    def select_ghost(self, sakura, sequential, event=1, vanished=0):
        keys = sorted(self.ghosts.keys())
        ghosts = [key for key in keys if self.ghosts[key] is not None]
        if len(ghosts) < 2:
            return
        # select another ghost
        set_type, i, j = sakura.current
        assert set_type == 'g'
        if sequential:
            i = (keys.index(i) + 1) % len(keys)
        else:
            ghosts.remove(i)
            i = random.choice(ghosts)
        surface = sorted(self.ghosts[i][3].keys())
        if self.current_sakura == sakura.current: # XXX
            self.current_sakura = ('g', i, surface[0])
        self.change_sakura(sakura, ('g', i, surface[0]), 'automatic',
                           event, vanished)

    def select_ghost_by_name(self, sakura, name, event=1):
        item = self.find_ghost_by_name(name, 0)
        if item is None:
            return
        self.change_sakura(sakura, item, 'automatic', event)

    def change_sakura(self, sakura, item, method, event=1, vanished=0):
        set_type, i, j = item
        assert self.ghosts[i] is not None
        if sakura.current == item: # XXX: needs reloading?
            return
        if sakura.current[1] == i:
            return # XXX: not shell change
        assert set_type == 'g'
        desc, shiori_dir, use_makoto, surface_set, prefix, \
            shiori_dll, shiori_name = self.ghosts[i]
        assert surface_set and j in surface_set
        name, surface_dir, surface_desc, surface_alias, surface, surface_tooltips = \
            surface_set[j]
        def proc(self=self, item=item):
            self.stop_sakura(sakura, self.start_sakura, item, sakura.current)
        if vanished:
            sakura.finalize()
            self.start_sakura(item, sakura.current, vanished)
            self.close_ghost(sakura)
        elif not event:
            proc()
        else:
            name = surface_desc.get('sakura.name', desc.get('sakura.name'))
            sakura.enqueue_event('OnGhostChanging', name, method, proc=proc)

    def stop_sakura(self, sakura, starter=None, *args):
        sakura.finalize()
        if starter is not None:
            starter(*args)
        self.close_ghost(sakura)

    def select_ghost_from_list(self, item):
        set_type, i, j = item
        assert set_type == 'g'
        assert self.ghosts[i] is not None
        num = self.get_index(i) ## FIXME
        assert self.ghost_list[num]['instance'] is not None
        return self.ghost_list[num]['instance']

    def start_sakura(self, item, prev=None, vanished=0, init=0, temp=0):
        set_type, i, j = item
        assert set_type == 'g'
        sakura = self.select_ghost_from_list(item)
        keys = sorted(self.ghosts.keys())
        if prev is not None:
            prev_num = keys.index(prev[1])
            assert self.ghosts[prev[1]] is not None
            assert self.ghost_list[prev_num]['instance'] is not None
        if init:
            ghost_changed = 0
        else:
            assert prev is not None ## FIXME
            if prev[0] == set_type and prev[1] == i:
                ghost_changed = 0
            else:
                ghost_changed = 1
        if ghost_changed:
            name = self.ghost_list[prev_num]['instance'].get_selfname()
        else:
            name = None
        sakura.notify_preference_changed()
        sakura.start(item, init, temp, vanished, ghost_changed, name)

    def notify_preference_changed(self):
        for sakura in self.get_working_ghost():
            sakura.notify_preference_changed()

    def start_sakura_cb(self, item):
        self.start_sakura(item, init=1)

    def get_balloon_description(self, subdir):
        balloon = self.find_balloon_by_subdir(subdir)
        if balloon is None:
            ##logging.warning('Balloon {0} not found.'.format(name))
            default_balloon = self.prefs.get('default_balloon')
            balloon = self.find_balloon_by_subdir(default_balloon)
        if balloon is None:
            balloons = self.balloons
            assert balloons
            key = sorted(balloons.keys())[0]
            balloon = balloons[key]
        return balloon

    def reload_current_sakura(self, sakura):
        self.save_preferences()
        set_type, i, j = sakura.current
        assert set_type == 'g'
        ghost_dir = os.path.split(sakura.get_prefix())[1] # XXX
        ghost_conf = ninix.home.search_ghosts([ghost_dir])
        if ghost_conf:
            self.ghosts[i] = ghost_conf[i]
            sakura.new(*ghost_conf[i]) # reset
        else:
            self.ghosts[i] = None ## FIXME
        self.update_ghost_list(i, sakura)
        item = (set_type, i, j)
        self.start_sakura(item, item, init=1) 

    def add_sakura(self, ghost_dir):
        if ghost_dir in self.ghosts:
            exists = 1
            logging.warning('INSTALLED GHOST CHANGED: {0}'.format(ghost_dir))
            ## FIXME: reload if working
        else:
            exists = 0
            logging.info('NEW GHOST INSTALLED: {0}'.format(ghost_dir))
        keys = list(self.ghosts.keys()) ## FIXME
        if not exists:
            keys.append(ghost_dir)
        keys.sort() ## FIXME
        num = keys.index(ghost_dir)
        ghost_conf = ninix.home.search_ghosts([ghost_dir])
        if ghost_conf:
            self.ghosts[ghost_dir] = ghost_conf[ghost_dir]
        else:
            self.ghosts[ghost_dir] = None ## FIXME
        item = self.__create_ghost_list_item(ghost_dir)
        if item is not None: # XXX
            item['instance'] = self.create_ghost(ghost_conf[ghost_dir])
        if exists:
            self.ghost_list[num] = item
        else:
            self.ghost_list.insert(num, item)

    def vanish_sakura(self, sakura):
        # remove ghost
        prefix = sakura.get_prefix()
        for filename in os.listdir(prefix):
            if os.path.isfile(os.path.join(prefix, filename)):
                if filename != 'HISTORY':
                    try:
                        os.remove(os.path.join(prefix, filename))
                    except:
                        logging.error(
                            '*** REMOVE FAILED *** : {0}'.format(filename))
            else: # dir
                try:
                    shutil.rmtree(os.path.join(prefix, filename))
                except:
                    logging.error(
                        '*** REMOVE FAILED *** : {0}'.format(filename))
        set_type, i, j = sakura.current
        assert set_type == 'g'
        self.select_ghost(sakura, 0, vanished=1)
        self.ghosts[i] = None
        self.update_ghost_list(i, sakura)

    def select_plugin(self, item, target): ## FIXME
        i, j = item
        plugin_name, plugin_dir, startup, menu_items = self.plugins[i]
        label, argv = menu_items[j]
        caller = {}
        caller['name'] = target.get_name()
        caller['directory'] = target.get_prefix()
        caller['ifghost'] = '{0},{1}'.format(target.get_selfname(),
                                             target.get_keroname())
        self.plugin_controler.exec_plugin(plugin_dir, argv, caller)

    def select_nekodorif(self, nekodorif_dir, target):
        ninix.nekodorif.Nekoninni().load(nekodorif_dir,
                                         self.katochan, target)

    def select_kinoko(self, data, target):
        ninix.kinoko.Kinoko(self.kinoko).load(data, target)

    def open_console(self):
        self.console.open()

    def open_ghost_manager(self):
        self.__ngm.show_dialog()

    def show_usage(self):
        for sakura in self.get_working_ghost():
            sakura.save_history()
        history = {}
        for i in self.ghosts:
            if self.ghosts[i] is None:
                continue
            desc = self.ghosts[i][0]
            name = desc.get('name', i)
            ghost_time = 0
            prefix = self.ghosts[i][4]
            path = os.path.join(prefix, 'HISTORY')
            if os.path.exists(path):
                try:
                    with open(path, 'r') as f:
                        for line in f:
                            if ',' not in line:
                                continue
                            key, value = line.split(',', 1)
                            key = key.strip()
                            if key == 'time':
                                try:
                                    ghost_time = int(value.strip())
                                except:
                                    pass
                except IOError as e:
                    code, message = e.args
                    logging.error('cannot read {0}'.format(path))
            ai_list = []
            dirlist = os.listdir(os.path.join(prefix, 'shell'))
            for subdir in dirlist:
                path = os.path.join(prefix, 'shell', subdir, 'ai.png')
                if os.path.exists(path):
                    ai_list.append(path)
            history[name] = (ghost_time, ai_list)
        self.usage_dialog.open(history)

    def search_ghosts(self): ## FIXME
        balloons = self.get_balloon_list()
        ghosts = [1 for item in self.ghost_list if item is not None]
        if len(ghosts) > 0 and len(balloons) > 0:
            self.confirmed = True
        return len(ghosts), len(balloons)

    def do_idle_tasks(self):
        if not self.confirmed:
            self.console.open()
        else:
            if not self.loaded:
                self.load()
                # start SSTP server
                self.sstp_controler.start_servers()
                # start plugins
                self.plugin_controler.start_plugins(self.plugins)
                self.loaded = True
            else:
                self.sstp_controler.handle_sstp_queue()
                self.sstp_controler.receive_sstp_request()
                self.plugin_controler.check_queue()
        return True


class Console(object):

    # DnD data types
    dnd_targets = [
        ('text/plain', 0, 0),
        ]

    def __init__(self, app):
        self.app = app
        self.window = gtk.Dialog()
        self.window.connect('delete_event', self.close)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.darea.connect('drag_data_received', self.drag_data_received)
        self.darea.drag_dest_set(gtk.DEST_DEFAULT_ALL, self.dnd_targets,
                                 gtk.gdk.ACTION_COPY)
        self.size = (330, 110) ## FIXME
        self.darea.set_size_request(*self.size)
        self.darea.connect('configure_event', self.configure)
        self.darea.connect('expose_event', self.redraw)
        self.window.vbox.pack_start(self.darea)
        self.darea.show()
        box = gtk.HButtonBox()
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('Install')
        button.connect('clicked', self.file_chooser)
        box.add(button)
        button.show()
        button = gtk.Button('Close')
        button.connect('clicked', self.close)
        box.add(button)
        button.show()
        self.file_chooser = gtk.FileChooserDialog(
            "Install..",
            None,
            gtk.FILE_CHOOSER_ACTION_OPEN,
            (gtk.STOCK_OPEN, gtk.RESPONSE_OK,
             gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
        self.file_chooser.set_default_response(gtk.RESPONSE_CANCEL)
        filter = gtk.FileFilter()
        filter.set_name("All files")
        filter.add_pattern("*")
        self.file_chooser.add_filter(filter)
        filter = gtk.FileFilter()
        filter.set_name("nar/zip")
        filter.add_mime_type("application/zip")
        filter.add_pattern("*.nar")
        filter.add_pattern("*.zip")
        self.file_chooser.add_filter(filter)
        self.opened = 0

    def update(self): ## FIXME
        self.darea.queue_draw()

    def open(self):
        if self.opened:
            return
        self.window.show()
        self.opened = 1

    def close(self, widget=None, event=None):
        self.window.hide()
        self.opened = 0
        if not self.app.confirmed: ## FIXME
            self.app.quit()
        return True

    def file_chooser(self, widget=None, event=None):
        response = self.file_chooser.run()
        if response == gtk.RESPONSE_OK:
            filename = self.file_chooser.get_filename()
            self.app.do_install(filename)
            self.update()
        elif response == gtk.RESPONSE_CANCEL:
            pass
        self.file_chooser.hide()

    def configure(self, darea, event):
        x, y, w, h = darea.get_allocation()
        self.size = (w, h)

    def draw_message(self, text): ## FIXME
        pass

    def redraw(self, darea, event):
        ghosts, balloons = self.app.search_ghosts() # XXX
        if ghosts > 0 and balloons > 0:
            self.window.set_title(_('Console'))
        else:
            self.window.set_title(_('Nanntokashitekudasai.'))
        layout = pango.Layout(darea.get_pango_context())
        font_desc = pango.FontDescription()
        font_desc.set_size(9 * pango.SCALE)
        font_desc.set_family('Sans') # FIXME
        layout.set_font_description(font_desc)
        cr = darea.window.cairo_create()
        w, h = self.size
        cr.set_source_rgb(0.0, 0.0, 0.0) # black
        cr.paint()
        layout.set_text('Ghosts: {0:d}'.format(ghosts))
        if ghosts == 0:
            cr.set_source_rgb(1.0, 0.2, 0.2) # red
        else:
            cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(20, 15)
        cr.show_layout(layout)
        w, h = layout.get_pixel_size()
        layout.set_text('Balloons: {0:d}'.format(balloons))
        if balloons == 0:
            cr.set_source_rgb(1.0, 0.2, 0.2) # red
        else:
            cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(20, 15 + h)
        cr.show_layout(layout)
        del cr

    def drag_data_received(self, widget, context, x, y, data, info, time):
        logging.info('Content-type: {0}'.format(data.type))
        logging.info('Content-length: {0:d}'.format(data.get_length()))
        if str(data.type) == 'text/plain':
            filelist = []
            for line in data.data.split('\r\n'):
                scheme, host, path, params, query, fragment = \
                        urlparse.urlparse(line)
                pathname = urllib.url2pathname(path)
                if scheme == 'file' and os.path.exists(pathname):
                    filelist.append(pathname)
            for filename in filelist:
                self.app.do_install(filename)
            self.update()
        return True


class UsageDialog(object):

    def __init__(self):
        self.window = gtk.Dialog()
        self.window.set_title('Usage')
        self.window.connect('delete_event', self.close)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.size = (550, 330)
        self.darea.set_size_request(*self.size)
        self.darea.connect('configure_event', self.configure)
        self.darea.connect('expose_event', self.redraw)
        self.window.vbox.pack_start(self.darea)
        self.darea.show()
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('Close')
        button.connect('clicked', self.close)
        box.add(button)
        button.show()
        self.opened = 0

    def open(self, history):
        if self.opened:
            return
        self.history = history
        self.items = \
            [(name, clock, path) for name, (clock, path) in \
                 self.history.items()]
        self.items[:] = [(x[1], x) for x in self.items]
        self.items.sort()
        self.items[:] = [x for x_1, x in self.items]
        self.items.reverse()
        ai_list = self.items[0][2]
        if ai_list:
            path = random.choice(ai_list)
            assert os.path.exists(path)
            self.pixbuf = ninix.pix.create_pixbuf_from_file(
                path, is_pnr=False)
            self.pixbuf.saturate_and_pixelate(self.pixbuf, 1.0, True)
        else:
            self.pixbuf = None
        self.window.show()
        self.opened = 1

    def close(self, widget=None, event=None):
        self.window.hide()
        self.opened = 0
        return True

    def configure(self, darea, event):
        x, y, w, h = darea.get_allocation()
        self.size = (w, h)

    def redraw(self, darea, event):
        if not self.items:
            return # should not reach here
        total = float(0)
        for name, clock, path in self.items:
            total += clock
        layout = pango.Layout(darea.get_pango_context())
        font_desc = pango.FontDescription()
        font_desc.set_size(9 * pango.SCALE)
        font_desc.set_family('Sans') # FIXME
        layout.set_font_description(font_desc)
        cr = darea.window.cairo_create()
        # redraw graph
        w, h = self.size
        cr.set_source_rgb(1.0, 1.0, 1.0) # white
        cr.paint()
        # ai.png
        if self.pixbuf:
            cr.set_source_pixbuf(self.pixbuf, 16, 32) # XXX
            cr.paint()
        w3 = w4 = 0
        rows = []
        for name, clock, path in self.items[:14]:
            layout.set_text(name)
            name_w, name_h = layout.get_pixel_size()
            rate = '{0:.1f}%'.format(clock / total * 100)
            layout.set_text(rate)
            rate_w, rate_h = layout.get_pixel_size()
            w3 = max(rate_w, w3)
            time = '{0}:{1:02d}'.format(*divmod(clock / 60, 60))
            layout.set_text(time)
            time_w, time_h = layout.get_pixel_size()
            w4 = max(time_w, w4)
            rows.append((clock, name, name_w, name_h, rate, rate_w, rate_h,
                         time, time_w, time_h))
        w1 = 280
        w2 = w - w1 - w3 - w4 - 70
        x = 20
        y = 15
        x += w1 + 10
        label = 'name'
        layout.set_text(label)
        label_name_w, label_name_h = layout.get_pixel_size()
        cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(x, y)
        cr.show_layout(layout)
        x = x + w2 + 10
        label = 'rate'
        layout.set_text(label)
        label_rate_w, label_rate_h = layout.get_pixel_size()
        cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(x + w3 - label_rate_w, y)
        cr.show_layout(layout)
        x += w3 + 10
        label = 'time'
        layout.set_text(label)
        label_time_w, label_time_h = layout.get_pixel_size()
        cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(x + w4 - label_time_w, y)
        cr.show_layout(layout)
        y += max([label_name_h, label_rate_h, label_time_h]) + 4
        for clock, name, name_w, name_h, rate, rate_w, rate_h, time, time_w, \
                time_h  in rows:
            x = 20
            bw = int(clock / total * w1)
            bh = max([name_h, rate_h, time_h]) - 1
            cr.set_source_rgb(0.8, 0.8, 0.8) # gray
            cr.rectangle(x + 1, y + 1, bw, bh)
            cr.stroke()
            cr.set_source_rgb(1.0, 1.0, 1.0) # white
            cr.rectangle(x, y, bw, bh)
            cr.stroke()
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.rectangle(x, y, bw, bh)
            cr.stroke()
            x += w1 + 10
            layout.set_text(name)
            end = len(name)
            while end > 0:
                w, h = layout.get_pixel_size()
                if w > 168:
                    end -= 1
                    layout.set_text(''.join((name[:end], u'...')))
                else:
                    break
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.move_to(x, y)
            cr.show_layout(layout)
            x += w2 + 10
            layout.set_text(rate)
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.move_to(x + w3 - rate_w, y)
            cr.show_layout(layout)
            x += w3 + 10
            layout.set_text(time)
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.move_to(x + w4 - time_w, y)
            cr.show_layout(layout)
            y += max([name_h, rate_h, time_h]) + 4
        del cr


if __name__ == '__main__':
    main()
