/**************************************************************************/
/*                                N I C E                                 */
/*             A high-level object-oriented research language             */
/*                        (c) Daniel Bonniot 2003                         */
/*                                                                        */
/*  This program is free software; you can redistribute it and/or modify  */
/*  it under the terms of the GNU General Public License as published by  */
/*  the Free Software Foundation; either version 2 of the License, or     */
/*  (at your option) any later version.                                   */
/*                                                                        */
/**************************************************************************/

package bossa.util;

import java.util.*;

/**
   Represents a portion of the input file.
   Used to report errors to the user.
  
   @see Located

   @author Daniel Bonniot (bonniot@users.sourceforge.net)
 */
public abstract class Location implements Located
{
  public static Location make
    (java.io.File file, 
     int startLine, int startColumn,
     int endLine, int endColumn)
  {
    return new Source(file, startLine, startColumn, endLine, endColumn);
  }

  public static Location make
    (java.io.File file, 
     int startLine, int startColumn)
  {
    return make(file, startLine, startColumn, -1, -1);
  }

  public static Location make
    (int startLine, int startColumn,
     int endLine, int endColumn)
  {
    return make(currentFile, startLine, startColumn, endLine, endColumn);
  }

  public static Location make(int startLine, int startColumn)
  {
    return make(currentFile, startLine, startColumn, -1, -1);
  }

  /**
   * The file beeing parsed.
   * Enables to set the file name just once,
   * not at each Location construction.
   */
  public static void setCurrentFile(java.io.File file)
  {
    currentFile = file;
  }
  private static java.io.File currentFile = null;

  public static final Location option = new Option();

  public static Location nowhere() 
  { 
    return new Location.File(currentFile);
  }

  public Location location()
  {
    return this;
  }

  /**
     Return an identifier, which is guaranteed to be different for
     different locations.
  */
  public String uniqueIdentifier(String root)
  {
    StringBuffer uniq = new StringBuffer(super.toString());
    // Replace the '@' in bossa.util.Location@12345 with '_'
    uniq.setCharAt("bossa.util.Location".length(), '_');

    return root + uniq.toString();
  }

  public String toShortString()
  {
    return toString();
  }

  /****************************************************************
   * Setting source location in the generated bytecode.
   ****************************************************************/

  public void write(gnu.expr.Expression exp)
  {
  }

  public void writeEnd(gnu.expr.Expression exp)
  {
  }

  public void write(gnu.expr.Declaration decl)
  {
  }

  /****************************************************************
   * Different behaviour wether compiler is invoked by an editor.
   ****************************************************************/

  public static boolean editorMode = false;

  /****************************************************************
   * File location
   ****************************************************************/

  public static class File extends Location
  {
    private File(java.io.File file)
    {
      this.file = file;
    }

    private java.io.File file;

    public java.io.File getFile() { return file; }

    public String toString()
    {
      if (file == null)
        return "";
      else
        return nice.tools.util.System.prettyPrint(file);
    }

    public String toShortString()
    {
      return (file == null) ? "" : file.getName();
    }

    public void write(gnu.expr.Expression exp)
    {
      if (file == null)
        return;

      // In the bytecode, the fully qualified name is not wanted.
      exp.setFile(file.getName());
    }

    public void write(gnu.expr.Declaration decl)
    {
      if (file == null)
        return;

      // In the bytecode, the fully qualified name is not wanted.
      decl.setFile(file.getName());
    }
  }

  /****************************************************************
   * Source location
   ****************************************************************/

  public static class Source extends File
  {
    private Source(java.io.File file, 
                   int startLine, int startColumn,
                   int endLine, int endColumn)
    {
      super(file);
      this.startLine = startLine;
      this.startColumn = startColumn;
      this.endLine = endLine;
      this.endColumn = endColumn;
    }

    private int startLine,startColumn,endLine,endColumn;
    
    public int getLine()
    {
      return startLine;
    }

    public int getColumn()
    {
      return startColumn;
    }
    
    public String toString()
    {
      String res = super.toString();
      if (editorMode)
        return 
          (res.length()>0 ? (res + ":") : "") +
          startLine + ":" + startColumn;
      else
        return 
          (res.length()>0 ? (res + ": ") : "") +
          "line "+startLine+", column "+startColumn;
    }

    public String toShortString()
    {
      return super.toShortString() + ":" + startLine;
    }

    public String uniqueIdentifier(String root)
    {
      return root + startLine + "_" + startColumn;
    }

    public void write(gnu.expr.Expression exp)
    {
      super.write(exp);
      exp.setLine(getLine(), getColumn());
    }

    public void writeEnd(gnu.expr.Expression exp)
    {
      super.write(exp);
      exp.setLine(endLine, endColumn);
    }

    public void write(gnu.expr.Declaration decl)
    {
      super.write(decl);
      decl.setLine(getLine(), getColumn());
    }
  }
    
  public static Location make(gnu.expr.Expression expr)
  {
    String file = expr.getFile();
    return new Source(file != null ? new java.io.File(file) : null,
                      expr.getLine(), expr.getColumn(),
                      -1, -1);
  }

  /****************************************************************
   * A compilation option
   ****************************************************************/

  public static class Option extends Location
  {
    public String toString()
    {
      return "Command line";
    }
  }
}
